/*
 * gnuclient.cpp -- pbreton Thu Jan 25 1996
 *
 * Client code to allow local and remote editing of files by GNU Emacs.
 *
 * This file is not part of GNU Emacs. 
 * 
 * Copying is permitted under those conditions described by the GNU
 * General Public License.
 *
 * Copyright (C) 1996 Peter Breton
 *
 * Author: Peter Breton (pbreton@cs.umb.edu)
 * Based heavily on Andy Norman's gnuserv (ange@hplb.hpl.hp.com), 
 * which in turn was based on 'etc/emacsclient.c' from the GNU 
 * Emacs 18.52 distribution.
 *
 * Modifications: Nico Francois (nico.francois@scala.nl) 
 *                Mon Jun 3 1996
 * See comment in gnuserv.c.
 *
 * Modifications: Nico Francois (nico.francois@scala.nl) 
 *                Sun Sep 15 1996
 * If WIN_VERSION is defined the -q (quick) flag is always turned on.
 * Will be defined when the client program is compiled and then linked
 * as a windows program.
 *
 * Modifications: Nico Francois (nico.francois@scala.nl) 
 *                Mon Oct 7 1996
 * filename_expand() has been modified to always make sure it has the
 * long filename of a file.
 *
 * Modifications: Nico Francois (nico.francois@scala.nl) 
 *                Fri Oct 11 1996
 * Fixed small bug in new long filename code.  gnuclient was no longer
 * accepting filenames of files that don't exist.
 *
 * Modifications: Nico Francois (nico.francois@scala.nl) 
 *                Mon Jan 13 1997
 * Incorporated small fix by William Sobel.  gnuclient(w) will now make
 * sure the path uses a lowercase drive letter before sending it to Emacs.
 * Emacs will then no longer consider files opened manually to be
 * different than when they are opened with gnuclient(w).
 * 
 * Modifications: Nico Francois (nico.francois@scala.nl) 
 *                Tue Feb 18 1997
 * Two new options to work around an Emacs 19.34 bug: -f and -F.  The -f
 * option will pop the Emacs window to the front after loading the file,
 * if Emacs is not iconified.
 * The -F option will uniconify Emacs if needed so its window will always
 * pop to the front.
 * 
 * Modifications: Nico Francois (nico.francois@scala.nl) 
 *                Thu Feb 27 1997
 * Major code cleanup (slightly C++-ified as well).
 * Added a version number that is printed when usage us requested (-? option).
 * Starting gnuclient without arguments now fails, and usage is printed.
 */

static char version_str[] = "gnuclient 1.7 (27 Feb 1997)";

#include "gnuserv.h"
#define __STDC__			1
#define __GNU_LIBRARY__		1
#include "getopt.h"
#include "ctype.h"

#define FORWARD_SLASH		'/'
#define BACKWARD_SLASH		'\\'

// filename_expand -- try to convert the given filename into a 
// fully-qualified pathname.
//
// 'fullpath' - points to buffer that will be filled with full path
// 'filename' - file to find path of
BOOL filename_expand (char *fullpath, char *filename)
{
	char *fname;

	if (!GetFullPathName (filename, MAXPATHLEN + 2, fullpath, &fname)) {
		fprintf (stderr, "Unable to get full pathname for %s\n", filename);
		return (FALSE);
		}

	// Make sure that the drive letter is lower case.
	if (fullpath[1] == ':' && isupper (fullpath[0])) {
		fullpath[0] = (char)tolower (fullpath[0]);
		}

    // Convert all backslashes to forward-slashes
    for (char *p = fullpath; *p; p++) {
		if (BACKWARD_SLASH == *p)
			*p = FORWARD_SLASH;
		}

	// Since GetFullPathName doesn't always seem to return the long
	// filename (as documented) we get it ourselves.
	WIN32_FIND_DATA finddata;
	HANDLE handle = FindFirstFile (filename, &finddata);
	if (handle != INVALID_HANDLE_VALUE) {
		strcpy (fname, finddata.cFileName);
		FindClose (handle);
		}

	return (TRUE);
}

void main (int argc, char *argv[])
{
	char *hostarg = GNUSERV_DEFAULT_HOST;	// hostname
	BOOL qflg = FALSE;						// quick edit
	BOOL dbgflg = FALSE;					// debug flag
	BOOL errflg = FALSE;					// option error
	BOOL wintofront = FALSE;				// wintofront flag
	BOOL uniconify = FALSE;					// uniconify flag
	BOOL noresult = FALSE;

#ifdef WIN_VERSION
	qflg = TRUE;
#endif

	// Parse arguments
	int c;
	while ((c = getopt (argc, argv, "h:qdfF?" )) != EOF) {
		switch (c) {
			case 'q':
				qflg = TRUE;
				break;

			case 'd':
				dbgflg = TRUE;
				break;

			case 'h':
				hostarg = optarg;
				// FIX! We don't support results from a server on another
				// FIX! host just yet.
				noresult = TRUE;
				break;

			case 'f':
				wintofront = TRUE;
				break;

			case 'F':
				wintofront = TRUE;
				uniconify = TRUE;
				break;

			case '?':
				errflg = TRUE;
			}
		}
  
	// Print a usage message and quit
	if (argc <= 1 || errflg) {
		fprintf (stderr,
				"%s\n"
				"Usage: %s [-q] [-d] [-f] [-F] [-h hostname] [[+line] path] ...\n",
				version_str,
				argv[0]);
		exit (1);
		}

	// First create mailslot to receive result
	DWORD clientid = GetCurrentProcessId();
	HANDLE resultslot = CreateGnuServSlot (TRUE, clientid);
	if (INVALID_HANDLE_VALUE == resultslot) {
		exit (1);
		}

	// Try to connect to the mailslot. Dies on failure.
	HANDLE h = ConnectToMailslot (hostarg, FALSE, 0);
	if (INVALID_HANDLE_VALUE == h) {
		exit (1);
		}

	// Send client id...
	char clientidstr[20];
	sprintf (clientidstr, "C:%d ", clientid);
	if (!SendString (h, clientidstr)) {
		exit (1);
		}

	// Then send a lisp command
	char *basecommand = qflg ?
		"(server-edit-files-quickly '(" : "(server-edit-files '("; 
	SendString (h, basecommand);
	if (dbgflg) {
		printf (basecommand);
		}

	// ...followed by file names to the server process
	int starting_line = 1;
	for (; optind < argc; optind++) {
		if (*argv[optind] == '+') {
			starting_line = atoi (argv[optind]);
			}
		else {
			char fullpath[MAXPATHLEN+2];
			if (filename_expand (fullpath, argv[optind])) {
				char command[MAXPATHLEN+50];
				sprintf (command, "(%d . \"%s\")", starting_line, fullpath);
				SendString (h, command);
				if (dbgflg) {
					printf (command);
					}
				starting_line = 1;
				}
			}
		}

	// Finish off the sexp and send an EOT
	SendString (h, "))");
	SendString (h, EOT_STR);

	if (dbgflg) {
		printf("))");
		}

	// Disconnect
	DisconnectFromMailslot (h);

	// Now wait for result, then exit.
	if (!noresult) WaitForServerResult (resultslot, FALSE);
	DisconnectFromMailslot (resultslot);

	// Handle -f and -F options (to pop the Emacs window to the front).
	if (wintofront) {
		HWND hWnd = FindWindow ("Emacs", NULL);
		if (hWnd) {
			// Is the Emacs window iconified ?
			if (IsIconic (hWnd)) {
				if (uniconify) {
					ShowWindow (hWnd, SW_SHOWNORMAL);
					// Need this since Emacs thinks it is still iconified otherwise!
					SendMessage (hWnd, WM_SYSCOMMAND, SC_RESTORE, 0);
					}
				}
			else {
				SetForegroundWindow (hWnd);
				}
			}
		}

	exit (0);
}
