package html;

import java.net.*;
import java.util.*;
import java.awt.*;
import java.awt.image.*;

class HtmlPager
{
  private static final int BUTTON = -1;
  private static final int MARGIN = 20;
  private static final int sizes[] = { 32, 28, 24, 20, 17, 15, 14 };
  private int offset = 0;
  private Stack fonts = new Stack();
  private Font font = null;
  private int spaceWidth = 0;
  private FontMetrics metrics = null;
  private Stack anchors = new Stack();
  private boolean anchor = false;
  private int leftMargin = MARGIN;
  private int rightMargin = -MARGIN;
  private int center = 0;
  private int preformatted = 0;
  private Stack lists = new Stack();
  private int list = BUTTON;
  private Vector hrefs = new Vector();
  private Hashtable names = new Hashtable();
  private int heights[] = null;
  private Vector lines = new Vector();
  private HtmlPagerLine line = null;
  private Href href = null;
  private boolean lineEmpty = true;
  private boolean prevLineEmpty = false;
  private URL url = null;
  private HtmlCanvas parent = null;
  private int width = -1;
  private Color bgColor;
  private Color textColor;
  private Color linkColor;
  private Stack fontColors = new Stack();
  private Color fontColor;

  protected HtmlPager(HtmlCanvas parent, int width)
  {
    this.width = width;
    this.parent = parent;
    rightMargin += width;
    pushFont("TimesRoman", Font.PLAIN, sizes[6]);
    drawNewLine(true);
    bgColor = parent.getBackground();
    textColor = parent.getForeground();
    linkColor = Color.blue;
    fontColor = textColor;
  }

  protected void finish()
  {
    int lineCount = lines.size();

    // create heights
    int heights[] = new int[lineCount + 2];
    heights[0] = MARGIN;
    for (int i = 0; i < lineCount; i++)
      heights[i + 1] = heights[i] + ((HtmlPagerLine)lines.elementAt(i)).getHeight();
    heights[lineCount + 1] = heights[lineCount] + MARGIN;

    // create hrefs
    Href hrefs[] = new Href[this.hrefs.size()];
    this.hrefs.copyInto(hrefs);

    // create temporary image storage
    Vector tmpImgs = new Vector();

    // create image
    Image image = null;
    if (width > 0)
    {
      image = parent.createImage(width, heights[lineCount + 1]);
      if (image != null)
      {
	parent.setBackground(bgColor);
	Graphics g = image.getGraphics();
	g.setColor(bgColor);
	g.fillRect(0, 0, width, heights[lineCount + 1]);
	for (int i = 0; i < lineCount; i++)
	  ((HtmlPagerLine)lines.elementAt(i)).draw(g, heights[i], tmpImgs);
      }
    }

    // create image array
    HtmlImage imgs[] = new HtmlImage[tmpImgs.size()];
    tmpImgs.copyInto(imgs);

    parent.setData(heights, hrefs, names, imgs, image);
  }

  protected void setBase(URL base)
  {
    url = base;
  }

  protected void setColors(Color bg, Color text, Color link)
  {
    if (bg != null)
      bgColor = bg;
    if (text != null)
      textColor = text;
    if (link != null)
      linkColor = link;
    pushFontColor(anchor ? linkColor : textColor);
  }

  protected void pushStandardFont()
  {
    pushFont("TimesRoman", font.getStyle(), font.getSize());
  }
  protected void pushFixedFont()
  {
     pushFont("Courier", font.getStyle(), font.getSize());
  }
  protected void pushFontSize(int size)
  {
     if (size < 0)
       size = 0;
     if (size >= sizes.length)
       size = sizes.length - 1;
     pushFont(font.getName(), font.getStyle(), sizes[size]);
  }
  protected int getFontSize()
  {
    int size = font.getSize();
    for (int i = 0; i < sizes.length; i++)
      if (size == sizes[i])
	return i;
    return -1;
  }
  protected void pushBold()
  {
     pushFont(font.getName(), font.getStyle() | Font.BOLD, font.getSize());
  }
  protected void pushItalic()
  {
     pushFont(font.getName(), font.getStyle() | Font.ITALIC, font.getSize());
  }
  private void pushFont(String name, int style, int size)
  {
    fonts.push(font);
    font = new Font(name, style, size);
    metrics = parent.getFontMetrics(font);
    spaceWidth = metrics.stringWidth(" ");
  }
  protected void popFont()
  {
    if (fonts.size() > 1)
    {
      font = (Font)fonts.pop();
      metrics = parent.getFontMetrics(font);
      spaceWidth = metrics.stringWidth(" ");
    }
  }

  protected void pushFontColor(Color color)
  {
    fontColors.push(fontColor);
    fontColor = color;
  }
  protected void popFontColor()
  {
    if (fontColors.size() > 0)
      fontColor = (Color)fontColors.pop();
  }
  protected Color getFontColor()
  {
    return fontColor;
  }

  protected void pushAnchor(String href, String name)
  {
    anchors.push(new Boolean(anchor));
    if (href != null)
    {
      anchor = true;
      pushFontColor(linkColor);
      this.href = new Href();
      this.href.startLine = lines.size() - 1;
      this.href.startOffset = offset;
      try
      {
        this.href.url = new URL(url, href);
      }
      catch (Exception e)
      {
        this.href.url = null;
      }
    }
    if (name != null)
    {
      names.put(name, new Integer(lines.size()));
    }
  }
  protected void popAnchor()
  {
    if (anchors.size() > 0)
    {
      if (anchor)
      {
	href.endLine = lines.size() - 1;
	href.endOffset = offset;
	hrefs.addElement(href);
	href = null;
      }
      anchor = ((Boolean)anchors.pop()).booleanValue();
      popFontColor();
    }
  }

  protected void pushLeftMargin(boolean newline)
  {
    leftMargin += MARGIN;
    if (newline)
      drawNewLine(true);
  }
  protected void popLeftMargin(boolean newline)
  {
    leftMargin -= MARGIN;
    if (newline)
      drawNewLine(true);
  }

  protected void pushRightMargin()
  {
    rightMargin -= MARGIN;
  }
  protected void popRightMargin()
  {
    rightMargin += MARGIN;
  }

  protected void pushCenter()
  {
    drawNewLine(false);
    center++;
  }
  protected void popCenter()
  {
    drawNewLine(false);
    center--;
  }

  protected void pushPreformatted()
  {
    pushFixedFont();
    drawNewLine(true);
    preformatted++;
    prevLineEmpty = true;
  }
  protected void popPreformatted()
  {
    preformatted--;
    drawNewLine(true);
    popFont();
  }

  protected void pushListButton()
  {
    lists.push(new Integer(list));
    list = BUTTON;
  }
  protected void pushListNumber()
  {
    lists.push(new Integer(list));
    list += 0;
  }
  protected void popList()
  {
    if (lists.size() > 0)
      list = ((Integer)lists.pop()).intValue();
  }

  private void finishLine()
  {
    prevLineEmpty = lineEmpty;
    if (center > 0)
      line.translate((rightMargin - offset) / 2);
    line = new HtmlPagerLine();
    lines.addElement(line);
    offset = 0;
    addItem(new HtmlPagerItem(fontColor, font, ""));
    offset = leftMargin;
    lineEmpty = true;
  }
  protected void drawNewLine(boolean allways)
  {
    if (lineEmpty)
      offset = leftMargin;
    if (lineEmpty && lines.size() == 1)
      return;
    if (lineEmpty)
    {
      if ((preformatted > 0 || allways) && !prevLineEmpty)
	finishLine();
    }
    else
    {
      finishLine();
      if (allways)
	finishLine();
    }
  }
  protected void drawRule()
  {
    drawNewLine(false);
    addItem(new HtmlPagerItem(fontColor, rightMargin - leftMargin));
    drawNewLine(false);
  }
  protected void drawListItem()
  {
    int oldOffset = offset;
    offset = leftMargin - MARGIN;
    if (list == BUTTON)
      addItem(new HtmlPagerItem(fontColor, font, "-"));
    else
      addItem(new HtmlPagerItem(fontColor, font, String.valueOf(++list)));
    offset = oldOffset;
  }
  synchronized protected void drawImage(HtmlImage img, String align)
  {
    Color color = anchor ? linkColor : null;
    drawNewLine(img.w);
    if ("TOP".equalsIgnoreCase(align))
      addItem(new HtmlPagerItem(color, img, HtmlPagerItem.TOP));
    else if ("MIDDLE".equalsIgnoreCase(align))
      addItem(new HtmlPagerItem(color, img, HtmlPagerItem.MIDDLE));
    else
      addItem(new HtmlPagerItem(color, img, HtmlPagerItem.BOTTOM));
    offset += img.w;
  }
  protected void drawText(String text)
  {
    if (preformatted > 0)
    {
      StringTokenizer lines = new StringTokenizer(text, "\n", true);
      while (lines.hasMoreTokens())
      {
	String line = lines.nextToken();
	if (line.equals("\n"))
	{
	  drawNewLine(false);
	  prevLineEmpty = false;
	}
	else
	{
	  addItem(new HtmlPagerItem(fontColor, font, line));
	  offset += metrics.stringWidth(line);
	}
      }
    }
    else
    {
      if (offset > leftMargin)
	offset += spaceWidth;
      StringTokenizer words = new StringTokenizer(text);
      while (words.hasMoreTokens())
      {
	String word = words.nextToken();
	int w = metrics.stringWidth(word);
	drawNewLine(w);
	addItem(new HtmlPagerItem(fontColor, font, word));
	offset += w + spaceWidth;
      }
      if (words.countTokens() > 0)
	offset -= spaceWidth;
    }
  }
  private void drawNewLine(int w)
  {
    if (offset + w >= rightMargin)
      drawNewLine(false);
  }
  private void addItem(HtmlPagerItem pi)
  {
    line.addItem(metrics, offset, pi);
    lineEmpty = false;
  }
}
