package html;

import java.awt.*;
import java.net.URL;
import java.util.Vector;

public class HtmlViewer extends Frame implements Runnable
{
  private static final int GAP = 5;

  private HtmlCanvas canvas;
  private Scrollbar scrollbar;
  private Button forward;
  private Button backward;
  private Button reload;
  private Label label;
  private TextField location;
  private Vector history;
  private int historyIndex;
  private Event evt = null;
  private String start = null;

  public HtmlViewer()
  {
    history = new Vector();
    historyIndex = -1;

    setTitle("(No document)");

    setLayout(null);

    add(backward = new Button("Backward"));
    add(forward = new Button("Forward"));
    add(reload = new Button("Reload"));
    add(label = new Label("Location:"));
    add(location = new TextField());
    add(canvas = new HtmlCanvas());
    add(scrollbar = new Scrollbar(Scrollbar.VERTICAL));

    backward.disable();
    forward.disable();
    canvas.setBackground(Color.white);
    canvas.resize(600, 500);

    setBackground(Color.lightGray);
    pack();
    show();
    // ccl 1998-12-06
    setLocation( 0, 0 );
  }

  public HtmlViewer(URL url)
  {
    this();

    setCursor(WAIT_CURSOR);
    try
    {
      changeNewDocument(new HtmlDocument(url));
    }
    catch (Exception e)
    {
       e.printStackTrace();
    }
    setCursor(DEFAULT_CURSOR);
  }

  public boolean handleEvent(Event evt)
  {
    if (evt.id == Event.WINDOW_DESTROY)
    {
      dispose();
      return true;
    }
    if (evt.target == scrollbar)
    {
      switch (evt.id)
      {
      case Event.SCROLL_ABSOLUTE:
      case Event.SCROLL_LINE_DOWN:
      case Event.SCROLL_LINE_UP:
      case Event.SCROLL_PAGE_DOWN:
      case Event.SCROLL_PAGE_UP:
	canvas.setStart(scrollbar.getValue());
	start = null;
	return true;
      }
    }
    if (evt.target == canvas && evt.id == Event.MOUSE_DOWN || evt.id == Event.ACTION_EVENT)
    {
      if (this.evt == null)
      {
	setCursor(WAIT_CURSOR);
	this.evt = evt;
	new Thread(this).start();
      }
      return true;
    }
    return super.handleEvent(evt);
  }

  public void run()
  {
    if (evt.target == canvas && evt.id == Event.MOUSE_DOWN)
    {
      int x = evt.x - canvas.location().x;
      int y = evt.y - canvas.location().y;
      URL href = canvas.getHref(x, y);
      if (href != null)
      {
	try
	{
	  changeNewDocument(new HtmlDocument(href));
	}
	catch (Exception e)
	{
	  e.printStackTrace();
	}
      }
    }
    if (evt.id == Event.ACTION_EVENT)
    {
      if (evt.target == forward)
      {
	changeDocument((HtmlDocument)history.elementAt(++historyIndex));
      }
      if (evt.target == backward)
      {
	changeDocument((HtmlDocument)history.elementAt(--historyIndex));
      }
      if (evt.target == reload && historyIndex >= 0)
      {
	try
	{
	  HtmlDocument document = (HtmlDocument)history.elementAt(historyIndex);
	  document = new HtmlDocument(new URL(document.getURLString()));
	  history.setElementAt(document, historyIndex);
	  changeDocument(document);
	}
	catch (Exception e)
	{
	  e.printStackTrace();
	}
      }
      if (evt.target == location)
      {
	try
	{
	  changeNewDocument(new HtmlDocument(new URL(location.getText())));
	}
	catch (Exception e)
	{
	  e.printStackTrace();
	}
      }
    }
    setCursor(DEFAULT_CURSOR);
    evt = null;
  }

  private void changeNewDocument(HtmlDocument document)
  {
    if (historyIndex < 0 ||
	!((HtmlDocument)history.elementAt(historyIndex)).getURLString().
    						equals(document.getURLString()))
    {
      history.setSize(++historyIndex + 1);
      history.setElementAt(document, historyIndex);
      changeDocument(document);
    }
  }

  private void changeDocument(HtmlDocument document)
  {
    canvas.changeDocument(document);
    if (historyIndex == history.size() - 1)
      forward.disable();
    else
      forward.enable();
    if (historyIndex <= 0)
      backward.disable();
    else
      backward.enable();
    location.setText(document.getLocation());
    setTitle(document.getTitle());
    scrollbar.setValue(canvas.setStart(start = document.getStart()));
    layout();
  }

  public Dimension minimumSize()
  {
    return layoutSize(true);
  }

  public Dimension preferredSize()
  {
    return layoutSize(false);
  }

  public void layout()
  {
    Insets i = insets();

    // first check whether minimum of preferred sizes must be used
    boolean minimum = false;
    Dimension available = size();
    available.width -= i.left + i.right;
    available.height -= i.top + i.bottom;
    Dimension d = layoutSize(false);
    if (d.width > available.width || d.height > available.height)
      minimum = true;

    int w = GAP;
    int h = controlHeight(minimum);
    d = componentSize(forward, minimum);
    forward.reshape(w + i.left, GAP + i.top, d.width, h);
    w += d.width + GAP;
    d = componentSize(backward, minimum);
    backward.reshape(w + i.left, GAP + i.top, d.width, h);
    w += d.width + GAP;
    d = componentSize(reload, minimum);
    reload.reshape(w + i.left, GAP + i.top, d.width, h);
    w += d.width + GAP;
    d = componentSize(label, minimum);
    label.reshape(w + i.left, GAP + i.top, d.width, h);
    w += d.width + GAP;
    d = componentSize(location, minimum);
    location.reshape(w + i.left, GAP + i.top, available.width - w - GAP, h);
    h += 2 * GAP;
    available.height -= h;

    if (canvas.setWidth(available.width) > available.height &&
	available.height > 0)
    {
      d = componentSize(scrollbar, minimum);
      available.width -= d.width;
      int needed = canvas.setWidth(available.width);
      /*scrollbar.setValues(scrollbar.getValue(), available.height,
					0, needed - available.height - 1);*/
      scrollbar.setValues(scrollbar.getValue(), available.height,
					0, needed - 1);
      scrollbar.setPageIncrement(available.height);
      scrollbar.setLineIncrement(3);
      scrollbar.reshape(available.width + i.left, h + i.top,
					d.width, available.height);
      scrollbar.show();
    }
    else
    {
      scrollbar.setValue(0);
      scrollbar.hide();
    }
    canvas.repaint(10);
    canvas.reshape(i.left, h + i.top, available.width, available.height);
    if (start == null)
      canvas.setStart(scrollbar.getValue());
    else
      scrollbar.setValue(canvas.setStart(start));
  }

  private int controlHeight(boolean minimum)
  {
    Dimension d;

    int h = 0;
    d = componentSize(forward, minimum);
    h = d.height;
    d = componentSize(backward, minimum);
    h = Math.max(h, d.height);
    d = componentSize(reload, minimum);
    h = Math.max(h, d.height);
    d = componentSize(label, minimum);
    h = Math.max(h, d.height);
    d = componentSize(location, minimum);
    h = Math.max(h, d.height);

    return h;
  }

  private Dimension layoutSize(boolean minimum)
  {
    Dimension d;

    int w = GAP;
    int h = 0;
    d = componentSize(forward, minimum);
    w += d.width + GAP;
    h = d.height;
    d = componentSize(backward, minimum);
    w += d.width + GAP;
    h = Math.max(h, d.height);
    d = componentSize(reload, minimum);
    w += d.width + GAP;
    h = Math.max(h, d.height);
    d = componentSize(label, minimum);
    w += d.width + GAP;
    h = Math.max(h, d.height);
    d = componentSize(location, minimum);
    w += d.width + GAP;
    h = Math.max(h, d.height);
    h += 2 * GAP;
    d = componentSize(canvas, minimum);
    w = Math.max(w, d.width);
    h += d.height;

    Insets i = insets();
    return new Dimension(i.left + i.right + w, i.top + i.bottom + h);
  }

  private Dimension componentSize(Component c, boolean minimum)
  {
    return minimum ? c.minimumSize() : c.preferredSize();
  }

}
