## ----setup, include=FALSE-----------------------------------------------------
knitr::opts_chunk$set(echo = TRUE, dev = "png")

## ----install, eval= FALSE-----------------------------------------------------
#  if (!requireNamespace("BiocManager", quietly = TRUE)) {
#      install.packages("BiocManager")
#  }
#  BiocManager::install("celda")

## ----load, eval=TRUE, message=FALSE-------------------------------------------
library(celda)

## ----load_10X, eval=TRUE, message=FALSE---------------------------------------
# Install TENxPBMCData if is it not already
if (!requireNamespace("TENxPBMCData", quietly = TRUE)) {
  if (!requireNamespace("BiocManager", quietly = TRUE)) {
    install.packages("BiocManager")
  }
  BiocManager::install("TENxPBMCData")
}

# Load PBMC data
library(TENxPBMCData)
pbmc4k <- TENxPBMCData("pbmc4k")
colnames(pbmc4k) <- paste(pbmc4k$Sample, pbmc4k$Barcode, sep = "_")
rownames(pbmc4k) <- rowData(pbmc4k)$Symbol_TENx

## ----decontX_background, eval=FALSE, message=FALSE----------------------------
#  pbmc4k <- decontX(x = pbmc4k, background = raw)

## ----decontX, eval=TRUE, message=FALSE----------------------------------------
pbmc4k <- decontX(x = pbmc4k)

## ----UMAP_Clusters------------------------------------------------------------
umap <- reducedDim(pbmc4k, "decontX_UMAP")
plotDimReduceCluster(x = pbmc4k$decontX_clusters,
    dim1 = umap[, 1], dim2 = umap[, 2])

## -----------------------------------------------------------------------------
plotDecontXContamination(pbmc4k)

## ---- message=FALSE-----------------------------------------------------------
library(scater)
pbmc4k <- logNormCounts(pbmc4k)
plotDimReduceFeature(as.matrix(logcounts(pbmc4k)),
    dim1 = umap[, 1],
    dim2 = umap[, 2],
    features = c("CD3D", "CD3E", "GNLY",
        "LYZ", "S100A8", "S100A9",
        "CD79A", "CD79B", "MS4A1"),
    exactMatch = TRUE)

## ----barplotCounts------------------------------------------------------------
markers <- list(Tcell_Markers = c("CD3E", "CD3D"),
    Bcell_Markers = c("CD79A", "CD79B", "MS4A1"),
    Monocyte_Markers = c("S100A8", "S100A9", "LYZ"),
    NKcell_Markers = "GNLY")
cellTypeMappings <- list(Tcells = 2, Bcells = 5, Monocytes = 1, NKcells = 6)
plotDecontXMarkerPercentage(pbmc4k,
    markers = markers,
    groupClusters = cellTypeMappings,
    assayName = "counts")

## ----barplotDecontCounts------------------------------------------------------
plotDecontXMarkerPercentage(pbmc4k,
    markers = markers,
    groupClusters = cellTypeMappings,
    assayName = "decontXcounts")

## ----barplotBoth--------------------------------------------------------------
plotDecontXMarkerPercentage(pbmc4k,
    markers = markers,
    groupClusters = cellTypeMappings,
    assayName = c("counts", "decontXcounts"))

## ----plotDecontXMarkerExpression----------------------------------------------
plotDecontXMarkerExpression(pbmc4k,
    markers = markers[["Monocyte_Markers"]],
    groupClusters = cellTypeMappings,
    ncol = 3)

## ----plot_norm_counts, eval = FALSE-------------------------------------------
#  pbmc4k <- scater::logNormCounts(pbmc4k,
#      exprs_values = "decontXcounts",
#      name = "dlogcounts")
#  
#  plotDecontXMarkerExpression(pbmc4k,
#      markers = markers[["Monocyte_Markers"]],
#      groupClusters = cellTypeMappings,
#      ncol = 3,
#      assayName = c("logcounts", "dlogcounts"))

## ----findDelta----------------------------------------------------------------
metadata(pbmc4k)$decontX$estimates$all_cells$delta

## ----newDecontX, eval=TRUE, message=FALSE-------------------------------------
pbmc4k.delta <- decontX(pbmc4k, delta = c(9, 20), estimateDelta = FALSE)

plot(pbmc4k$decontX_contamination, pbmc4k.delta$decontX_contamination,
     xlab = "DecontX estimated priors",
     ylab = "Setting priors to estimate higher contamination")
abline(0, 1, col = "red", lwd = 2)

## -----------------------------------------------------------------------------
sessionInfo()

