## ----style, echo = FALSE, results = 'asis'--------------------------------------------------------
BiocStyle::markdown()
options(width = 100, max.print = 1000)
knitr::opts_chunk$set(
    eval = as.logical(Sys.getenv("KNITR_EVAL", "TRUE")),
    cache = as.logical(Sys.getenv("KNITR_CACHE", "TRUE")), 
    tidy.opts = list(width.cutoff = 100), tidy = FALSE)

## ----setting, eval=TRUE, echo=FALSE---------------------------------------------------------------
if (file.exists("spr_project")) unlink("spr_project", recursive = TRUE)
is_win <- Sys.info()[['sysname']] != "Windows"

## ----load_library, eval=TRUE, include=FALSE-------------------------------------------------------
suppressPackageStartupMessages({
    library(systemPipeR)
})

## ----utilities, eval=TRUE, echo=FALSE, out.width="100%", fig.align = "center", fig.cap= "Relevant features in `systemPipeR`. Workflow design concepts are illustrated under (A). Examples of `systemPipeR's` visualization functionalities are given under (B)."----
knitr::include_graphics(system.file("extdata/images", "utilities.png", package = "systemPipeR"))

## ----sysargslistImage, eval=TRUE, echo=FALSE, out.width="100%", fig.align = "center", fig.cap= "Workflow steps with input/output file operations are controlled by `SYSargs2` objects. Each `SYSargs2`instance is constructed from one targets and two param files. The only input provided by the user is the initial targets file. Subsequent targets instances are created automatically, from the previous output files. Any number of predefined or custom workflow steps are supported. One or many `SYSargs2` objects are organized in an `SYSargsList` container."----

knitr::include_graphics(system.file("extdata/images", "SPR_WF.png", package = "systemPipeR"))

## ----genNew_wf, eval=TRUE-------------------------------------------------------------------------
systemPipeRdata::genWorkenvir(workflow = "new", mydirname = "spr_project")

## ---- eval=FALSE, warning=FALSE-------------------------------------------------------------------
#  setwd("spr_project")

## ----setting_dir, include=FALSE, warning=FALSE----------------------------------------------------
knitr::opts_knit$set(root.dir = 'spr_project')

## ----SPRproject_ex, eval=TRUE---------------------------------------------------------------------
sal <- SPRproject() 

## ----importwf_example, eval=TRUE------------------------------------------------------------------
sal <- importWF(sal, 
                    file_path = system.file("extdata", "spr_simple_wf.Rmd", package = "systemPipeR"),
                    verbose = FALSE)

## ----run_echo, eval=is_win------------------------------------------------------------------------
#  sal <- runWF(sal)

## ----plot_echo, eval=TRUE-------------------------------------------------------------------------
plotWF(sal, width = "80%", rstudio = TRUE)

## ----status_echo, eval=TRUE-----------------------------------------------------------------------
statusWF(sal)

## ----logs_echo, eval=FALSE------------------------------------------------------------------------
#  sal <- renderLogs(sal)

## ----install, eval=FALSE--------------------------------------------------------------------------
#  if (!requireNamespace("BiocManager", quietly=TRUE)) install.packages("BiocManager")
#  BiocManager::install("systemPipeR")
#  BiocManager::install("systemPipeRdata")

## ----documentation, eval=FALSE--------------------------------------------------------------------
#  library("systemPipeR") # Loads the package
#  library(help="systemPipeR") # Lists package info
#  vignette("systemPipeR") # Opens vignette

## ----dir, eval=TRUE, echo=FALSE, out.width="100%", fig.align = "center", fig.cap= "*systemPipeR's* preconfigured directory structure."----
knitr::include_graphics(system.file("extdata/images", "spr_project.png", package = "systemPipeR"))  

## ----targetsSE, eval=TRUE-------------------------------------------------------------------------
targetspath <- system.file("extdata", "targets.txt", package = "systemPipeR") 
showDF(read.delim(targetspath, comment.char = "#"))

## ----targetsPE, eval=TRUE-------------------------------------------------------------------------
targetspath <- system.file("extdata", "targetsPE.txt", package = "systemPipeR")
showDF(read.delim(targetspath, comment.char = "#"))

## ----targets_echo, eval=TRUE----------------------------------------------------------------------
targetspath <- system.file("extdata/cwl/example/targets_example.txt", package = "systemPipeR")
showDF(read.delim(targetspath, comment.char = "#"))

## ----comment_lines, echo=TRUE---------------------------------------------------------------------
targetspath <- system.file("extdata", "targetsPE.txt", package = "systemPipeR")
readLines(targetspath)[1:4]

## ----targetscomp, eval=TRUE-----------------------------------------------------------------------
readComp(file = targetspath, format = "vector", delim = "-")

## ----targetsFig, eval=TRUE, echo=FALSE, out.width="100%", fig.align = "center", fig.cap= "_`systemPipeR`_ automatically creates the downstream `targets` files based on the previous steps outfiles. A) Usually, users provide the initial `targets` files, and this step will generate some outfiles, as demonstrated on B. Then, those files are used to build the new `targets` files as inputs in the next step. _`systemPipeR`_ (C) manages this connectivity among the steps automatically for the users."----
knitr::include_graphics(system.file("extdata/images", "targets_con.png", package = "systemPipeR"))  

## ----cleaning, eval=TRUE, include=FALSE-----------------------------------------------------------
if (file.exists(".SPRproject")) unlink(".SPRproject", recursive = TRUE)
## NOTE: Removing previous project create in the quick starts section

## ----SPRproject, eval=TRUE------------------------------------------------------------------------
sal <- SPRproject(projPath = getwd()) 

## ----SPRproject_logs, eval=FALSE------------------------------------------------------------------
#  sal <- SPRproject(logs.dir= ".SPRproject", sys.file=".SPRproject/SYSargsList.yml")

## ----SPRproject_dir, eval=FALSE-------------------------------------------------------------------
#  sal <- SPRproject(data = "data", param = "param", results = "results")

## ----SPRproject_env, eval=FALSE-------------------------------------------------------------------
#  sal <- SPRproject(envir = new.env())

## ----projectInfo, eval=TRUE-----------------------------------------------------------------------
sal
projectInfo(sal)

## ----length, eval=TRUE----------------------------------------------------------------------------
length(sal)

## ----sal_check, eval=TRUE-------------------------------------------------------------------------
sal

## ---- firstStep_R, eval=TRUE----------------------------------------------------------------------
appendStep(sal) <- LineWise(code = {
                              mapply(function(x, y) write.csv(x, y),
                                     split(iris, factor(iris$Species)),
                                     file.path("results", paste0(names(split(iris, factor(iris$Species))), ".csv"))
                                     ) 
                            },
                            step_name = "export_iris")

## ----show, eval=TRUE------------------------------------------------------------------------------
sal

## ----codeLine, eval=TRUE--------------------------------------------------------------------------
codeLine(sal)

## ----gzip_secondStep, eval=TRUE-------------------------------------------------------------------
targetspath <- system.file("extdata/cwl/gunzip", "targets_gunzip.txt", package = "systemPipeR")
appendStep(sal) <- SYSargsList(step_name = "gzip", 
                      targets = targetspath, dir = TRUE,
                      wf_file = "gunzip/workflow_gzip.cwl", input_file = "gunzip/gzip.yml",
                      dir_path = system.file("extdata/cwl", package = "systemPipeR"),
                      inputvars = c(FileName = "_FILE_PATH_", SampleName = "_SampleName_"), 
                      dependency = "export_iris")

## -------------------------------------------------------------------------------------------------
sal

## -------------------------------------------------------------------------------------------------
cmdlist(sal, step="gzip")

## -------------------------------------------------------------------------------------------------
outfiles(sal)

## ----gunzip, eval=TRUE----------------------------------------------------------------------------
appendStep(sal) <- SYSargsList(step_name = "gunzip", 
                      targets = "gzip", dir = TRUE,
                      wf_file = "gunzip/workflow_gunzip.cwl", input_file = "gunzip/gunzip.yml",
                      dir_path = system.file("extdata/cwl", package = "systemPipeR"),
                      inputvars = c(gzip_file = "_FILE_PATH_", SampleName = "_SampleName_"), 
                      rm_targets_col = "FileName", 
                      dependency = "gzip")

## ----targetsWF_3, eval=TRUE-----------------------------------------------------------------------
targetsWF(sal[3])

## ----outfiles_2, eval=TRUE------------------------------------------------------------------------
outfiles(sal[3])

## -------------------------------------------------------------------------------------------------
sal

## ---- eval=TRUE-----------------------------------------------------------------------------------
cmdlist(sal["gzip"], targets = 1)

## ----getColumn, eval=TRUE-------------------------------------------------------------------------
getColumn(sal, step = "gunzip", 'outfiles')

## ---- iris_stats, eval=TRUE-----------------------------------------------------------------------
appendStep(sal) <- LineWise(code = {
                    df <- lapply(getColumn(sal, step = "gunzip", 'outfiles'), function(x) read.delim(x, sep = ",")[-1])
                    df <- do.call(rbind, df)
                    stats <- data.frame(cbind(mean = apply(df[,1:4], 2, mean), sd = apply(df[,1:4], 2, sd)))
                    stats$species <- rownames(stats)
                    
                    plot <- ggplot2::ggplot(stats, ggplot2::aes(x = species, y = mean, fill = species)) + 
                      ggplot2::geom_bar(stat = "identity", color = "black", position = ggplot2::position_dodge()) +
                      ggplot2::geom_errorbar(ggplot2::aes(ymin = mean-sd, ymax = mean+sd), width = .2, position = ggplot2::position_dodge(.9)) 
                    },
                    step_name = "iris_stats", 
                    dependency = "gzip")

## ----importWF_rmd, eval=TRUE----------------------------------------------------------------------
sal_rmd <- SPRproject(logs.dir = ".SPRproject_rmd") 

sal_rmd <- importWF(sal_rmd, 
                file_path = system.file("extdata", "spr_simple_wf.Rmd", package = "systemPipeR"))

## ----importWF_details-----------------------------------------------------------------------------
stepsWF(sal_rmd)
dependency(sal_rmd)
codeLine(sal_rmd)
targetsWF(sal_rmd)

## ----fromFile_example_rules_cmd, eval=FALSE-------------------------------------------------------
#  targetspath <- system.file("extdata/cwl/example/targets_example.txt", package = "systemPipeR")
#  appendStep(sal) <- SYSargsList(step_name = "Example",
#                        targets = targetspath,
#                        wf_file = "example/example.cwl", input_file = "example/example.yml",
#                        dir_path = system.file("extdata/cwl", package = "systemPipeR"),
#                        inputvars = c(Message = "_STRING_", SampleName = "_SAMPLE_"))

## ----fromFile_example_rules_r, eval=FALSE---------------------------------------------------------
#  appendStep(sal) <- LineWise(code = {
#                                library(systemPipeR)
#                              },
#                              step_name = "load_lib")

## ----runWF, eval=is_win---------------------------------------------------------------------------
#  sal <- runWF(sal)

## ----runWF_error, eval=FALSE----------------------------------------------------------------------
#  sal <- runWF(sal, steps = c(1,3))

## ----runWF_force, eval=FALSE----------------------------------------------------------------------
#  sal <- runWF(sal, force = TRUE, warning.stop = FALSE, error.stop = TRUE)

## ----runWF_env, eval=FALSE------------------------------------------------------------------------
#  viewEnvir(sal)

## ----runWF_saveenv, eval=FALSE--------------------------------------------------------------------
#  sal <- runWF(sal, saveEnv = TRUE)

## ----show_statusWF, eval=TRUE---------------------------------------------------------------------
sal

## ----statusWF, eval=TRUE--------------------------------------------------------------------------
statusWF(sal)

## ----clusterRun, eval=FALSE-----------------------------------------------------------------------
#  library(batchtools)
#  resources <- list(walltime=120, ntasks=1, ncpus=4, memory=1024)
#  sal <- clusterRun(sal, FUN = runWF,
#                    more.args = list(),
#                    conffile=".batchtools.conf.R",
#                    template="batchtools.slurm.tmpl",
#                    Njobs=1, runid="01", resourceList=resources)

## ---- eval=TRUE-----------------------------------------------------------------------------------
plotWF(sal, show_legend = TRUE, width = "80%", rstudio = TRUE)

## ---- eval=FALSE----------------------------------------------------------------------------------
#  sal <- renderLogs(sal)

## ---- eval=FALSE----------------------------------------------------------------------------------
#  sal2rmd(sal)

## ---- eval=FALSE----------------------------------------------------------------------------------
#  sal2bash(sal)

## ----SPR_resume, eval=FALSE-----------------------------------------------------------------------
#  sal <- SPRproject(resume = TRUE, logs.dir = ".SPRproject",
#                    sys.file = ".SPRproject/SYSargsList.yml")

## ----resume_load, eval=FALSE----------------------------------------------------------------------
#  sal <- SPRproject(resume = TRUE, load.envir = TRUE)

## ----envir, eval=FALSE----------------------------------------------------------------------------
#  viewEnvir(sal)
#  copyEnvir(sal, list="plot", new.env = globalenv())

## ----restart_load, eval=FALSE---------------------------------------------------------------------
#  sal <- SPRproject(restart = TRUE, load.envir = FALSE)

## ----SPR_overwrite, eval=FALSE--------------------------------------------------------------------
#  sal <- SPRproject(overwrite = TRUE)

## -------------------------------------------------------------------------------------------------
names(sal)

## -------------------------------------------------------------------------------------------------
length(sal)

## -------------------------------------------------------------------------------------------------
stepsWF(sal)

## -------------------------------------------------------------------------------------------------
cmdlist(sal, step = c(2,3), targets = 1)

## -------------------------------------------------------------------------------------------------
statusWF(sal)

## -------------------------------------------------------------------------------------------------
targetsWF(sal[2])

## -------------------------------------------------------------------------------------------------
outfiles(sal[2])

## -------------------------------------------------------------------------------------------------
dependency(sal)

## ---- eval=FALSE----------------------------------------------------------------------------------
#  targetsheader(sal, step = "Quality")

## -------------------------------------------------------------------------------------------------
stepName(sal)

## -------------------------------------------------------------------------------------------------
SampleName(sal, step = "gzip")
SampleName(sal, step = "iris_stats")

## -------------------------------------------------------------------------------------------------
getColumn(sal, "outfiles", step = "gzip", column = "gzip_file")
getColumn(sal, "targetsWF", step = "gzip", column = "FileName")

## -------------------------------------------------------------------------------------------------
codeLine(sal, step = "export_iris")

## -------------------------------------------------------------------------------------------------
viewEnvir(sal)

## -------------------------------------------------------------------------------------------------
copyEnvir(sal, list = c("plot"), new.env = globalenv(), silent = FALSE)

## -------------------------------------------------------------------------------------------------
yamlinput(sal, step = "gzip")

## -------------------------------------------------------------------------------------------------
sal[1]
sal[1:3]
sal[c(1,3)]

## -------------------------------------------------------------------------------------------------
sal_sub <- subset(sal, subset_steps = c( 2,3), input_targets = ("SE"), keep_steps = TRUE)
stepsWF(sal_sub)
targetsWF(sal_sub)
outfiles(sal_sub)

## ---- eval=FALSE----------------------------------------------------------------------------------
#  sal[1] + sal[2] + sal[3]

## ---- eval=TRUE-----------------------------------------------------------------------------------
sal_c <- sal
## check values
yamlinput(sal_c, step = "gzip")
## check on command-line
cmdlist(sal_c, step = "gzip", targets = 1)
## Replace
yamlinput(sal_c, step = "gzip", paramName = "ext") <- "txt.gz"

## check NEW values
yamlinput(sal_c, step = "gzip")
## Check on command-line
cmdlist(sal_c, step = "gzip", targets = 1)

## ---- sal_lw_rep, eval=TRUE-----------------------------------------------------------------------
appendCodeLine(sal_c, step = "export_iris", after = 1) <- "log_cal_100 <- log(100)"
codeLine(sal_c, step = "export_iris")

replaceCodeLine(sal_c, step="export_iris", line = 2) <- LineWise(code={
                    log_cal_100 <- log(50)
                    })
codeLine(sal_c, step = 1)

## -------------------------------------------------------------------------------------------------
renameStep(sal_c, step = 1) <- "newStep"
renameStep(sal_c, c(1, 2)) <- c("newStep2", "newIndex")
sal_c
names(outfiles(sal_c))
names(targetsWF(sal_c))
dependency(sal_c)

## ---- eval=FALSE----------------------------------------------------------------------------------
#  sal_test <- sal[c(1,2)]
#  replaceStep(sal_test, step = 1, step_name = "gunzip" ) <- sal[3]
#  sal_test

## -------------------------------------------------------------------------------------------------
sal_test <- sal[-2]
sal_test

## -------------------------------------------------------------------------------------------------
dir_path <- system.file("extdata/cwl", package = "systemPipeR")
cwl <- yaml::read_yaml(file.path(dir_path, "example/example.cwl"))

## -------------------------------------------------------------------------------------------------
cwl[1:2]

## -------------------------------------------------------------------------------------------------
cwl[3]

## -------------------------------------------------------------------------------------------------
cwl[4]

## -------------------------------------------------------------------------------------------------
cwl[5]

## -------------------------------------------------------------------------------------------------
cwl[6]

## -------------------------------------------------------------------------------------------------
cwl.wf <- yaml::read_yaml(file.path(dir_path, "example/workflow_example.cwl"))

## -------------------------------------------------------------------------------------------------
cwl.wf[1:2]

## -------------------------------------------------------------------------------------------------
cwl.wf[3]

## -------------------------------------------------------------------------------------------------
cwl.wf[4]

## -------------------------------------------------------------------------------------------------
cwl.wf[5]

## -------------------------------------------------------------------------------------------------
yaml::read_yaml(file.path(dir_path, "example/example_single.yml"))

## ----fromFile, eval=TRUE--------------------------------------------------------------------------
HW <- SYSargsList(wf_file = "example/workflow_example.cwl", 
                  input_file = "example/example_single.yml", 
                  dir_path = system.file("extdata/cwl", package = "systemPipeR"))
HW
cmdlist(HW)

## -------------------------------------------------------------------------------------------------
yml <- yaml::read_yaml(file.path(dir_path, "example/example.yml"))
yml

## -------------------------------------------------------------------------------------------------
targetspath <- system.file("extdata/cwl/example/targets_example.txt", package = "systemPipeR")
read.delim(targetspath, comment.char = "#")

## ----fromFile_example, eval=TRUE------------------------------------------------------------------
HW_mul <- SYSargsList(step_name = "echo", 
                      targets=targetspath, 
                      wf_file="example/workflow_example.cwl", input_file="example/example.yml", 
                      dir_path = dir_path, 
                      inputvars = c(Message = "_STRING_", SampleName = "_SAMPLE_"))
HW_mul
cmdlist(HW_mul)

## ----sprCWL, eval=TRUE, echo=FALSE, out.width="100%", fig.align = "center", fig.cap= "WConnectivity between CWL param files and targets files."----
knitr::include_graphics(system.file("extdata/images", "SPR_CWL_hello.png", package = "systemPipeR"))

## ----cmd, eval=TRUE-------------------------------------------------------------------------------
command <- "
    hisat2 \
    -S <F, out: ./results/M1A.sam> \
    -x <F: ./data/tair10.fasta> \
     -k <int: 1> \
    -min-intronlen <int: 30> \
    -max-intronlen <int: 3000> \
    -threads <int: 4> \
    -U <F: ./data/SRR446027_1.fastq.gz>
"

## -------------------------------------------------------------------------------------------------
cmd <- createParam(command, writeParamFiles = FALSE)

## ----saving, eval=FALSE---------------------------------------------------------------------------
#  writeParamFiles(cmd, overwrite = TRUE)

## -------------------------------------------------------------------------------------------------
printParam(cmd, position = "baseCommand") ## Print a baseCommand section
printParam(cmd, position = "outputs")
printParam(cmd, position = "inputs", index = 1:2) ## Print by index
printParam(cmd, position = "inputs", index = -1:-2) ## Negative indexing printing to exclude certain indices in a position

## -------------------------------------------------------------------------------------------------
cmd2 <- subsetParam(cmd, position = "inputs", index = 1:2, trim = TRUE)
cmdlist(cmd2)

cmd2 <- subsetParam(cmd, position = "inputs", index = c("S", "x"), trim = TRUE)
cmdlist(cmd2)

## -------------------------------------------------------------------------------------------------
cmd3 <- replaceParam(cmd, "base", index = 1, replace = list(baseCommand = "bwa"))
cmdlist(cmd3)

## -------------------------------------------------------------------------------------------------
new_inputs <- new_inputs <- list(
    "new_input1" = list(type = "File", preF="-b", yml ="myfile"),
    "new_input2" = "-L <int: 4>"
)
cmd4 <- replaceParam(cmd, "inputs", index = 1:2, replace = new_inputs)
cmdlist(cmd4)

## -------------------------------------------------------------------------------------------------
newIn <- new_inputs <- list(
    "new_input1" = list(type = "File", preF="-b1", yml ="myfile1"),
    "new_input2" = list(type = "File", preF="-b2", yml ="myfile2"),
    "new_input3" = "-b3 <F: myfile3>"
)
cmd5 <- appendParam(cmd, "inputs", index = 1:2, append = new_inputs)
cmdlist(cmd5)

cmd6 <- appendParam(cmd, "inputs", index = 1:2, after=0, append = new_inputs)
cmdlist(cmd6)

## -------------------------------------------------------------------------------------------------
new_outs <- list(
    "sam_out" = "<F: $(inputs.results_path)/test.sam>"
) 
cmd7 <- replaceParam(cmd, "outputs", index = 1, replace = new_outs)
output(cmd7) 

## ----sysargs2, eval=TRUE--------------------------------------------------------------------------
cmd <- "
    hisat2 \
    -S <F, out: _SampleName_.sam> \
    -x <F: ./data/tair10.fasta> \
    -k <int: 1> \
    -min-intronlen <int: 30> \
    -max-intronlen <int: 3000> \
    -threads <int: 4> \
    -U <F: _FASTQ_PATH1_>
"
WF <- createParam(cmd, overwrite = TRUE, writeParamFiles = TRUE, confirm = TRUE) 
targetspath <- system.file("extdata", "targets.txt", package = "systemPipeR")
WF_test <- loadWorkflow(targets = targetspath, wf_file="hisat2.cwl",
                   input_file="hisat2.yml", dir_path = "param/cwl/hisat2/")
WF_test <- renderWF(WF_test, inputvars = c(FileName = "_FASTQ_PATH1_"))
WF_test
cmdlist(WF_test)[1:2]

## -------------------------------------------------------------------------------------------------
plotWF(sal, in_log = TRUE)

## -------------------------------------------------------------------------------------------------
plotWF(sal, plot_method = "png")

## -------------------------------------------------------------------------------------------------
plotWF(sal, rstudio = TRUE)

## -------------------------------------------------------------------------------------------------
plotWF(sal, responsive = FALSE)

## -------------------------------------------------------------------------------------------------
plotWF(sal, layout = "vertical", height = "600px")

## -------------------------------------------------------------------------------------------------
plotWF(sal, layout = "horizontal")

## -------------------------------------------------------------------------------------------------
plotWF(sal, layout = "execution", height = "600px", responsive = FALSE)

## ----collapse=TRUE--------------------------------------------------------------------------------
plotWF(sal, layout = "vertical", branch_method = "choose", branch_no = 1, verbose = FALSE)

## -------------------------------------------------------------------------------------------------
plotWF(sal, mark_main_branch = FALSE, height = "500px")

## -------------------------------------------------------------------------------------------------
plotWF(sal, show_legend = FALSE, height = "500px")

## -------------------------------------------------------------------------------------------------
plotWF(sal, out_format = "html", out_path = "example_out.html")
file.exists("example_out.html")

## -------------------------------------------------------------------------------------------------
plotWF(sal, out_format = "dot", out_path = "example_out.dot")
cat(readLines("example_out.dot")[1:5], sep = "\n")

## ----eval=FALSE-----------------------------------------------------------------------------------
#  plotWF(sal, out_format = "dot_print") #

## ----eval=FALSE-----------------------------------------------------------------------------------
#  # remotes::install_github("rstudio/webshot2")

## ----eval=FALSE-----------------------------------------------------------------------------------
#  #plotWF(sal, out_format = "html", out_path = "example_out.html")
#  # file.exists("example_out.html")

## -------------------------------------------------------------------------------------------------
# webshot2::webshot("example_out.html", "example_out.png")

## ----sysargs2_cwl_structure, echo = FALSE, eval=FALSE---------------------------------------------
#  hisat2.cwl <- system.file("extdata", "cwl/hisat2/hisat2-mapping-se.cwl", package = "systemPipeR")
#  yaml::read_yaml(hisat2.cwl)

## ----sysargs2_yaml_structure, echo = FALSE, eval=FALSE--------------------------------------------
#  hisat2.yml <- system.file("extdata", "cwl/hisat2/hisat2-mapping-se.yml", package = "systemPipeR")
#  yaml::read_yaml(hisat2.yml)

## ----SYSargs2_structure, eval=TRUE----------------------------------------------------------------
library(systemPipeR)
targetspath <- system.file("extdata", "targets.txt", package = "systemPipeR")
dir_path <- system.file("extdata/cwl", package = "systemPipeR")
WF <- loadWF(targets = targetspath, wf_file = "hisat2/hisat2-mapping-se.cwl",
                   input_file = "hisat2/hisat2-mapping-se.yml",
                   dir_path = dir_path)

WF <- renderWF(WF, inputvars = c(FileName = "_FASTQ_PATH1_", 
                                 SampleName = "_SampleName_"))

## ----names_WF, eval=TRUE--------------------------------------------------------------------------
names(WF)

## ----cmdlist, eval=TRUE---------------------------------------------------------------------------
cmdlist(WF)[1]

## ----output_WF, eval=TRUE-------------------------------------------------------------------------
output(WF)[1]

## ---- targets_WF, eval=TRUE-----------------------------------------------------------------------
targets(WF)[1]
as(WF, "DataFrame")

## ---- module_WF, eval=TRUE------------------------------------------------------------------------
modules(WF)

## ---- other_WF, eval=FALSE------------------------------------------------------------------------
#  files(WF)
#  inputvars(WF)

## ----lw, eval=TRUE--------------------------------------------------------------------------------
rmd <- system.file("extdata", "spr_simple_lw.Rmd", package = "systemPipeR")
sal_lw <- SPRproject(overwrite = TRUE)
sal_lw <- importWF(sal_lw, rmd, verbose = FALSE)
codeLine(sal_lw)

## ---- lw_coerce, eval=TRUE------------------------------------------------------------------------
lw <- stepsWF(sal_lw)[[2]]
## Coerce
ll <- as(lw, "list")
class(ll)
lw <- as(ll, "LineWise")
lw

## ---- lw_access, eval=TRUE------------------------------------------------------------------------
length(lw)
names(lw)
codeLine(lw)
codeChunkStart(lw)
rmdPath(lw)

## ---- lw_sub, eval=TRUE---------------------------------------------------------------------------
l <- lw[2]
codeLine(l)
l_sub <- lw[-2]
codeLine(l_sub)

## ---- lw_rep, eval=TRUE---------------------------------------------------------------------------
replaceCodeLine(lw, line = 2) <- "5+5"
codeLine(lw)
appendCodeLine(lw, after = 0) <- "6+7"
codeLine(lw)

## ---- sal_rep_append, eval=FALSE------------------------------------------------------------------
#  replaceCodeLine(sal_lw, step = 2, line = 2) <- LineWise(code={
#                                                               "5+5"
#                                                                  })
#  codeLine(sal_lw, step = 2)
#  
#  appendCodeLine(sal_lw, step = 2) <- "66+55"
#  codeLine(sal_lw, step = 2)
#  
#  appendCodeLine(sal_lw, step = 1, after = 1) <- "66+55"
#  codeLine(sal_lw, step = 1)

## ---- eval=TRUE, echo=FALSE, out.width="100%", fig.align = "center", fig.cap= "Workflow design structure of *`systemPipeR`* using previous version of *`SYSargs`*"----
knitr::include_graphics(system.file("extdata/images", "SystemPipeR_Workflow.png", package = "systemPipeR"))  

## ----table_tools, echo=FALSE, message=FALSE-------------------------------------------------------
library(magrittr)
SPR_software <- system.file("extdata", "SPR_software.csv", package = "systemPipeR")
software <- read.delim(SPR_software, sep = ",", comment.char = "#")
colors <- colorRampPalette((c("darkseagreen", "indianred1")))(length(unique(software$Category)))
id <- as.numeric(c((unique(software$Category))))
software %>%
  dplyr::mutate(Step = kableExtra::cell_spec(Step, color = "white", bold = TRUE,
    background = factor(Category, id, colors)
  )) %>%
   dplyr::select(Tool, Description, Step) %>%
  dplyr::arrange(Tool) %>% 
  kableExtra::kable(escape = FALSE, align = "c", col.names = c("Tool Name",	"Description", "Step")) %>%
  kableExtra::kable_styling(c("striped", "hover", "condensed"), full_width = TRUE) %>%
  kableExtra::scroll_box(width = "80%", height = "500px")

## ----test_tool_path, eval=FALSE-------------------------------------------------------------------
#  tryCMD(command="gzip")

## ----sessionInfo----------------------------------------------------------------------------------
sessionInfo()

