/**
   SSLDHPrivateKey.java

   Copyright (C) 1999, Claymore Systems, Inc.
   All 3
   Rights Reserved.

   ekr@rtfm.com  Sun May  9 16:31:12 1999

   This package is a SSLv3/TLS implementation written by Eric Rescorla
   <ekr@rtfm.com> and licensed by Claymore Systems, Inc.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions
   are met:
   1. Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
   2. Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
   3. All advertising materials mentioning features or use of this software
      must display the following acknowledgement:
      This product includes software developed by Claymore Systems, Inc.
   4. Neither the name of Claymore Systems, Inc. nor the name of Eric
      Rescorla may be used to endorse or promote products derived from this
      software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
   ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
   ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
   FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
   DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
   OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
   HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
   LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
   OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
   SUCH DAMAGE.

   $Id: SSLDHPrivateKey.java,v 1.3 1999/06/22 06:24:53 ekr Exp $

*/


package COM.claymoresystems.ptls;
import java.security.SecureRandom;
import java.math.BigInteger;

class SSLDHPrivateKey
     implements java.security.PrivateKey
{
     private BigInteger X;
     private BigInteger Y;
     private BigInteger g;
     private BigInteger p;
     private SecureRandom rand;
     private static int PRIME_CERTAINTY=50;
     
     public SSLDHPrivateKey(BigInteger g_,BigInteger p_,SecureRandom rand_){
       g=g_;
       p=p_;
       rand=rand_;
       generatePrivate();
     }

     public SSLDHPrivateKey(SecureRandom rand_,int keylength){
       BigInteger p_tmp;
       
       do {
	 p_tmp=new BigInteger(keylength,rand_);
       } while(p_tmp.isProbablePrime(PRIME_CERTAINTY));

       rand=rand_;
       p=p_tmp;
       g=new BigInteger("2");

       generatePrivate();
     }
     
     private byte[] toBytes(BigInteger num){
       byte[] tmp=num.toByteArray();

       if(tmp[0]==0){
	 byte[] trim=new byte[tmp.length-1];

	 System.arraycopy(tmp,1,trim,0,tmp.length-1);
	 return trim;
       }
       else
	 return tmp;
     }
     
     public BigInteger getX(){
       return X;
     }

     public BigInteger getY(){
       return Y;
     }

     public byte[] getYBytes(){
       return toBytes(Y);
     }
	 
     public BigInteger getg(){
       return g;
     }

     public byte[] getgBytes(){
       return toBytes(g);
     }
     
     public BigInteger getp(){
       return p;
     }

     public byte[] getpBytes(){
       return toBytes(p);
     }
     

     public byte[] keyAgree(SSLDHPublicKey pub,boolean check){
       BigInteger pY;
       BigInteger pg;
       BigInteger pp;

       if(check){
	 pp=pub.getp();
	 pg=pub.getg();

	 if(pg!=null || pp != null){
	   if(g.compareTo(pg)!=0)
	     throw new Error("DH parameters don't match (g)");
	   if(p.compareTo(pp)!=0)
	     throw new Error("DH parameters don't match (p)");
	 }
       }
       
       pY=pub.getY();

       return toBytes(pY.modPow(X,p));
     }
       
     private void generatePrivate(){
       int bits=p.bitLength();
       int bytelen=bits/8;
       int bitr=bits%8;
       int shift;
       
       if(bitr>0) bytelen++;
       
       byte[] buf=new byte[bytelen];

       // We want to make a number with the high bit low and
       // the others preserved
       rand.nextBytes(buf);

       shift=(bitr>0)?8-bitr:1;
       buf[0] &= (byte)(0xff>>shift);

       X=new BigInteger(1,buf);

       Y=g.modPow(X,p);
     }

     public String getFormat(){
       return "RAW";
     }

     public byte[] getEncoded(){
       return null;
     }

     public String getAlgorithm(){
       return "DH";
     }
}

     
       
