#!/usr/bin/perl -Tw
#-----------------------------------------------------------------------------
# webitrans.pl
#
# Web interface to "itrans"
# This script is similar to the script used for the Online Interface at:
# http://www.aczoom.com/itrans/online/
# and is provided as an example. There is no documention for this script,
# other than program comments in this file itself.
# -----------------------------------------------------------------------
# This program accepts www web form input using ITRANS encoded-English,
# and produces printouts in Indian Language Scripts.
#-----------------------------------------------------------------------------
my $VERSION = "2.11";
# Created: March 2001.
# Major updates by Avinash Chopde <avinash@aczoom.com> http://www.aczoom.com/
# Minor fixes, last modified: Jun 2008
# Minor: 2.10 to 2.11: note time taken to download, which sometimes is high
#-----------------------------------------------------------------------------
# Based on script by Hari Adiseshu
# - changes by Atul Narkhede (for alkhemy.com website)
# - changes by shree on 3/2/2000
# - changes Shashikant Joshi 03/10/2000
# - changes by Nandu
# -----------------------------------------------------------------------
# March 2001
# Many changes.
# --> Accept file upload also (multipart forms).
# --> .ps to .gif transformation modified to create anti-aliased text.
# --> more fonts added, more sizes added.
# --> two formats added 
#     "itrans-itx" for handling .itx files directly
#     "isongs-s" for handling .s ISONGS files directly
# --> HTML output in both ISO-8859-1 (Latin1) and Unicode UTF-8
# May 2003
# --> Added "fileurl" input field, to read in data from a HTTP url
#     This is not displayed on the web page, usually a hidden field
# Nov 2004, use LWP library to GET URLs instead of opening sockets here
# --> 
# Jun 2008, for unicode output, call itrans-post.pl to fix characters
#------------------------------
# Nov 2009, now works with texlive. See ITRANS top level CHANGES file
# for config - all that is needed is to update the texlive config file
# with the /usr/share/itrans53/lib path, and put this before all texlive
# folders since texlive may have an older version of idevn.tex.
# This release also uses dvnb10 instead of dvng10 Velthuis font, just like
# ITRANS 5.31, also released in Nov 2009.
#------------------------------.
# Rough outline of things needed to make this work at your site:
# Programs needed, on Linux: TeX (teTeX)
# GhostScript - (on slow pre-1998 PCs, use version 5.10 - twice as fast as 6.5)
# netpbm - using version 9.22 (Dec 2001)
# itrans - install itrans 5.3+, normally goes in /usr/local/itransNN
# CGI.pm - latest one - 2.752 or newer
# ----
# Following steps outline things I had to do to get my Linux (Slackware)
# system up and running.
# I keep all my web files under the /home/cgi folder, but these are not
# all under the httpd /var/www/html folder - only webitrans.pl needs to
# be in the cgi-bin folder of httpd.
# So, the ROOTDIR for web interface to itrans is at /home/cgi/webitrans.
# /home/cgi/ also contains two other folders: tmp and bin, used by all
# CGI tools I serve from this Linux box.
#
# Make this folder non-writeable:
# $ROOTDIR
# Safest is to make all files and directories non-writeable,
# and all directories below $ROOTDIR should be additionally non-readable
# by group and others (ROOTDIR itself needs to be readable, Apache reads
# it to find index.html, index.shtml, etc).
# commands: chmod -R a-w <ITRANS/ROOTDIR>; chmod go-r <all subdirectories>
# 
# Files/folders in $ROOTDIR:
#   commonWebInt.txt
#   isonline.txt (read-only file, existence needed for script to run)
#  Template files, defined in script:
#   $INITIAL_TEMPLATE, $FORM_TEMPLATE, $WORKING_TEMPLATE, $DONE_TEMPLATE
#   $TEX_BASE
#   $LAT1_BASE
#   $UTF8_BASE
#   $TEX_BASE_INC
#   $LAT1_BASE_INC
#   $UTF8_BASE_INC
#
# Additional  files in ROOTDIR:
#   prepi2h.pl
# Files in http:/cgi-bin/
#   webitrans.pl     [I symlink /home/cgi/webitrans/webitrans.pl to this]
# ----
#   /home/cgi/tmp --> Make this folder writeable by everybody
#
# Config files:
#   /home/cgi/webitrans/dvipsrc [not necessary with texlive and ITRANS 3.1]
# ---
# Finally, edit the top section of this CGI script to point to files at
# your site.
# To test, run with the -t option, using the example webtest.txt or similar
# files:
# webitrans.pl -t /tmp/webtest.txt
# -t <filename> should use full path for filename.
# THEN:
# 1: run dvips on test file, using all possible fonts of the
#    web interface. [use webitrans.pl -t ~cgi/itrans/webtest.txt to 
#    create .tex/.dvi files.]
#    dvips <name of .dvi file>
# 2: this will create some *pk files in current folder. Then, as root, do:
#    mkdir -p /var/cache/fonts/pk/jetiiisi/other
#    or mkdir -p /var/lib/texmf/pk/jetiiisi/other
#    mv *pk <folder created above>
# 3: update ls-lr [as root]
#    mktexlsr
# More info:
# This script configures TeX/LaTeX to never run mktextfm, mktekpk when
# running under the web - so, an assumption is made that each
# web admininstrator will create all required TFM/PK files (use the
# test mode -t webtest.txt of this script, and then manually run dvips
# on the created .tex file - in the tmp folder).
# Of course, this is to the discretion of the web admin - you can decide
# whether to turn this on or off -- see the LATEX_E/DVIPS_E vars below.
# -- Make dvips use 300 dpi fonts - jetiiisi mode, makes for much smaller
# .ps files than if 600 dpi/ljfour is used! [
# Run "texconfig-sys" (available on Linux if using teTeX). Also:
# In your shell, set: export DVIPSRC="~cgi/itrans/dvipsrc"; ]
# -- Hardest thing to do on my machine was getting teTeX to put *pk
# fonts in the right place! Could not get that work - I wanted
# fonts to go in /var/cache/fonts, but mktexpk always put ITRANS fonts
# in current directory.
# dvips/mktexpk did put the cmr*pk fonts in /var/cache/fonts, it only
# messed up on dvng, tel, kan, wntml fonts. BUT - even the cmr*pk files
# were not handled correctly - I had enabled "nomode" (don't use printer
# name subdir for pk files), while mktexpk honored that, dvips still
# looked for that folder to search for cmr*pk files - which did not work
# of course! So, each time I ran dvips, it wanted to ceated the cmr*pk files!
# kpathsea in teTeX and texconfig has problems - don't disable
# "mode" from "Font directories -> Options" - it will mess up a lot!
# Don't change the defaults, they cause more trouble than it is worth.
# I run dvips in -M mode (don't run maketexpk), so I pre-created all
# the required *pk files. These will be left in the current dir.
# I then copied them to the /var/cache/fonts/pk/jetiiisi/other folder
# [I created the "other" folder in the standard /var/cache/fonts/pk/jetiiisi
# folder]
# Then, run "mktexlsr" (this will update the /var/cache/fonts/ls-R file.)
# Font bitmaps needed: [added June 17, was missing cmr*pk files on server]
#  /var/cache/fonts/pk/: (or /usr/share/texmf/fonts/pk:)
#  total 4
#  drwxr-xr-x    4 root     root         4096 Jun 17 15:40 jetiiisi/
#  
#  /var/cache/fonts/pk/jetiiisi: (or /usr/share/texmf/fonts/pk/jetiiisi:)
#  total 8
#  drwxr-xr-x    2 root     root         4096 Dec 10  2001 other/
#  drwxr-xr-t    3 root     root         4096 Jun 17 15:40 public/
#  
#  /var/cache/fonts/pk/jetiiisi/other:
#  total 584
#  -rw-r--r--    1 avinash  users       82656 Dec 10  2001 dvng10.1200pk
#  -rw-r--r--    1 avinash  users       16916 Dec 10  2001 dvng10.300pk
#  -rw-r--r--    1 avinash  users       20508 Dec 10  2001 dvng10.360pk
#  -rw-r--r--    1 avinash  users       32460 Dec 10  2001 dvng10.540pk
#  -rw-r--r--    1 avinash  users       12452 Dec 10  2001 kan12.375pk
#  -rw-r--r--    1 avinash  users       14208 Dec 10  2001 kan18.300pk
#  -rw-r--r--    1 avinash  users       23608 Dec 10  2001 kan18.467pk
#  -rw-r--r--    1 avinash  users       57712 Dec 10  2001 kan18.900pk
#  -rw-r--r--    1 avinash  users       15428 Dec 10  2001 tel12.375pk
#  -rw-r--r--    1 avinash  users       17704 Dec 10  2001 tel18.300pk
#  -rw-r--r--    1 avinash  users       29316 Dec 10  2001 tel18.467pk
#  -rw-r--r--    1 avinash  users       71756 Dec 10  2001 tel18.900pk
#  -rw-r--r--    1 avinash  users       13788 Dec 10  2001 wntml12.300pk
#  -rw-r--r--    1 avinash  users       20736 Dec 10  2001 wntml17.295pk
#  -rw-r--r--    1 avinash  users       31052 Dec 10  2001 wntml17.399pk
#  -rw-r--r--    1 avinash  users       81288 Dec 10  2001 wntml17.833pk
#  
#  /var/cache/fonts/pk/jetiiisi/public:
#  total 4
#  drwxr-xr-t    2 root     root         4096 Jun 17 15:40 cm/
#  
#  /var/cache/fonts/pk/jetiiisi/public/cm:
#  total 20
#  -rw-r--r--    1 root     root         5524 Jun 17 15:40 cmr10.300pk
#  -rw-r--r--    1 root     root         3800 Jun 17 15:40 cmr6.300pk
#  -rw-r--r--    1 root     root         4620 Jun 17 15:40 cmr8.300pk
#----
# Install latex2html (latest version) 
# http://saftsack.fs.uni-bayreuth.de/~latex2ht/current/
# This is needed just to get correct html.sty, so if any user
# sends any ITRANS doc/*.itx file, it will be processed correctly
# by this Online ITRANS interface.
# latext2html is also useful for the pstoimg script - it can create
# multipage GIF output:
# pstoimg -multipage -aaliastext -crop tblr -density 100 -white -transparent -type gif -out <output>.gif <input>.ps
#----
# TeX security - texmf.cnf file (in /usr/share/texmf/web2c)
# turn off shell_escape \write18,
# restrict input, openin, openout - how to do it for stuff under
# this perl script only? Don't know - for now, edit texmf.cnf
# and set openin_any = p (to paranoid) and openout_any p (to paranoid)
# and shell_escape = f  (write18 not allowed).
# May 2001: Only openout set to paranoid for now, openin left as is...
#----
# Apache: turn off Indexes directive for all Directories to prevent directory
# listings (since tmp directories here are readable by all, this is done
# to prevent casual users from seeing file names in some other tmp folder).
#----
# Sep 2004
# Add REMOTE_ADDR to temp directory
# Reason: using just hour/minute in the dir, there was one site
# continually requesting conversion, causing the 0 to 999 folder names
# be filled up very fast. Adding REMOTE_ADDR - IP address will prevent
# that one IP address from spamming the system... only one allowed per
# minute from a particular IP address
#----
# Apr/May 2006
# use hex address format for IP in directory name, reduce number of characters
# use HTML::Template to present initial page, as well as working and done pages.
# use HTML::Template and HTML::FillInForm to present form on output page.
#
#-----------------------------------------------------------------------------
use 5.005; # perl newer than 5.005 required
use CGI 3.21 qw(escapeHTML); # version 2.50 for Vars(), 3.21 for POST_MAX fix
use CGI::Carp qw(fatalsToBrowser);
use File::Copy;
use File::Basename;
use Cwd;
use Getopt::Std;
use POSIX qw(floor);
use LWP;
use Socket qw(:DEFAULT :crlf);
use Net::IP;
use HTML::Template;
use HTML::FillInForm;
#-----------------------------------------------------------------------------
$CGI::POST_MAX=1024 * 500;  # max size posts accepted, bytes
# must call before any CGI operation

$sent_working_header = 0;

$start_time = time();

$query = new CGI;
$cgierror = $query->cgi_error(); # post too big, or user hit STOP, etc...

getopts('t:'); # -t <filename> is test mode. Filename is used as $instring

{ # was sub BEGIN, but looks like I cannot call subroutines in here...
    # HOMEDIR is a temporary variable, used in setting all the global
    # variables used by this script. You don't have to use this or set it.
  $HOMEDIR="/home/cgi"; # contains tmp/  bin/  itrans/  cdinsert/  etc

    # ROOTDIR dir is the folder which Apache uses for http:/$ROOTHTTP (locally)
    # "/usr/local/apache/htdocs/cgi" is symlink to "/home/cgi" on my machine:
  $ROOTDIR = "$HOMEDIR/webitrans"; # contains html files

    # log file - global script issues, not related to any one invocation
    # make sure this file exists, and is chmod 662 - writeable, but not readable
    # by others so casual web hackers can't read this directly using a browser.
  $LOGFILE = "$HOMEDIR/tmp/weblog.txt";

    # isonline file - this file should exist, and be readble for this
    # script to run, otherwise, this script will exit without doing any work
    # [2007 - not implemented here yet - is implemented in web client,
    # which may be a better place, though it will not stop direct calls
    # to this CGI script ]
  # $ISONLINEFILE = "$ROOTDIR/isonline.txt";

  &initialize();

  $hostname = $query->remote_host() || '';
  $hostaddr = $query->remote_addr() || '';

  # untaint variable used in dir creation
  $hostaddr =~ /([\d\w\-\.]*)/; # no / allowed in name
  $hostaddr = $1;
  $hostaddrhex = $hostaddr;
  $ip = new Net::IP($hostaddr);
  if (defined $ip && $ip) { $hostaddrhex = sprintf("%08X", $ip->intip()); }
  else { $hostaddrhex = $hostaddr; }

# collect user input

$FORMATPROSE = "itrprose"; # leave line breaking to TeX, or Web Browser, etc
$FORMATITX = "itrans-itx"; # itrans .itx file provided as input
$FORMATISONGS = "isongs-s"; # itrans song book .s file provided as input

  if ($opt_t) {
    $hostname = qq(testing mode "-t $opt_t");
    $useragent = 'test mode - no useragent';
    $referer = '';
    $inlanguage = 'sanskrit';
    $infontsize = 'normal';
    $informat = 'itrlines';
    $infile = '';
    $infileurl = '';
    $inoutput = '';
    open(INPUT, "$opt_t")
      or die("Could not open input file (-t $opt_t) [Make sure full path given -t /tmp/x.txt etc]: $!");
    while (<INPUT>) {
	# read each line to get correct EOLN value for this platform (works?)
	s/$CR?$LF/\n/; # variables from Socket package
        $instring .= $_; 
    }
    close INPUT;
  } else {
      # got CGI query
    $useragent = $query->user_agent() || 'CGI query missing useragent';
    $referer = $query->referer() || '';
    $inlanguage = $query->param('language') || 'sanskrit';
    $infontsize = $query->param('fontsize') || 'small';
    $informat = $query->param('format') || $FORMATPROSE;
    $inoutput = $query->param('output') || 'itrtex' ;
    $instring = $query->param('string') || '';
    $infile = $query->param('filename') || '';
    $infileurl = $query->param('fileurl') || '';
  }
}

#-----------------------------------------------------------------------------
$DEBUG = 1; # 0 no debug messages, 1 some messages, 2 more.
# Debug messages may go to Apache error_log at the beginning, but
# once the LOGFILE/MSGFILE is created, they will go there.

#----------------------------
# Following vars need to be set specifically for each site 

$ENV{'PATH'}="/bin:/usr/bin:/usr/local/bin:$HOMEDIR/bin";
# security blanket - make sure all folders/files are non-writeable by others!

$TEX_BASE = "$ROOTDIR/itrtex.itx"; # itrans input, latex mode
$LAT1_BASE = "$ROOTDIR/itrlat1.itx"; # itrans input, html latin1 mode
$UTF8_BASE = "$ROOTDIR/itrutf8.itx"; # itrans input, html unicode mode

$TEX_BASE_INC="itrtex.inc"; # $TEX_BASE will have line #include=<this file>
$LAT1_BASE_INC="itrlat1.inc"; # $LAT1_BASE will have line #include=<file>
$UTF8_BASE_INC="itrutf8.inc"; # $UTF8_BASE will have line #include=<file>

# .tmpl files are HTML::Template files - all in $ROOTDIR
$INITIAL_TEMPLATE="initial.tmpl"; # first page, diplayed with form

$FORM_TEMPLATE="form.tmpl"; # just the form

$WORKING_TEMPLATE="working.tmpl"; # "working..." display template - this is
# the HTML output of this script  to stdout, while it is doing the processing

$DONE_TEMPLATE="done.tmpl"; # Template page with form (includes form.tmpl)

# for each invocation - semi-random name for files - gets some privacy, since
# these folders and files are readable by the world.
$WORKID = floor(rand(1e4)); # max 4 digits
$WORKTEX = "tex" . $WORKID; # file name to use for intermediate files
$WORKLAT1 = "lat1" . $WORKID; # file name to use for intermediate files
$WORKUTF8 = "utf8" . $WORKID; # file name to use for intermediate files
$SAVEFORM = "form" . $WORKID; # save form data here

$SAVINPUT="input.txt"; # file exact copy of form input string, saved for user

#-----------------------------------------------------------------------------
# check if we need to do any work

# if called without any form data (or -t test option), then just display
# a page with the form and exit
unless ($opt_t || $cgierror || $query->param()) {
  my $tmpl = HTML::Template->new(filename => $INITIAL_TEMPLATE,
             path => [$ROOTDIR], die_on_bad_params => 0);
  print $tmpl->output();
  exit(0);
}

#-----------------------------------------------------------------------------
$SIG{HUP} = $SIG{INT} = $SIG{QUIT} = $SIG{PIPE} = $SIG{TERM} = \&sighandler;
# --------------------------------
# now create the temporary work directory and associated file path names
# create temp folder name - incrementing directory number for given
# remote host ipaddress.
# if a directory cannot be created, this process will exit
$TDATE = sprintf("%02d%02d%1d", (localtime($start_time))[3], (localtime($start_time))[2], (localtime($start_time))[1]/10); # current date hour minute(tens)
# $TDATE = sprintf("%02d%02d", (localtime($start_time))[3], (localtime($start_time))[2]); # current date hour
$WORKDIRNAME = "it" . $TDATE;
# $WORKDIRNAME = &mktempdir("$HOMEDIR/tmp", "$WORKDIRNAME-$hostaddr", floor(rand(1e2))); # 1e3 -> max 3 digits, 1e2 - 2 digits, etc
$WORKDIRNAME = &mktempdir("$HOMEDIR/tmp", "$WORKDIRNAME-$hostaddrhex", "1"); # no random number suffix
# --------------------------------

$WORKDIR = "$HOMEDIR/tmp/$WORKDIRNAME"; 
$WORKHTTP = "/cgi/tmp/$WORKDIRNAME"; 

$MSGFILENAME = "log$WORKID.txt";
$MSGFILE = "$WORKDIR/$MSGFILENAME";
# STDOUT/STDERR messages collected here, different file for each invocation
# STDERR is redirected to $LOGFILE

# input templates, include files: all these files reside in $ROOTDIR
# define these after call to mktempdir
# to allow myexit()/send_working_trailer() to work
$WEBDONE_HTML = "done$WORKID.html"; # complete HTML file (created in $WORKDIR)

# following may be left alone, in most cases
$ENV{'SHELL'} = "/bin/sh";
$ENV{'TMPDIR'} = $WORKDIR;
$ENV{'TEMP'} = $WORKDIR;
$ENV{'TZ'} = "EST5EDT";

# Using ITRANS-5.31/INSTALL.unix file doc for texlive, its texmf.cnf
# contains things needed for TEXPSHEADERS, etc
# None of the following are needed, as of texlive 2008 and ITRANS 5.31 2009
# $ENV{'DVIPSRC'} = "$ROOTDIR/dvipsrc";
# DVIPSRC can be omitted if the system psfonts.map (not user-specific one)
# contains the right itrans.map entries - see ITRANS 5.31 INSTALL.unix for
# details.
# $ITRANSDIR = "/usr/share/itrans53"; # where itrans has been installed
# $ENV{'TEXFONTS'} = ":$ITRANSDIR/lib/fonts";
# $ENV{'TEXPSHEADERS'} = ":$ITRANSDIR/lib/fonts";
# $ENV{'MFINPUTS'} = ":$ITRANSDIR/lib/fonts";
# $ENV{'TEXINPUTS'} = "$ROOTDIR:";

delete @ENV{qw(IFS CDPATH ENV BASH_ENV)};   # Make %ENV safer

# constants ----------------------------------------------------------

# Each language/script supported:
@SCRIPTS = (
    'gujarati',
    'bengali',
    'marathi',
    'hindi',
    'sanskrit',
    'tamil',
    'telugu',
    'kannada',
    'gurmukhi',
    'roman',
);

# Each text size supported, array holds HTML output to be produced
%FONTSIZES = (
    'small', 'SIZE="-1"',
    'normal', 'SIZE="+0"',
    'large', 'SIZE="+2"',
    'huge', 'SIZE="+4"',
);

# --- MRKR_* strings found in input HTML template files,
# to be replaced with job-specific values. See sub replace_markers()
# for list of all the marker tags - only non-HTML::Template tags is
# SHOWSTATUS_HERE
$SHOWSTATUS_HERE = "SHOWSTATUS_HERE\n"; # split WORKING_TEMPLATE on this line

#--- tag printed in the HTML file, denotes start and end of user entered text
# must be a HTML comment
$HTML_TAG_USER_TEXT = "<!-- User Text -->";

#--- executables - since am setting PATH, no need to give full path for each

$REDIRECT="< /dev/null 2>&1";

# -- cmd 0:
$ITRANS_E = "itrans -i $WORKTEX.itx -o $WORKTEX.tex $REDIRECT"; # tex 
# -- cmd 1:
$LATEX_E = "latex -no-mktex=tfm $WORKTEX.tex $REDIRECT"; # don't run mktextfm
# -- cmd 2:
# -d 3650 tex path debug
$DVIPS_E = "dvips -M -R -mode jetiiisi $WORKTEX.dvi -o $WORKTEX.ps $REDIRECT"; # don't run mktexpk (-M), and run in secure mode (-R)
# -- cmd 3:
$PS2PGM_E = "gs -q -dNOPAUSE -dBATCH -dTextAlphaBits=4 -dGraphicsAlphaBits=4 -r100 -sDEVICE=pgmraw -sOutputFile=$WORKTEX.pgm -f $WORKTEX.ps $REDIRECT";
# gs options:
#   - made it anti-alias text (-dTextAlphaBits)
#   - use 100dpi resolution (default is 72, I think)
#   - pgm (gray scale) is faster and 66% smaller file than ppm (full color)
#     pgmraw is even more smaller, and is 10 times faster too, for large files!
#   ? doing an initial 'save' helps keep fonts from being flushed between pages.
# -- cmd 4:
$PGM2GIF_E = "(pnmcrop $WORKTEX.pgm | pnmpad -white  -left=5 -right=5 -top=5 -bottom=5 | ppmtogif -transparent rgb:ff/ff/ff -interlace) 2>&1 > $WORKTEX.gif";
# 2>&1 and then > file makes STDERR go to old STDOUT, and new STDOUT to file
# -- cmd 5:
$PS2PDF_E = "ps2pdf $WORKTEX.ps $REDIRECT";

# HTML output commands

# UTF-8 charset
$ITRANSU_E = "(itrans -U -i $WORKUTF8.itx | itrans-post.pl -o $WORKUTF8.htm) $REDIRECT"; # html UTF-8

# user-defined charset (fake ISO-Latin1 fonts displaying Indian Languages)
$PREPI2H_E = "$ROOTDIR/prepi2h.pl $LAT1_BASE_INC > tmp$LAT1_BASE_INC $REDIRECT && mv tmp$LAT1_BASE_INC $LAT1_BASE_INC $REDIRECT"; # html
$ITRANSH_E = "itrans -8 -i $WORKLAT1.itx -o $WORKLAT1.htm $REDIRECT"; # html

@ALLCMDS = ($ITRANS_E, $LATEX_E, $DVIPS_E, $PS2PDF_E, $PS2PGM_E, $PGM2GIF_E);
@ALLHTMLCMDS = ($ITRANSU_E, $PREPI2H_E, $ITRANSH_E);

#-----------------------------------------------------------------------------
# initialize 

$time_taken = 0;
$time_taken_units = $time_taken;
$status_count = -2;
$errflag = 0;
$errcmd = -1;
$dontredirect = ""; # make this 1 to prevent automatic redirect in WORKING_TEMPLATE
$form_html = ""; # filled in form, displayed on results page

chdir $WORKDIR or myexit("cd ($WORKDIR) failed: $!");

open(MSGFILE, ">> $MSGFILE")  
  or myexit("Could not open message file ($MSGFILE): $!");
select((select(MSGFILE), $| = 1)[0]); # set to autoflush

open(WEBDONE_HTML, "> $WORKDIR/$WEBDONE_HTML")
  or myexit("Could not open WEBDONE_HTML file ($WORKDIR/$WEBDONE_HTML): $!");

open(SAVEFORM, ">> $SAVEFORM")  
  or myexit("Could not open SAVEFORM file ($SAVEFORM): $!");

open(WORKING_TEMPLATE, "< $ROOTDIR/$WORKING_TEMPLATE")
  or myexit("Could not open WORKING_TEMPLATE file ($WORKING_TEMPLATE): $!");

$datestr = localtime($start_time);
print MSGFILE "[$datestr] webitrans.pl $VERSION: Starting job in directory $WORKDIRNAME\n";

print LOGFILE "$WORKDIRNAME itrans [$datestr] $hostname - started\n";

# ---- copy WORKING_TEMPLATE to output, piece by piece, showing status
# -- this relies on the client side of things - the web browser - to display
# the page even though it is not complete. Most browsers do this - IE on PC,
# Netscape on PC and Mac. But IE on Mac waits until it sees the last </BODY>
# so this does not work well in all cases... 

# start_html etc is not used - instead, a template HTML file is read
# in and output to STDOUT
read(WORKING_TEMPLATE, $working_all_lines, $CGI::POST_MAX)
	    or myexit("Could not read WORKING_TEMPLATE file ($WORKING_TEMPLATE): $!");
($working_header, $working_trailer) = split(/$SHOWSTATUS_HERE/, $working_all_lines, 2);

$working_header_tmpl = HTML::Template->new(scalarref => \$working_header,
             path => [$ROOTDIR], die_on_bad_params => 0);

&replace_markers($working_header_tmpl);

# sent to client browser - half - the HTML page - after this, we will
# print one line at a time, using show_status, for each command as it is
# executed
$working_header_tmpl->output(print_to => *STDOUT);
$sent_working_header = 1;

print &show_status("");

# allocate trailer after the header, as soon as possible,
# in case of any calls to myexit(), so that trailer can be displayed
if (!defined $working_trailer) {
  myexit("Internal error - $WORKING_TEMPLATE missing SHOWSTATUS_HERE marker.");
}
$working_trailer_tmpl = HTML::Template->new(scalarref => \$working_trailer,
             path => [$ROOTDIR], die_on_bad_params => 0);

#----------------------------------------------------------------
$html_only = ($inoutput eq 'itrhtmlonly'); # html latin1 and utf8 output only...

print MSGFILE "[$datestr] checking cgi_error...\n";

if ($cgierror) {
    if ($cgierror =~ /413/) {
	myexit("Uploaded files too large?<br/> Received too much data - got <b>" . int($ENV{'CONTENT_LENGTH'}/1024) . "</b> KBytes, can only receive maximum of <b>" . int($CGI::POST_MAX/1024) . "</b> KBytes.");
    } else {
	myexit($cgierror);
    }
}

$datestr = localtime(time());
print MSGFILE "----------------------------------------------------\n";
print MSGFILE "Got these values from the form:\n";
my %params = $query->Vars();
while (($key, $value) = each %params) {
    # assuming value is single string - if multi-valued, need
    # to split on \0 to get array of values...
    $value = "<see file $SAVINPUT>" if ( $key =~ /^string$/ );
    print MSGFILE "  $key = '$value'\n";
}
print MSGFILE "Some environment vars:\n";
print MSGFILE "  remote_host = '$hostname'\n";
print MSGFILE "  user_agent = '$useragent'\n";
print MSGFILE "  referer = '$referer'\n";
print MSGFILE "----------------------------------------------------\n";

$gotstring = ($instring =~ /\S+/);
$gotfile = ($infile =~ /\S+/);
$goturl = ($infileurl =~ /\S+/);

($gotstring || $gotfile || $goturl) or myexit("Empty input - no input text or file found!");

# save users string in file, in case they need to use it again
open(OUTPUT, "> $SAVINPUT")
  or myexit("Could not open output file to save form input ($SAVINPUT): $!");
print MSGFILE "opened file $SAVINPUT to save input text\n" if ($DEBUG >= 1);

# save form data
$query->save(\*SAVEFORM);

$fname = "";
if ($gotfile) { # ignore $gotstring, file takes precedence
    if ($gotstring) {
	print MSGFILE "** Warning: user entered text as well as filename, ignoring text.\n",
    }
    while (<$infile>) {
	# read each line to get correct EOLN value for this platform (works?)
	s/$CR?$LF/\n/; # variables from Socket package
	print OUTPUT $_;
    }

    print MSGFILE "... read in uploaded file: $infile\n" if ($DEBUG >= 1);
    $fname = $infile;
} elsif ($goturl) {
    my $ua = LWP::UserAgent->new();
    $ua->agent("webitrans.pl/0.1 " . $ua->agent);
    my $http_get = HTTP::Request->new(GET => '');

    $http_get->uri( $infileurl );
    $response = $ua->request($http_get);
    if ($response->is_success && $response->content_type =~ /^text/) {
	print OUTPUT $response->content;
	$fname = $infileurl;
	print MSGFILE "... read in URL : $infileurl\n...  status: (".
	       $response->status_line .
	       ")\n...  type:   (" .  $response->content_type .
	       ")\n... request was: (\n" . $http_get->as_string() .
	       "\n)\n" 
	       if ($DEBUG >= 1);
    } else {
        myexit("Input File Retrieve Failed: ($infileurl)\n Status: (" .
	       $response->status_line .
	       ")\n Type should be text: (" .  $response->content_type .
	       ")\n Request was: (\n" . $http_get->as_string() .
	       ")\n");
    }
} else {
    # fix for correct EOLN value for this platform (works?)
    $instring =~ s/$CR?$LF/\n/g; # variables from Socket package
    print OUTPUT  $instring;
}
close OUTPUT;

if ($fname =~ /\.itx$/ && $informat !~ /^$FORMATITX$/) {
    # file name ends in .itx, but informat is wrong...
    $informat = $FORMATITX; # force format 
    print MSGFILE "** Warning: user sent .itx file, but wrong page format, changed to $FORMATITX.\n";
}
if ($fname =~ /\.s$/ && $informat !~ /^$FORMATISONGS$/) {
    # file name ends in .s, but informat is wrong...
    $informat = $FORMATISONGS; # force format 
    print MSGFILE "** Warning: user sent .s file, but wrong page format, changed to $FORMATISONGS.\n";
}

# have now read in form data or filurl data, and can begin processing
$start_processing_time = time();

$datestr = localtime($start_processing_time);
print MSGFILE "----------------------------------------------------\n";
print MSGFILE "[$datestr] $hostname :: Starting programs...\n";
print MSGFILE "----------------------------------------------------\n";

#-----------------------------------------------------------------------------
# two formats need to be treated specially - .itx files and .s files
# all other formats are web-input text formats, containing commands
# like {script-size} that need to be replaced by ITRANS commands

# LaTeX pre-processing....
if ($informat =~ /^$FORMATITX$/) {
    # format is $FORMATITX, use the user supplied file or text directly
    copy($SAVINPUT, "$WORKTEX.itx")
      or myexit("Could not copy ($SAVINPUT) to ($WORKTEX.itx): $!");
} else {
    # handle all other formats, including $FORMATISONGS
    &preprocess_itrinput();
}

# HTML output pre-processing....
if ($informat =~ /^$FORMATITX$/) {
    # format is $FORMATITX, use the user supplied file or text directly
    copy($SAVINPUT, "$WORKLAT1.itx")
      or myexit("Could not copy ($SAVINPUT) to ($WORKLAT1.itx): $!");
    copy($SAVINPUT, "$WORKUTF8.itx")
      or myexit("Could not copy ($SAVINPUT) to ($WORKUTF8.itx): $!");
} else {
    # handle all other formats, including $FORMATISONGS
    &preprocess_htminput();
}

#-----------------------------------------------------------------------------
# now have $WORKTEX.itx and $WORKLAT1/UTF8.itx
# got all data in ITRANS format, now run all the programs to finally
# get the .gif and .ps files, as well as HTML output

# first, create HTML output

print MSGFILE "-------  Creating HTML Output - UTF-8 and Latin1 ---\n";
print MSGFILE "----------------------------------------------------\n";

foreach $i (0 .. $#ALLHTMLCMDS) {
    my ($t1, $td);

    $cmd = $ALLHTMLCMDS[$i];
    print &show_status($cmd); # status messages to web 0 .. 1 .. etc
    $t1 = time();
    $out = `$cmd`;
    $td = time() - $t1;
    $returncode = ($? >> 8); 

    print MSGFILE qq("$cmd" executed\n -- Took $td seconds, returns $returncode\n);
    print MSGFILE "$out\n---------------------\n";

    if ($returncode != 0) {
	if ($html_only) {
	    myexit("ITRANS failed - HTML output mode.");
        } else {
	    print "<BR>\n";
	    print "Session ID# <B>$WORKDIRNAME</B> - Warning - HTML output not produced\n<BR>\n";
	    print "Warning: problem executing \"$cmd\", will continue anyway.\n";
	    print "<P>\n";
	}
	# last; # no last continue - try other commands - UTF8/Latin1/etc
    }
}

# add <BR> if needed to the  output html files
$returncode = &postprocess_htm();
if ($returncode != 0) {
    if ($html_only) {
	myexit("ITRANS post-processing failed - HTML output mode.");
    } else {
	print "<BR>\n";
	print "Session ID# <B>$WORKDIRNAME</B> - Warning - HTML output not produced correctly\n<BR>\n";
	print "Warning: problem post-processing HTML files, will continue anyway.\n";
	print "<P>\n";
    }
}

if (! $html_only) {
    #------------------------------------------------------------------------
    # second, run all the programs to get GIF/PS etc output, LaTeX processing
    print MSGFILE "-------  Creating GIF/PS/PDF Output ----------------\n";
    print MSGFILE "----------------------------------------------------\n";

    foreach $i (0 .. $#ALLCMDS) {
	my ($t1, $td);

	$cmd = $ALLCMDS[$i];
	print &show_status($cmd); # status messages to web 0 .. 1 .. etc
	$t1 = time();
	$out = `$cmd`;
	$td = time() - $t1;
	$returncode = ($? >> 8); 

	print MSGFILE qq("$cmd" executed\n -- Took $td seconds, returns $returncode\n);
	print MSGFILE "$out\n---------------------\n";

	if ($returncode != 0) {
	    print "<P>\n";
	    print "Session ID# <B>$WORKDIRNAME</B> - an error occured, cannot proceed.\n<BR>\n";
	    # print "Command: ", escapeHTML($cmd), ", \$? is $?\n<BR>";
	    # $out = escapeHTML($out);
	    # $out =~ s/\n/\n<BR>/;
	    # print qq($out\n<BR>\n);
	    $errflag++;
	    $errcmd = $i;
	    last;
	}
    }
}

print " Done!\n<P>\n"; # status messages to web...

print MSGFILE "commands executed. \$errflag='$errflag' \$errcmd='$errcmd'\n";

# remove intermediate log files
unlink("$WORKTEX.tex")
   unless ($errflag || $opt_t || $DEBUG >= 1);
unlink("$WORKTEX.aux", "$WORKTEX.dvi", "$WORKTEX.log", "$WORKTEX.pgm")
   unless ($opt_t || $DEBUG >= 2);

$end_time = time();
$time_taken = $end_time - $start_time;
$receive_time_taken = $start_processing_time - $start_time;
$time_taken_units = ($time_taken <= 1) ? "1 second" : "$time_taken seconds";
$datestr = localtime($end_time);
$datestr = "$WORKDIRNAME itrans [$datestr] took $time_taken secs";
$datestr .= " (download $receive_time_taken) " if ($gotfile || $goturl);
$datestr .= " [error]"  if ($errflag);
$datestr .= "\n";
print LOGFILE $datestr;
print MSGFILE $datestr;
print MSGFILE "----------------------------------------------------\n";

# add the form to the done file, with filled in values
$form_tmpl = HTML::Template->new(filename => $FORM_TEMPLATE,
             path => [$ROOTDIR], die_on_bad_params => 0);
$form_html = $form_tmpl->output();
$form_fillin = HTML::FillInForm->new();
$form_html = $form_fillin->fill(scalarref => \$form_html, fobject => $query);

# copy output HTML page - show output or error, as appropriate
$done_tmpl = HTML::Template->new(filename => $DONE_TEMPLATE,
             path => [$ROOTDIR], die_on_bad_params => 0);
&replace_markers($done_tmpl);
$done_tmpl->output(print_to => *WEBDONE_HTML);
close(WEBDONE_HTML);

# complete rest of HTML from this CGI script, will redirect to WEBDONE
&send_working_trailer();

close(MSGFILE);
close(LOGFILE);

exit($errflag);

#-----------------------------------------------------------------------------
# Subroutines

sub preprocess_itrinput {
    # Pre-process the input string, to replace web commands {script-textsize}
    # into ITRANS commands.

    print MSGFILE "--- pre-processing input, for tex/gif/ps/pdf output\n";

    copy($TEX_BASE, "$WORKTEX.itx")
      or myexit("Could not copy ($TEX_BASE) to ($WORKTEX.itx): $!");

    open(OUTPUT, "> $TEX_BASE_INC")  
      or myexit("Could not open \$TEX_BASE_INC file ($TEX_BASE_INC): $!");

    open(INPUT, "< $SAVINPUT")
      or myexit("Could not re-open saved input string file ($SAVINPUT)]: $!");
    print MSGFILE "Re-opened input file $SAVINPUT\n" if ($DEBUG >= 1);

    # begin: pre-pend the default script/size
    if ($informat =~ /^$FORMATISONGS$/) {
	print OUTPUT  "\\itrsize$infontsize\\itrfonteng ";
    } else {
	print OUTPUT  "\\begin{$informat}\n\\itrsize$infontsize\\itrfonteng #$inlanguage ";
    }

    $i = 0;
    while (<INPUT>) {
      my $line = $_;
      $i++;
      my %scriptsizes = &find_tags($line, $i);

      # now replace all script-size commands
      while ( ($key) = each %scriptsizes) { # just $key, don't need $value
	$key =~ /(\w+)-(\w+)/;
	# $replace = "##\\itrsize$2";  # end prev language, set font size
	# $replace .= "\\itrfonteng #$1 "; # set english font, start new script
	$script = $1;
	$size = $2;
	$line =~ s/\{$script-$size\}/##\\itrsize$size\\itrfonteng #$script /g;
	print MSGFILE "$0: line $i found key $key replacing ($script-$size)\n" if ($DEBUG >= 2);
      }

      print OUTPUT $line;
    } # end while <INPUT>

    # end: append the end<script> ITRANS commands
    if ($informat =~ /^$FORMATISONGS$/) {
	print OUTPUT  "\n";
    } else {
	print OUTPUT  "##\\end{$informat}\n";
    }

    close(INPUT);
    close(OUTPUT);

    print MSGFILE "--> input text pre-processed and copied to $TEX_BASE_INC\n"
      if ($DEBUG >= 1);
}

#----------------------------------------
sub preprocess_htminput {
    # Pre-process the input string, to replace web commands {script-textsize}
    # into ITRANS commands.
    my ($str);
    local (*OUTPUT, *INPUT);

    print MSGFILE "--- pre-processing input, for html output\n";

    copy($LAT1_BASE, "$WORKLAT1.itx")
      or myexit("Could not copy ($LAT1_BASE) to ($WORKLAT1.itx): $!");
    copy($UTF8_BASE, "$WORKUTF8.itx")
      or myexit("Could not copy ($UTF8_BASE) to ($WORKUTF8.itx): $!");

    open(OUTPUT, "> $LAT1_BASE_INC")  
      or myexit("Could not open \$LAT1_BASE_INC file ($LAT1_BASE_INC): $!");

    print OUTPUT $HTML_TAG_USER_TEXT, "\n";

    open(INPUT, "< $SAVINPUT")
      or myexit("Could not re-open saved input string file ($SAVINPUT)]: $!");
    print MSGFILE "Re-opened input file $SAVINPUT\n" if ($DEBUG >= 1);

    # begin: pre-pend the default script/size
    if ($informat =~ /^$FORMATISONGS$/) {
	$str = "<FONT $FONTSIZES{$infontsize}>";
    } else {
	$str = "<FONT $FONTSIZES{$infontsize}>#$inlanguage ";
    }
    print OUTPUT $str;

    $i = 0;
    while (<INPUT>) {
      my $line = $_;
      $i++;
      my %scriptsizes = &find_tags($line, $i);

      # now replace all script-size commands
      while ( ($key) = each %scriptsizes) { # just $key, don't need $value
	$key =~ /(\w+)-(\w+)/;
	$script = $1;
	$size = $2;
	$line =~ s!\{$script-$size\}!##</FONT><FONT $FONTSIZES{$size}>#$script !g;
	print MSGFILE "$0: line $i found key $key replacing ($script-$size)\n" if ($DEBUG >= 2);
      }

      print OUTPUT $line;
    } # end while <INPUT>

    # end: append the end<script> ITRANS commands
    $str = ($informat =~ /^$FORMATISONGS$/) ? "</FONT>\n" : "##</FONT>\n";
    print OUTPUT $str;

    print OUTPUT $HTML_TAG_USER_TEXT, "\n";

    close(INPUT);
    close(OUTPUT);

    print MSGFILE "--> input text pre-processed and copied to $LAT1_BASE_INC & $UTF8_BASE_INC\n"
      if ($DEBUG >= 1);

    copy($LAT1_BASE_INC, $UTF8_BASE_INC)
      or myexit("Could not copy ($LAT1_BASE_INC) to ($UTF8_BASE_INC): $!");
}

#----------------------------------------
sub postprocess_htm {
    # Post-process the output HTML file, add <BR> if needed
    my ($marker, $ret);
    local (*OUTPUT, *INPUT);

    print MSGFILE "--- post-processing html output, for verse and isongs\n";
    $ret = 0;

    if ($informat =~ /^$FORMATPROSE$/ || $informat =~ /^$FORMATITX$/) {
      print MSGFILE "    no post-processing needed, is not verse or isongs\n";
      return $ret;
    }

    # append a <BR> to every line between HTML_TAG_USER_TEXT

    foreach $file ("$WORKLAT1.htm", "$WORKUTF8.htm") {

      open(INPUT, "$file")
	or (print MSGFILE "Warning: open html output file ($file): $!"), $ret++, next;

      print MSGFILE "Re-opened output file $file\n" if ($DEBUG >= 1);

      open(OUTPUT, "> $file.tmp")  
	or (print MSGFILE "Warning: open temp output file ($file.tmp): $!"), $ret++, next;

      $marker = 0; # 1 means seen first HTML_TAG_USER_TEXT, 2 means seen both

      while (<INPUT>) {
	$marker++ if (/$HTML_TAG_USER_TEXT$/);
	s/$/<BR>/ if ($marker == 1);
	print OUTPUT;
      }

      close(INPUT);
      close(OUTPUT);

      move("$file.tmp", "$file")
        or (print MSGFILE "Warning: rename temp file ($file.tmp to $file): $!"), $ret++, next;

   }

   print MSGFILE "---------------------\n";

   return $ret;
}

#----------------------------------------
sub show_status {
    my($cmd) = @_;

    $status_count++;
    if ($status_count < 0) { return ("Working . .<BR>\n"); }

    # if cmd contains | char, look for last | and the command after that
    # otherwise, use the first word as the command
    # command can be any combination of \w / \ . characters.
    ($cmd =~ /\|\s+([\\\/\.\w]+)[^|]*$/) || ($cmd =~ /\s*([\\\/\.\w]+)/); # $1 is last command.
    if ($1) {
	$cmd = basename($1) || $1;
        return "&nbsp;&nbsp; ($cmd) . .<BR>\n";
    }

    return "&nbsp;&nbsp; $status_count . .<BR>\n";
}

#----------------------------------------
sub replace_markers {
    my($tmpl) = @_;
    # apply to only WORKING_TEMPLATE
    $tmpl->param(MRKR_DONTREDIRECT => $dontredirect);

    # apply to both WORKING_TEMPLATE and DONE_TEMPLATE
    $tmpl->param(MRKR_WEBDONEHTML => "$WORKHTTP/$WEBDONE_HTML");
    $tmpl->param(MRKR_HOSTNAME => "$hostname ( $hostaddr )");
    $tmpl->param(MRKR_ANYERROR => $errflag);
    $tmpl->param(MRKR_MSGFILE => "$WORKHTTP/$MSGFILENAME");

    # apply to only DONE_TEMPLATE
    $tmpl->param(MRKR_INPUT => "$WORKHTTP/$SAVINPUT") if (-s "$SAVINPUT");
    $tmpl->param(MRKR_PDF => "$WORKHTTP/$WORKTEX.pdf") if (-s "$WORKTEX.pdf");
    $tmpl->param(MRKR_GIF => "$WORKHTTP/$WORKTEX.gif") if (-s "$WORKTEX.gif");
    $tmpl->param(MRKR_HTML_LATIN1 => "$WORKHTTP/$WORKLAT1.htm") if (-s "$WORKLAT1.htm");
    $tmpl->param(MRKR_HTML_UTF8 => "$WORKHTTP/$WORKUTF8.htm") if (-s "$WORKUTF8.htm");
    $tmpl->param(MRKR_POSTSCRIPT => "$WORKHTTP/$WORKTEX.ps") if (-s "$WORKTEX.ps");
    $tmpl->param(MRKR_TIMETAKEN => $time_taken_units);
    $tmpl->param(MRKR_FORM_HTML => $form_html);

    if ($errflag) {
	if ($ALLCMDS[$errcmd] =~ /$ITRANS_E/) {
	  $tmpl->param(MRKR_ITRANSERROR=> 1);
	} elsif ($ALLCMDS[$errcmd] =~ /$PGM2GIF_E/) {
	  $tmpl->param(MRKR_GIFERROR=> 1);
	} else {
	  $tmpl->param(MRKR_SYSERROR=> 1);
	}
    }
}

#----------------------------------------
sub send_working_trailer {
    if (defined $working_trailer_tmpl) {
	&replace_markers($working_trailer_tmpl);
	$working_trailer_tmpl->output(print_to => *STDOUT);
    }
    # this completes the output HTML file displayed while processing job
}
#----------------------------------------
sub sighandler {

    my ($str);

    unlink("$WORKTEX.tex");
    unlink("$WORKTEX.aux", "$WORKTEX.dvi", "$WORKTEX.log", "$WORKTEX.pgm");

    $datestr = localtime(time());
    $str = "$WORKDIRNAME itrans [$datestr] user terminated --\n";
    print LOGFILE $str if (defined fileno LOGFILE);
    print MSGFILE $str if (defined fileno MSGFILE);

    close(WEBDONE_HTML);
    close(LOGFILE);
    close(MSGFILE);

    exit($errflag);
}
#----------------------------------------

sub myexit {
    # if myexit() is called after working template copied to STDOUT, 
    # the automatic redirect in that file needs to be suppressed
    $dontredirect = "1";

    $errflag++;

    my($mesg) = @_;

    print "<html><head><title>Error</title></head><body>\n" unless ($sent_working_header);
    print "<P>\nSession ID# <B>$WORKDIRNAME</B> - Error - $mesg\n<P>\n";
    # displayed in middle of WORKING_TEMPLATE

    $datestr = localtime(time());
    $str = "$WORKDIRNAME [$datestr] $hostname Error - $mesg\n";

    print STDERR $str; # goes to Apache error_log or LOGFILE
    print MSGFILE $str if (defined fileno MSGFILE);

    # complete rest of HTML from this CGI script
    if ($sent_working_header) { &send_working_trailer(""); }
    else { print "</body></html>\n"; }

    close(WEBDONE_HTML);
    close(LOGFILE);
    close(MSGFILE);

    exit($errflag);
}

#----------------------------------------
sub find_tags {
  my ($line, $lineno) = @_;
  my ($script, $size);
  %scriptsizes = ();
  REMATCH:
  {
    if ( $line =~ /\G\{(\w+)-(\w+)\}/gc ) { # found {script-size}

	# look for {word-word} patterns
	$script = $1;
	$size = $2;

	print MSGFILE "$0: line $lineno loop ($1-$2) found\n" if ($DEBUG > 2);
	if (! grep(/^$script$/, @SCRIPTS) || ! defined $FONTSIZES{$size} ) {
	    print MSGFILE "Warning - line $lineno - unknown script-fontsize found: \"$script-$size\", skipping.\n";
	} else {
	    # using hash as a set data structure
	    $scriptsizes{"$script-$size"} = 1;
	}

	redo REMATCH;
    } # end of if() found script-size

    if ( $line =~ /\G.[^{]*/gc ) {
	# first dot consumes one char ( may be a { )
	# then ignore all characters other than another {
	redo REMATCH;
    } # end of if() found script-size
  } # end REMATCH block that contains if()

  %scriptsizes;
}

#----------------------------------------
# Try a few times to make unique temp directory by appending chars if needed

sub mktempdir {
    my $dir = shift; # directory to create tmp dir in
    my $try = shift; # prefix to use for directory name
    my $n = shift; # a numeric suffix to use - is incremented until unique
    my $done = 0;
    my $i;
    my $dirname;
    my $maxtry = 12;

    # if automated call to cgi scipt, using fileurl, don't allow
    # too many attempts, those invocations come in very fast and 
    # require no manual user typing, so restict number of attempts
    # note: there may be other automated calls, not using fileurl, so
    # don't make maxtry number too high - no easy way to figure out if a
    # human is generating the requests, or a machine is!
    # this applies to a 10 minute period

    $maxtry = 20 unless (defined $infileurl && $infileurl);

    umask(0);
    for ($i = 0; $i < $maxtry; $i++) { # try these many times to increment $n
	$dirname = "$try-$n";
	$done = mkdir("$dir/$dirname", 0777);
	last if ($done);
	$n++;
    }
    $done || myexit("Too many jobs being submitted in a short interval?<br> Try again after 10 minutes<br>mktempdir: ($dirname) failed: $!");
    $done ? $dirname : "";
}

#----------------------------------------
sub initialize {

    umask(0);

    select((select(STDOUT), $| = 1)[0]);

    # output HTML page headers, rest of HTML is output by copying WORKING_TEMPLATE
    print CGI::header();

    open(LOGFILE, ">> $LOGFILE")  
      or myexit("Could not open log file ($LOGFILE): $!");
    select((select(LOGFILE), $| = 1)[0]);

    # web users come in as same userid as the script is run on,
    # so to prevent them from reading the log file (small measure of
    # security), make it not-readable by this owner, assuming that
    # the file was created by this script!
    if (-r $LOGFILE && -o $LOGFILE) {
	chmod 0226, $LOGFILE
	  or print LOGFILE "Warning: Could not chmod log file: $!";
    }

    # all STDERR output is caught in local log file, so any perl errors
    # or uncaught STDERR from commands goes here, instead of the global
    # apache log file
    open (STDERR, '>>& LOGFILE')
      or myexit("Could not redirect STDERR to log file ($LOGFILE): $!");
}

#-----------------------------------------------------------------------------
