; Oberon Bootstrap Loader
;
; Written by P.J. Muller using Turbo Assembler
; Based on Gneiss Bootstrap Loader
;
; Ref. P.J. Muller, "A bootstrap mechanism for the Gneiss kernel",
; ITR, University of Stellenbosch, 1995
;
; 1995-03-23 pjm Started
; 1995-06-06 pjm Screen mode setting
; 1995-07-17 pjm Video call (type = 6)
; 1995-08-04 pjm Oberon
; 1995-09-26 pjm Prompt
; 1996-01-23 pjm Fixed boot header
; 1996-03-19 pjm Video replaced with Init, "" removing, deleting, writing
; 1996-03-20 pjm Cleaned up
; 1996-03-22 pjm Changed id back to OBERON
; 1996-04-26 pjm Fixed error in editenv, now v2.07
; 1996-08-15 pjm save regs in init, now v2.08
; 1996-08-30 pjm v2.1
; 1997-01-23 pjm v2.2
; 1997-02-18 pjm APM update, fancy footwork to save space
; 1998-06-01 pjm use int 15 to get extended memory size
; 1998-06-15 pjm sector + 1
; 1998-07-09 pjm save more space, new readsectors
; 1998-10-13 pjm v2.3.0 flag bits
; 1998-10-16 pjm large stack
; 1998-10-20 pjm Boot.Bin relative to Oberon file system, "w" message
; 1998-12-21 pjm v2.3.1
; 1999-01-04 pjm v2.3.2
; 1999-01-28 pjm v2.3.3
; 1999-03-24 pjm v2.3.4b
; 1999-05-10 pjm v2.3.5b
; 1999-05-13 pjm v2.3.6
; 2000-01-06 pjm version removed
; 2000-03-14 pjm space for AosFS ID
; 2000-03-16 pjm max 63.5k read on hard disk
; 2000-12-11 pjm added ee33 to fix config string delete bug

; Boot page layout:
; Boot loader is moved to 2nd last 4k page below memtop as returned by int 12h
; Top 2k (tablesize) of this page is used for the boot table
; Bottom 2k (lsize) of this page is used for the boot loader
; The page above the boot loader is reserved for the boot loader stack.
; The kernel can find the boot page by looking for the boot area (type 3)
; After booting, the boot area (at least the bottom 2k) can be reused
; The config string area in the top 2k of the boot page will still be in use.

ideal		; use turbo assembler syntax
p386n		; enable non-privileged 386 instructions

trace = 0

; size of bootstrap loader in bytes (multiple of sector size & > 512)
lsize = 4*512

tableofs = 800h		; where boot table is placed in loader segment
tablesize = 4*512	; size of table

reserved = (lsize+tablesize+511)/512	; reserved sectors

sectorid = 055aah	; check value for 2nd sector

; define fill string
filler equ "oberon"
fsize = 6		; size of filler

; the fillto macro is used to generate fill bytes up to a specified offset
macro	fillto ofs
if $-entry GT ofs
	display "**Error** No space for filler"
	err
endif
if ofs-($-entry) GT fsize
	db (ofs-($-entry))/fsize dup (filler)
endif
if ofs-($-entry) GT 0
	db ofs-($-entry) dup (0)
endif
endm

; trace macro
if trace
macro	TRAC ch
	push ax
	mov al,ch
	call write
	pop ax
endm
else
macro	TRAC ch
endm
endif

segment code
assume cs:code, ds:code, es:code, ss:code

; This code resides on the first sector of the boot floppy
; It is loaded at 7c0:0 by the BIOS

org	0

entry:	jmp short entry0	; 00 2-byte jmp
	nop			; must have NOP here

; BPB -- Bios Parameter Block (entries marked with * are set by init util)
org	3
hello0	db 'OBERON',0		; 03 OEM id
flag	db 0			; 0A (*) flag
	dw 512			; 0B bytes per sector
	db 2			; 0D sectors per cluster (# 0)
	dw reserved		; 0E (*) reserved sectors
	db (tablesize+511)/512	; 10 (*) table sectors [number of FAT copies]
	dw 224			; 11 root dir size
	dw 2880			; 13 (*) total sectors on disk (#)
	db 0f0h			; 15 media
	dw 9			; 16 sectors per FAT (0 for FAT32)
sectr	dw 18			; 18 (*) sectors per track
heads	dw 2			; 1A (*) number of heads
sector	dd 0			; 1C (*) hidden sectors (boot sector number)
	dd 0			; 20 (*) total sectors on disk, if (#) = 0
drive	db 0			; 24 (*) drive number
sum	db ?                    ; 25 [current head]
oper	db 0			; 26 signature (28h or 29h)
;	dd ?			; 27 serial number
;	db 'Oberon',0,'    '	; 2B volume label (11 bytes)
;	db '        '		; 36 system id (8 bytes)
hellop	dw offset hello0

bufbeg0	db 0			; buffer overwrites code later
bufbeg	db ?			; also see bufend

entry0:	cli
	db 0eah			; jmp 7c0h:offset entry1
	dw offset entry1
	dw 07c0h		; set cs to 7c0h

entry1:	mov ax,cs		; set up segment registers & stack
	mov ds,ax
	mov ss,ax
	mov sp,1000h		; use 7c0h:1000h as temporary stack

	call mark		; 'O'

	int 12h			; get low memory size in k in AX

	and al,not 3		; round down to 4k boundary
	sub ax,8		; subtract 8k
	mov dx,1024/16		; convert k to paragraphs
	mul dx

	mov es,ax		; move first sector to 2nd top page
	xor si,si
	xor di,di
	mov cx,512/2
	cld
	rep movsw

	push es			; jump to loader in top page
	push offset entry2
	retf

entry2:	mov ax,cs		; set up segment registers & stack
	mov ds,ax
	mov es,ax
	mov ss,ax
	mov sp,2000h		; top of top page
	sti

	call mark		; 'b'

; read rest of bootstrap loader
	inc [sector]		; next sector
	mov eax,[sector]
	mov bx,512
	mov cx,lsize/512-1
	add [sector],lsize/512-1
	call readsectors

; check it
	call mark		; 'e'
	cmp [nextid],sectorid
	jne $

	call mark		; 'r'

	call readboottable	; also updates [sector] to end of boot table

; find the kernel's description in the table
	mov si,tableofs

fk0:	mov al,[si]			; get type
	inc al
	jz $				; type = -1, end of table => halt
	cmp al,8			; type = 7, fragmented kernel
	jz short fk2
	add si,[si+4]			; step to next entry in table
	jmp fk0

; read the fragmented kernel
fk2:
	push si
	mov si,[hellop]
	call writestring	; complete the message
	mov si,offset hello1
	call writestring
	pop si

	mov eax,[si+16]			; entry point
	mov [kentry],eax
	mov bl,[si+14]			; -chksum
	mov [sum],bl
	mov di,[si+12]			; number of fragments
	mov eax,[si+8]			; address
	shr eax,4			; convert to segment
	mov es,ax			; es = address
	add si,20			; si = offset of fragment desc

rk3:	push [dword si]			; save start & number
	push [word si+4]

rk0:	mov ax,127			; max sectors that may be read on hd
	cmp [drive],80h
	jae short rk8			; (can delete this code to save space)

	mov ax,es
	mov bx,ax
	or ax,4095			; ax = end of 64k area
	sub ax,bx
	inc ax				; ax = num of paras that may be read
	shr ax,5			; ax = num of sectors "   "  "   "

rk8:	mov cx,[si+4]			; cx = number of sectors
	cmp ax,cx
	jae short rk1			; ok
;	cmp [drive],80h			; hard disk?
;	jae short rk1			; yes, don't worry
	mov cx,ax

rk1:	mov eax,[si]			; eax = start sector
	add eax,[sector]		; relative to file system start
	xor bx,bx
	push cx
	call readsectors
	pop cx

	mov bx,cx
	shl bx,9			; bx = sectors * 512
rk2:	mov al,[es:bx-1]
	add [sum],al
	dec bx
	jnz rk2

	mov ax,es
	mov bx,cx
	shl bx,5			; bx = sectors * 512/16
	add ax,bx
	mov es,ax

	add [si],cx			; next start sector
	adc [word si+2],0

	sub [si+4],cx			; number of sectors left in fragment
	jnz rk0

	pop [word si+4]
	pop [dword si]

	add si,8			; step to next fragment
	dec di				; last fragment?
	jnz rk3

; kernel read

rv2:	jmp nextsector			; note: si=last entry in table

bufend	db 0				; see bufbeg

; write - write character / mark - write next mark
; in: al=char / in: none
; mod: none / mod: ax,bx

proc	mark
	mov bx,[hellop]
	mov al,[bx]
	inc [hellop]
; FALL-THROUGH

write:	push ax
	push bx
	mov ah,14
	mov bx,7
	int 10h
	pop bx
	pop ax
	ret
endp

; readsectors/writesectors
; in: eax=start sector number, es:bx=buffer, cx=number of sectors (<128)
; mod: ax,bx,cx,dx,high(edi)
; note: it is the caller's responsibility that the transfer will not
; exceed a 64k address boundary.  if es:bx is page aligned, and cl <= 8,
; then a 64k boundary will not be exceeded.

writesectors:
	mov [oper],3			; write
	jmp short rs10

proc	readsectors
	mov [oper],2			; read

rs10:	push si				; entry point from writesectors
	push di

rs5:	push eax
	push cx

	mov si,cx
; ***
	movzx edi,[sectr]
	xor edx,edx
	div edi
	mov cl,dl
	inc cl				; s = secnum mod [sectr]+1
	xor dx,dx			; upper part already 0
	movzx edi,[heads]
	div edi
; ***
;	div [sectr]			; { dx:ax div [sectr] < 65536 }
;	mov cl,dl
;	inc cl				; s = secnum mod [sectr]+1
;	xor dx,dx
;	div [heads]
; ***
	mov ch,al			; c = secnum div [sectr] div [heads]
	shl ah,6
	or cl,ah			; top 2 bits of c
	mov dh,dl			; h = secnum div [sectr] mod [heads]
	mov dl,[drive]			; drive number

	cmp dl,80h			; hard disk?
	jae short rs4			; yes, don't worry about sectr

	mov ax,si			; number of sectors
	dec ax
	add al,cl			; note: top 2 bits used on hd
	cmp al,[byte sectr]
	jbe short rs4			; start+num-1 <= sectr

	mov al,cl
	cbw
	neg ax
	add ax,[sectr]
	inc ax
	mov si,ax			; sectr-start+1

rs4:	mov ax,si			; number of sectors
	mov ah,[oper]			; read/write

	mov di,4			; number of tries
rs0:	push ax
	int 13h				; read
	jnc short rs1			; ok

	pushf
	mov ah,0			; reset
	int 13h
	popf

rs1:	pop ax
	jnc short rs2			; ok

	dec di				; does not clear carry
	jnz rs0				; retry loop

	mov si,offset readerr		; error
	call writestring
	jmp $				; endless loop

rs2:	pop cx
	pop eax				; sector number
	sub cx,si
	jz short rs6			; done
	add ax,si
	adc dx,0
	shl si,9			; convert sectors to bytes
	add bx,si
	jmp rs5				; again

rs6:	pop di
	pop si
	ret
endp

; messages for first sector

readerr	db 13,10,'I/O error!',7,0

; place the Aos FS table and boot sector signature

	fillto 1f0h
	dd ?			; fileSystemOfs (in blocks, relative to this)
	dd ?			; fileSystemSize (in sectors)
	db 'IDID'		; id
	db ?			; version
	db ?			; sectorSizeLog2
	db 55h			; bootID0
	db 0aah			; bootID1
	fillto 512

; end of boot sector
; --------------------------------------------------------------------
; start of next sector

nextid	dw sectorid		; used for check

; some procedures moved to next sector for space

readboottable:
	mov eax,[sector]
	mov [tabsec],eax	; store for later
	mov bx,tableofs
	mov cx,tablesize/512
	add [sector],tablesize/512
	jmp readsectors

; writestring
; in: si=zero terminated string
; mod: ax,si

proc	writestring
	cld
ws0:	lodsb
	or al,al
	jz short ws1
	call write
	jmp ws0
ws1:	ret
endp

nextsector:
; check the checksum

	cmp [sum],0
	je short cc0
	mov si,offset badchk
	call writestring
	jmp $
cc0:

; update the boot table

	mov si,tableofs

fe0:	mov al,[si]			; get type
	cmp al,8
	jne short fe2
	lea bx,[si+8]			; store env. begin
	mov [envbeg],bx
	mov bx,[si+4]			; env size
	add bx,si
	mov [envend],bx			; entry after env

fe2:	inc al
	jz short fe1			; type = -1, end of table
	add si,[si+4]			; step to next entry in table
	jmp fe0

fe1:	mov [dword si],3		; boot area (must be first of extra)
	mov [dword si+4],16		; size of this record
	xor eax,eax
	mov ax,cs			; boot area starts at cs:0
	shl eax,4
	mov [si+8],eax			; eax=boot area address
	mov [dword si+12],8192		; two pages
	add si,[si+4]

	mov ah,88h			; get extended memory size
	int 15h
	and eax,0ffffh			; eax = size in k

	mov [dword si],4		; free area
	mov [dword si+4],16		; size of this record
	mov [dword si+8],100000h	; free area (extended memory)
	shl eax,10			; convert k to bytes
	jz short bt1			; no extended memory
	mov [si+12],eax
	add si,[si+4]

; initialise the hard disk parameters

bt1:	mov al,12h			; bios disk setup
	call readcmos

	cld
	push ds				; ES := DS
	pop es

	test al,0f0h
	jz short bt2			; hd0 not present

	mov [dword si],5		; hd params
	mov [dword si+4],28		; size of this record
	mov [dword si+8],0		; hd0

	push si
	push ds
	lea di,[si+12]			; es:di = params (dest)
	mov ds,[si+8]			; ds=0
	lds si,[4*41h]			; ds:si = int vector 41h (source)
	mov cx,8
	rep movsw
	pop ds
	pop si
	add si,[si+4]

bt2:	test al,0fh
	jz short bt3			; hd1 not present

	mov [dword si],5		; hd params
	mov [dword si+4],28		; size of this record
	mov [dword si+8],1		; hd1

	push si
	push ds
	lea di,[si+12]			; es:di = params (dest)
	mov ds,[si+8+2]			; ds=0
	lds si,[4*46h]			; ds:si = int vector 46h (source)
	mov cx,8
	rep movsw
	pop ds
	pop si
	add si,[si+4]

; add boot table end marker

bt3:	mov [dword si],-1
	mov [dword si+4],0
	lea bx,[si+8]
	mov [tabend],bx

	cmp [envbeg],0
	je pm9				; no environment, skip edit & video

	cmp [flag],0			; force environment edit?
	je short ee0			; yes

	push ds
	mov ax,0
	mov ds,ax
	mov al,[417h]			; keyboard flag byte
	pop ds
	test al,[flag]			; normally 31 = 11111b
	jnz short ee0			; if any pressed, edit
	jmp ee1

; write boot table

ee22:	mov si,tableofs
ee24:	cmp [byte si],3			; find boot area (added stuff)
	je short ee23
	add si,[si+4]
	jmp short ee24
ee23:	mov [dword si],-1		; temporary disable

	mov eax,[tabsec]
	mov bx,tableofs
	mov cx,tablesize/512
	call writesectors

	mov [dword si],3		; re-enable

	mov si,offset wrote
	call writestring

	jmp short ee0

; edit keys
	
ee20:	mov al,[bufbeg]
	and al,not 32			; CAP
	cmp al,'C'
	je ee1
	cmp al,0
	je short ee0
	cmp al,'W'
	je short ee22
	mov si,offset help
	call writestring
	jmp short ee21

; edit environment

ee0:	call showenv
ee21:	mov si,offset prompt
	call writestring
	call editline

	mov di,offset bufbeg
ee12:	cmp [byte di],0
	je ee20				; no '=' found
	inc di
	cmp [byte di],'='
	jne short ee12
	mov [byte di],0			; zero terminate name
	inc di				; di=new value, bufbeg=name

; remove "'s

	cmp [byte di],'"'
	jne short ee19
	push di
	lea si,[di+1]
ee18:	lodsb				; assume es=ds & cld
	stosb
	cmp al,0
	jne ee18
	mov [byte di-2],0		; chop off last character
	pop di

; delete existing value, if any
ee19:	TRAC 'B'
	
	cmp [byte di],0			; empty value?
	jne short ee25
	mov di,offset bufbeg0		; will delete old and not add new

ee25:	push di				; assume cld & ds,es=code
ee10:	mov di,[envbeg]
ee3:	mov si,offset bufbeg
	mov bx,di			; save start of name
	mov al,0
ee14:	cmp [si],al			; end of name in buf?
	je short ee13
	cmpsb
	je ee14
	dec di				; undo inc at mismatch
	jmp short ee33			; do not delete if mismatch

ee13:	cmp [di],al			; end of name in env?
	je short ee2			; yes, go delete it

ee33:	mov cx,-1
	repne scasb			; scan for end of name
	repne scasb			; and skip value
	cmp [byte di],0			; end of env?
	je short ee4			; yes, end delete
	jmp ee3				; continue search

ee2:	TRAC 'D'
	inc di				; point to value
	cmp [di],al
	jne ee2				; skip value
	inc di				; di = next name

	mov si,bx
	xchg si,di			; di=delete name, si=next name
ee8:	lodsb				; copy from si to di
	stosb
	cmp al,0			; end of env?
	je short ee5			; yes, go copy rest of table
ee6:	lodsb
	stosb
	cmp al,0
	jne ee6				; copy a full name
ee7:	lodsb
	stosb
	cmp al,0
	jne ee7				; copy a full value
	jmp ee8				; go copy next name,value pair
ee5:	sub di,[envbeg]			; di = size of new env
	test di,3
	jz short ee9			; yes, is multiple of 4
	or di,3
	inc di				; make multiple of 4
ee9:	add di,8			; convert to recsize
	mov si,[envbeg]
	xchg di,[si-4]			; di=old recsize, recsize=di (new)
	mov cx,di
	sub cx,[si-4]   		; cx=old-new=diff in recsize
	jcxz short ee11
	mov si,[envend]
	mov di,si
	sub di,cx
	sub [envend],cx			; set new vars
	sub [tabend],cx
	mov cx,[tabend]			; number of bytes to move
	sub cx,[envend]
	rep movsb			; move the rest of the table down
ee11:
	jmp ee10			; loop, delete again

ee4:	mov di,[envbeg]			; end of delete	
	mov cx,-1
	mov al,0
ee16:	cmp [di],al			; end of env?
	je short ee15
	repne scasb			; skip name
	repne scasb			; skip value
	jmp ee16

ee15:	TRAC 'I'
	mov si,di			; si = adr of last 0
	inc di
	sub di,[envbeg]
	mov bx,di			; bx = env length

	pop di				; di = new value
	push si				; save adr of last 0
	repne scasb			; find end of new value
	sub di,offset bufbeg		; length of new name,value
	mov ax,bx			; ax = envlen
	mov bx,di			; bx = length {bx >= 0}  (was 3)

	add ax,8
	mov si,[envbeg]
	sub ax,[si-4]			; ax = -pad = -(env recsize-8-envlen)
	add ax,bx			; ax = length-pad
	test ax,3
	jz short ee17			; multiple of 4
	or al,3
	inc ax				; make multiple of 4
ee17:					; ax = up4(length-pad)
	mov cx,[tabend]
	sub cx,[envend]			; bytes to move
	mov si,[envend]			; source
	mov di,si
	add di,ax			; destination
	add [envend],ax			; shift the vars
	add [tabend],ax
	mov ax,cx			; adjust for copy overlap
	dec ax
	add si,ax
	add di,ax
	std				; copy backwards
	rep movsb			; make space

	cld
	mov si,offset bufbeg
	pop di				; di = adr of last 0
	mov cx,bx			; cx = bx = length of name,val
	rep movsb
	mov [byte di],0			; add last 0

	mov si,[envbeg]
	mov ax,[envend]
	sub ax,si
	add ax,8
	mov [si-4],ax			; set new recsize

	jmp ee0				; next env string

ee1:	; end of env editing

; ------------------------------------------------------------------

; Do Init

in10:	TRAC 'i'
	mov di,[envbeg]
in3:	mov si,offset init
	mov al,0
in14:	cmp [si],al			; end of name in buf?
	je short in13
	cmpsb
	je in14
in13:	cmp [di],al			; end of name in env?
	je short in2			; yes, found
	mov cx,-1
	repne scasb			; scan for end of name
	repne scasb			; and skip value
	cmp [byte di],0			; end of env?
	je short in4			; yes, not found
	jmp short in3			; continue search at next name

in2:	inc di				; point to value

in5:	cmp [byte di],0			; end of string?
	je short in4

	mov si,offset bufbeg

in1:	mov al,[di]
	cmp al,0
	je short in6
	inc di
	call hexdig
	jnc short in6			; digit out of range (seperator)
	mov ah,al

	mov al,[di]
	cmp al,0
	je short in6
	inc di
	call hexdig
	jnc short in6

	shl ah,4
	or ah,al
	mov [si],ah
	inc si
	cmp si,offset bufend
in8:	je in8
	jmp in1

in6:	mov [byte si],0c3h		; ret
	pusha
	push ds
	push es

	mov ax,cs
	sub ax,1024/16
	mov es,ax			; es:0 = 1k scratch segment

	xor ax,ax
	mov bx,ax
	mov cx,ax
	mov dx,ax
	mov si,ax
	mov di,ax
	mov ds,ax

	cld
	call near offset bufbeg

	pop es
	pop ds
	mov [word kpar0],ax		; save ax-dx for kernel
	mov [word kpar0+2],bx
	mov [word kpar1],cx
	mov [word kpar1+2],dx
	popa
	jmp in5

in4:	; end of init

; ------------------------------------------------------------------

; go to PM

pm9:
	mov ax,5300h			; APM installation check
	xor bx,bx
	int 15h
	jc short apm0			; no APM
	
	mov ax,5303h			; APM PM 32-bit connect
	xor bx,bx
	int 15h
	jc short apm0
	
	shl eax,4
	mov [word apm32+2],ax		; base 0..15
	shr eax,16
	and al,0fh
	mov [apm32+4],al		; base 16..19
	shl ecx,4
	mov [word apm16+2],cx		; base 0..15
	shr ecx,16
	and cl,0fh
	mov [apm16+4],al		; base 16..19
	shl edx,4
	mov [word apmds+2],dx		; base 0..15
	shr edx,16
	and dl,0fh
	mov [apmds+4],dl		; base 16..19
	mov [apmofs],ebx
	
	mov ax,530eh			; APM driver version 1.1
	xor bx,bx
	mov cx,0101h
	int 15h
	jc short apm0
	
	mov [gdtptr],6*8-1		; 6 segments

apm0:
	mov dx,3f2h
	mov al,0ch
	out dx,al			; stop the floppy motor

; enable the A20 address line (AT tech. ref. p. 5-172)

	TRAC '3'
	cli
	call empty8042
	mov al,0d1h
	out 64h,al
	call empty8042
	mov al,0dfh
	out 60h,al
	call empty8042
	
; disable hardware interrupts

	mov al,-1
	out 21h,al			; mask off IRQ0-7
	out 0a1h,al			; mask off IRQ8-15

	mov ax,1020h			; do 16 EOI's
di0:	out 20h,al
	xor cx,cx
	loop $				; small pause
	dec ah
	jnz di0

p386					; now allow protected instructions

; load descriptor tables

	lidt [idtptr]			; load null IDT

        xor eax,eax			; load GDT
	mov ax,cs
	shl eax,4
	add eax,offset gdt
	mov [dword gdtptr+2],eax
        lgdt [pword ptr gdtptr]

; set up stack

	push large 1*8			; kernel code segment selector
	push [dword kentry]		; kernel entry point

	xor edx,edx
	mov dx,ss
	shl edx,4
	mov eax,edx
	add eax,tableofs
	push eax
	push [dword kpar0]
	push [dword kpar1]

	xor eax,eax
	mov ebp,eax			; clear ebp
	mov ax,sp
	add edx,eax			; edx = value for esp

; enter 80286 protected mode

	smsw ax
	or al,1
	lmsw ax
	jmp short pm0			; flush instruction queue

; load segment selectors

pm0:	mov ax,2*8			; kernel data segment selector
	mov ss,ax
	mov ds,ax
	mov es,ax
	mov fs,ax
	mov gs,ax

; jump to kernel

	mov esp,edx
	pop edi				; kpar1
	pop esi				; kpar0
	pop eax				; boot table linear address
	db 066h
	retf				; to kernel

; empty the 8042 buffer
; mod: al

proc	empty8042
e0:	in al,64h
	and al,2
	jnz e0			; endless loop on error
	ret
endp

; hex digit to value
; in: al=digit
; out: al=value, c=ok

proc	hexdig
	sub al,'0'
	cmp al,10
	jc short dig0
	and al,not 32	; CAP
	sub al,7
	cmp al,16
	jc short dig0
	clc
dig0:	ret
endp

; readcmos - read cmos register
; in: al=register number
; out: al=value
; mod: al

proc	readcmos

	or al,80h			; disable NMI
	out 70h,al
	jmp $+2
	jmp $+2
	jmp $+2
	in al,71h
	ret
endp

; read char
; out: al=ASCII char
; mod: ax

proc	readchar
rc0:	mov ah,0
	int 16h
	cmp al,0
	je rc0
	ret
endp

; edit line
; in: none
; out: bufbeg-bufend contains 0-terminated line

proc	editline
	mov di,offset bufbeg

el0:	call readchar
	cmp al,13		; enter
	jne short el1
	mov [byte di],0		; 0-terminate
	ret

el1:	cmp al,8		; backspace
	jne short el2
	cmp di,offset bufbeg
	je el0
	dec di
	mov si,offset backsp
	call writestring
	jmp el0
	
el2:	cmp di,offset bufend
	je el0
	mov [di],al
	inc di
	call write
	jmp el0
endp

; showenv - display environment

proc	showenv
	mov si,[envbeg]
se1:	lodsb
	cmp al,0
	je short se0
	push si
	push ax
	mov si,offset nl
	call writestring
	pop ax
	pop si
se2:
	call write
	lodsb
	cmp al,0
	jne se2
	mov al,'='
	call write
	;mov al,'"'
	;call write
se3:	lodsb
	cmp al,0
	je short se4
	call write
	jmp se3
se4:	;mov al,'"'
	;call write
	jmp se1
se0:	ret
endp

; Global descriptor table (see ref)

align 4
gdt	db 8 dup (0)			; null descriptor
	db 0ffh,0ffh,0,0,0,9ah,0cfh,0	; kernel code segment
	db 0ffh,0ffh,0,0,0,92h,0cfh,0	; kernel data segment
apm32	db 0ffh,0ffh,?,?,?,9ah,0cfh,0	; APM 32-bit code segment
apm16	db 0ffh,0ffh,?,?,?,9ah,000h,0	; APM 16-bit code segment
apmds	db 0ffh,0ffh,?,?,?,92h,0cfh,0	; APM 32-bit data segment
apmofs	dd ?				; APM entry point (directly after GDT)
gdtptr	dw 3*8-1,?,?			; 48-bit pointer to gdt
idtptr	dp 0				; 48-bit null pointer

; more messages

envbeg	dw 0	; actual first char of env
envend	dw ?	; next entry after env
tabend	dw ?	; next entry after table
tabsec	dd ?	; sector where table starts
kentry	dd ?	; kernel entry point
kpar0	dd ?	; kernel parameter 1
kpar1	dd ?	; kernel parameter 2

init	db 'Init',0
hello1	db ' loading...'
	db 0			; (can remove this to save a byte)
nl	db 13,10,0		; together with previous line

backsp	db 8,' ',8,0
prompt	db 13,10,10,'OBL>',0

wrote	db 13,10,'Done!',10,0

help	db 13,10,'Help:'
	db 13,10,'  name=val - set config string'
	db 13,10,'  name=    - delete config string'
	db 13,10,'  c        - continue booting'
	db 13,10,'  w        - write config back'
	db 0
badchk	db 'Boot.Bin checksum bad',7
	db 0

; fill rest of loader, and check if size ok
	fillto lsize

ends
end entry
