import { Es2019Array } from "./Es2019Array";
import { Optional, ValueEmbedder } from "./Monad";
import { Lang } from "./Lang";
var objAssign = Lang.objAssign;
import { append, appendIf, assign, assignIf, resolve, shallowMerge } from "./AssocArray";
/**
 * specialized value embedder
 * for our Configuration
 */
class ConfigEntry extends ValueEmbedder {
    constructor(rootElem, key, arrPos) {
        super(rootElem, key);
        this.arrPos = arrPos !== null && arrPos !== void 0 ? arrPos : -1;
    }
    get value() {
        if (this.key == "" && this.arrPos >= 0) {
            return this._value[this.arrPos];
        }
        else if (this.key && this.arrPos >= 0) {
            return this._value[this.key][this.arrPos];
        }
        return this._value[this.key];
    }
    set value(val) {
        if (this.key == "" && this.arrPos >= 0) {
            this._value[this.arrPos] = val;
            return;
        }
        else if (this.key && this.arrPos >= 0) {
            this._value[this.key][this.arrPos] = val;
            return;
        }
        this._value[this.key] = val;
    }
}
/*default value for absent*/
ConfigEntry.absent = ConfigEntry.fromNullable(null);
export const CONFIG_VALUE = "__END_POINT__";
export const CONFIG_ANY = "__ANY_POINT__";
/**
 * Config, basically an optional wrapper for a json structure
 * (not Side - effect free, since we can alter the internal config state
 * without generating a new config), not sure if we should make it side - effect free
 * since this would swallow a lot of performance and ram
 */
export class Config extends Optional {
    constructor(root, configDef) {
        super(root);
        this.configDef = configDef;
    }
    /**
     * shallow copy getter, copies only the first level, references the deeper nodes
     * in a shared manner
     */
    get shallowCopy() {
        return this.shallowCopy$();
    }
    shallowCopy$() {
        let ret = new Config({});
        ret.shallowMerge(this.value);
        return ret;
    }
    /**
     * deep copy, copies all config nodes
     */
    get deepCopy() {
        return this.deepCopy$();
    }
    deepCopy$() {
        return new Config(objAssign({}, this.value));
    }
    /**
     * creates a config from an initial value or null
     * @param value
     */
    static fromNullable(value) {
        return new Config(value);
    }
    /**
     * simple merge for the root configs
     */
    shallowMerge(other, overwrite = true, withAppend = false) {
        //shallow merge must be mutable so we have to remap
        let newThis = shallowMerge(overwrite, withAppend, this.value, other.value);
        if (Array.isArray(this._value)) {
            this._value.length = 0;
            this._value.push(...newThis);
        }
        else {
            Object.getOwnPropertyNames(this._value).forEach(key => delete this._value[key]);
            Object.getOwnPropertyNames(newThis).forEach(key => this._value[key] = newThis[key]);
        }
    }
    /**
     * assigns a single value as array, or appends it
     * to an existing value mapping a single value to array
     *
     *
     * usage myConfig.append("foobaz").value = "newValue"
     *       myConfig.append("foobaz").value = "newValue2"
     *
     * resulting in myConfig.foobaz == ["newValue, newValue2"]
     *
     * @param {string[]} accessPath
     */
    append(...accessPath) {
        return append(this._value, ...accessPath);
    }
    /**
     * appends to an existing entry (or extends into an array and appends)
     * if the condition is met
     * @param {boolean} condition
     * @param {string[]} accessPath
     */
    appendIf(condition, ...accessPath) {
        return appendIf(condition, this._value, ...accessPath);
    }
    /**
     * assigns a new value on the given access path
     * @param accessPath
     */
    assign(...accessPath) {
        return assign(this.value, ...accessPath);
    }
    /**
     * assign a value if the condition is set to true, otherwise skip it
     *
     * @param condition the condition, the access accessPath into the config
     * @param accessPath
     */
    assignIf(condition, ...accessPath) {
        return assignIf(condition, this._value, ...accessPath);
    }
    /**
     * get if the access path is present (get is reserved as getter with a default, on the current path)
     * TODO will be renamed to something more meaningful and deprecated, the name is ambiguous
     * @param accessPath the access path
     */
    getIf(...accessPath) {
        this.assertAccessPath(...accessPath);
        return this.getClass().fromNullable(resolve(this.value, ...accessPath));
    }
    /**
     * gets the current node and if none is present returns a config with a default value
     * @param defaultVal
     */
    get(defaultVal) {
        return this.getClass().fromNullable(super.get(defaultVal).value);
    }
    //empties the current config entry
    delete(key) {
        if (key in this.value) {
            delete this.value[key];
        }
        return this;
    }
    /**
     * converts the entire config into a json object
     */
    toJson() {
        return JSON.stringify(this.value);
    }
    getClass() {
        return Config;
    }
    setVal(val) {
        this._value = val;
    }
    /**
     * asserts the access path for a semi typed access
     * @param accessPath
     * @private
     */
    assertAccessPath(...accessPath) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j;
        accessPath = this.preprocessKeys(...accessPath);
        if (!this.configDef) {
            //untyped
            return;
        }
        const ERR_ACCESS_PATH = "Access Path to config invalid";
        let currAccessPos = Optional.fromNullable(Object.keys(this.configDef).map(key => {
            let ret = {};
            ret[key] = this.configDef[key];
            return ret;
        }));
        for (let cnt = 0; cnt < accessPath.length; cnt++) {
            let currKey = this.keyVal(accessPath[cnt]);
            let arrPos = this.arrayIndex(accessPath[cnt]);
            //key index
            if (this.isArray(arrPos)) {
                if (currKey != "") {
                    currAccessPos = Array.isArray(currAccessPos.value) ?
                        Optional.fromNullable((_b = (_a = new Es2019Array(...currAccessPos.value)
                            .find(item => {
                            var _a;
                            return !!((_a = item === null || item === void 0 ? void 0 : item[currKey]) !== null && _a !== void 0 ? _a : false);
                        })) === null || _a === void 0 ? void 0 : _a[currKey]) === null || _b === void 0 ? void 0 : _b[arrPos]) :
                        Optional.fromNullable((_e = (_d = (_c = currAccessPos.value) === null || _c === void 0 ? void 0 : _c[currKey]) === null || _d === void 0 ? void 0 : _d[arrPos]) !== null && _e !== void 0 ? _e : null);
                }
                else {
                    currAccessPos = (Array.isArray(currAccessPos.value)) ?
                        Optional.fromNullable((_f = currAccessPos.value) === null || _f === void 0 ? void 0 : _f[arrPos]) : Optional.absent;
                }
                //we noe store either the current array or the filtered look ahead to go further
            }
            else {
                //we now have an array and go further with a singular key
                currAccessPos = (Array.isArray(currAccessPos.value)) ? Optional.fromNullable((_g = new Es2019Array(...currAccessPos.value)
                    .find(item => {
                    var _a;
                    return !!((_a = item === null || item === void 0 ? void 0 : item[currKey]) !== null && _a !== void 0 ? _a : false);
                })) === null || _g === void 0 ? void 0 : _g[currKey]) :
                    Optional.fromNullable((_j = (_h = currAccessPos.value) === null || _h === void 0 ? void 0 : _h[currKey]) !== null && _j !== void 0 ? _j : null);
            }
            if (!currAccessPos.isPresent()) {
                throw Error(ERR_ACCESS_PATH);
            }
            if (currAccessPos.value == CONFIG_ANY) {
                return;
            }
        }
    }
    isNoArray(arrPos) {
        return arrPos == -1;
    }
    isArray(arrPos) {
        return !this.isNoArray(arrPos);
    }
}
//# sourceMappingURL=Config.js.map