/*
  ==============================================================================

   This file is part of the JUCE library - "Jules' Utility Class Extensions"
   Copyright 2004-6 by Raw Material Software ltd.

  ------------------------------------------------------------------------------

   JUCE can be redistributed and/or modified under the terms of the
   GNU General Public License, as published by the Free Software Foundation;
   either version 2 of the License, or (at your option) any later version.

   JUCE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with JUCE; if not, visit www.gnu.org/licenses or write to the
   Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
   Boston, MA 02111-1307 USA

  ------------------------------------------------------------------------------

   If you'd like to release a closed-source product which uses JUCE, commercial
   licenses are also available: visit www.rawmaterialsoftware.com/juce for
   more information.

  ==============================================================================
*/

#include "../../../src/juce_core/basics/juce_StandardHeader.h"
#include <sys/stat.h>
#include <sys/dir.h>
#include <sys/param.h>
#include <sys/mount.h>
#include <unistd.h>
#include <fnmatch.h>
#include <utime.h>
#include <pwd.h>
#include <fcntl.h>
#include <Carbon/Carbon.h>

BEGIN_JUCE_NAMESPACE


#include "../../../src/juce_core/io/files/juce_FileInputStream.h"
#include "../../../src/juce_core/io/files/juce_FileOutputStream.h"
#include "../../../src/juce_core/io/network/juce_URL.h"
#include "../../../src/juce_core/basics/juce_SystemStats.h"
#include "../../../src/juce_core/misc/juce_PlatformUtilities.h"
#include "../../../src/juce_core/io/files/juce_NamedPipe.h"


//==============================================================================
const tchar File::separator = T('/');
const tchar* File::separatorString = T("/");

static String executableFilename;


//==============================================================================
void PlatformUtilities::copyToStr255 (Str255& d, const String& s)
{
    unsigned char* t = (unsigned char*) d;
    t[0] = jmin (254, s.length());
    s.copyToBuffer ((char*) t + 1, 254);
}

void PlatformUtilities::copyToStr63 (Str63& d, const String& s)
{
    unsigned char* t = (unsigned char*) d;
    t[0] = jmin (62, s.length());
    s.copyToBuffer ((char*) t + 1, 62);
}

const String PlatformUtilities::cfStringToJuceString (CFStringRef cfString)
{
    String result;

    if (cfString != 0)
    {
#if JUCE_STRINGS_ARE_UNICODE
        CFRange range = { 0, CFStringGetLength (cfString) };
        UniChar* const u = (UniChar*) juce_malloc (sizeof (UniChar) * (range.length + 1));

        CFStringGetCharacters (cfString, range, u);
        u[range.length] = 0;

        result = convertUTF16ToString (u);

        juce_free (u);
#else
        const int len = CFStringGetLength (cfString);
        char* buffer = (char*) juce_malloc (len + 1);
        CFStringGetCString (cfString, buffer, len + 1, CFStringGetSystemEncoding());
        result = buffer;
        juce_free (buffer);
#endif
    }

    return result;
}

CFStringRef PlatformUtilities::juceStringToCFString (const String& s)
{
#if JUCE_STRINGS_ARE_UNICODE
    const int len = s.length();
    const juce_wchar* t = (const juce_wchar*) s;

    UniChar* temp = (UniChar*) juce_malloc (sizeof (UniChar) * len + 4);

    for (int i = 0; i <= len; ++i)
        temp[i] = t[i];

    CFStringRef result = CFStringCreateWithCharacters (kCFAllocatorDefault, temp, len);
    juce_free (temp);

    return result;

#else
    return CFStringCreateWithCString (kCFAllocatorDefault,
                                      (const char*) s,
                                      CFStringGetSystemEncoding());
#endif
}

const String PlatformUtilities::convertUTF16ToString (const UniChar* utf16)
{
    String s;

    while (*utf16 != 0)
        s += (juce_wchar) *utf16++;

    return s;
}

//==============================================================================
static bool juce_stat (const String& fileName, struct stat& info)
{
    return fileName.isNotEmpty()
            && (stat (fileName.toUTF8(), &info) == 0);
}

//==============================================================================
bool juce_isDirectory (const String& fileName)
{
    if (fileName.isEmpty())
        return true;

    struct stat info;

    return juce_stat (fileName, info)
            && ((info.st_mode & S_IFDIR) != 0);
}

bool juce_fileExists (const String& fileName, const bool dontCountDirectories)
{
    if (fileName.isEmpty())
        return ! dontCountDirectories;

    const char* const fileNameUTF8 = fileName.toUTF8();
    bool exists = access (fileNameUTF8, F_OK) == 0;

    if (exists && dontCountDirectories)
    {
        struct stat info;
        const int res = stat (fileNameUTF8, &info);

        if (res == 0 && (info.st_mode & S_IFDIR) != 0)
            exists = false;
    }

    return exists;
}

int64 juce_getFileSize (const String& fileName)
{
    struct stat info;

    if (juce_stat (fileName, info))
        return info.st_size;

    return 0;
}

const unsigned int macTimeToUnixTimeDiff = 0x7c25be90;

void juce_getFileTimes (const String& fileName,
                        int64& modificationTime,
                        int64& accessTime,
                        int64& creationTime)
{
    modificationTime = 0;
    accessTime = 0;
    creationTime = 0;

    FSSpec file;
    if (PlatformUtilities::makeFSSpecFromPath (&file, fileName))
    {
        CInfoPBRec info;
        zerostruct (info);
        info.hFileInfo.ioNamePtr = file.name;
        info.hFileInfo.ioVRefNum = file.vRefNum;
        info.hFileInfo.ioDirID = file.parID;

        if (PBGetCatInfoSync (&info) == noErr)
            creationTime = (int64)(info.hFileInfo.ioFlCrDat - macTimeToUnixTimeDiff) * 1000;
    }

    struct stat info;
    if (juce_stat (fileName, info))
    {
        modificationTime = (int64) info.st_mtime * 1000;
        accessTime = (int64) info.st_atime * 1000;
    }
}

bool juce_setFileTimes (const String& fileName,
                        int64 modificationTime,
                        int64 accessTime,
                        int64 creationTime)
{
    bool ok = true;

    if (modificationTime != 0 || accessTime != 0)
    {
        ok = false;

        struct stat info;
        if (juce_stat (fileName, info))
        {
            struct utimbuf times;

            if (accessTime != 0)
                times.actime = (time_t)(accessTime / 1000);
            else
                times.actime = info.st_atime;

            if (modificationTime != 0)
                times.modtime = (time_t)(modificationTime / 1000);
            else
                times.modtime = info.st_mtime;

            ok = (utime (fileName.toUTF8(), &times) == 0);
        }
    }

    if (ok && (creationTime != 0))
    {
        ok = false;

        FSSpec file;
        if (PlatformUtilities::makeFSSpecFromPath (&file, fileName))
        {
            CInfoPBRec info;
            zerostruct (info);
            info.hFileInfo.ioNamePtr = file.name;
            info.hFileInfo.ioVRefNum = file.vRefNum;
            info.hFileInfo.ioDirID = file.parID;

            if (PBGetCatInfoSync (&info) == noErr)
            {
                info.hFileInfo.ioDirID = file.parID;
                info.hFileInfo.ioFlCrDat = (unsigned long)(macTimeToUnixTimeDiff + creationTime / 1000);
                ok = (PBSetCatInfoSync (&info) == noErr);
            }
        }
    }

    return ok;
}

bool juce_canWriteToFile (const String& fileName)
{
    return access (fileName.toUTF8(), W_OK) == 0;
}

bool juce_setFileReadOnly (const String& fileName, bool isReadOnly)
{
    const char* const fileNameUTF8 = fileName.toUTF8();

    struct stat info;
    const int res = stat (fileNameUTF8, &info);

    bool ok = false;

    if (res == 0)
    {
        info.st_mode &= 0777;   // Just permissions

        if (isReadOnly)
            info.st_mode &= ~(S_IWUSR | S_IWGRP | S_IWOTH);
        else
            // Give everybody write permission?
            info.st_mode |= S_IWUSR | S_IWGRP | S_IWOTH;

        ok = chmod (fileNameUTF8, info.st_mode) == 0;
    }

    return ok;
}

bool juce_deleteFile (const String& fileName)
{
    const char* const fileNameUTF8 = fileName.toUTF8();

    if (juce_isDirectory (fileName))
        return rmdir (fileNameUTF8) == 0;
    else
        return remove (fileNameUTF8) == 0;
}

bool juce_copyFile (const String& src, const String& dst)
{
    const File destFile (dst);

    if (! destFile.create())
        return false;

    FSRef srcRef, dstRef;

    if (! (PlatformUtilities::makeFSRefFromPath (&srcRef, src)
            && PlatformUtilities::makeFSRefFromPath (&dstRef, dst)))
    {
        return false;
    }

    int okForks = 0;

    CatPositionRec iter;
    iter.initialize = 0;
    HFSUniStr255 forkName;

    // can't just copy the data because this is a bloody Mac, so we need to copy each
    // fork separately...
    while (FSIterateForks (&srcRef, &iter, &forkName, 0, 0) == noErr)
    {
        SInt16 srcForkNum = 0, dstForkNum = 0;
        OSErr err = FSOpenFork (&srcRef, forkName.length, forkName.unicode, fsRdPerm, &srcForkNum);

        if (err == noErr)
        {
            err = FSOpenFork (&dstRef, forkName.length, forkName.unicode, fsRdWrPerm, &dstForkNum);

            if (err == noErr)
            {
                MemoryBlock buf (32768);
                SInt64 pos = 0;

                for (;;)
                {
                    ByteCount bytesRead = 0;
                    err = FSReadFork (srcForkNum, fsFromStart, pos, buf.getSize(), (char*) buf, &bytesRead);

                    if (bytesRead > 0)
                    {
                        err = FSWriteFork (dstForkNum, fsFromStart, pos, bytesRead, (const char*) buf, &bytesRead);
                        pos += bytesRead;
                    }

                    if (err != noErr)
                    {
                        if (err == eofErr)
                            ++okForks;

                        break;
                    }
                }

                FSFlushFork (dstForkNum);
            }
        }
    }

    return okForks > 0; // some files seem to be ok even if not all their forks get copied..
}

bool juce_moveFile (const String& source, const String& dest)
{
    if (rename (source.toUTF8(), dest.toUTF8()) == 0)
        return true;

    if (juce_canWriteToFile (source)
         && juce_copyFile (source, dest))
    {
        if (juce_deleteFile (source))
            return true;

        juce_deleteFile (dest);
    }

    return false;
}

void juce_createDirectory (const String& fileName)
{
    mkdir (fileName.toUTF8(), 0777);
}

void* juce_fileOpen (const String& fileName, bool forWriting)
{
    const char* const fileNameUTF8 = fileName.toUTF8();
    const char* mode = "rb";

    if (forWriting)
    {
        if (juce_fileExists (fileName, false))
        {
            FILE* const f = fopen (fileNameUTF8, "r+b");

            if (f != 0)
                fseek (f, 0, SEEK_END);

            return (void*) f;
        }
        else
        {
            mode = "w+b";
        }
    }

    return (void*) fopen (fileNameUTF8, mode);
}

void juce_fileClose (void* handle)
{
    if (handle != 0)
        fclose ((FILE*) handle);
}

int juce_fileRead (void* handle, void* buffer, int size)
{
    if (handle != 0)
        return fread (buffer, 1, size, (FILE*) handle);

    return 0;
}

int juce_fileWrite (void* handle, const void* buffer, int size)
{
    if (handle != 0)
        return fwrite (buffer, 1, size, (FILE*) handle);

    return 0;
}

int64 juce_fileSetPosition (void* handle, int64 pos)
{
    if (handle != 0 && fseek ((FILE*) handle, pos, SEEK_SET) == 0)
        return pos;

    return -1;
}

int64 juce_fileGetPosition (void* handle)
{
    if (handle != 0)
        return ftell ((FILE*) handle);
    else
        return -1;
}

void juce_fileFlush (void* handle)
{
    if (handle != 0)
        fflush ((FILE*) handle);
}

const StringArray juce_getFileSystemRoots()
{
    StringArray s;
    s.add (T("/"));
    return s;
}

const String juce_getVolumeLabel (const String& filenameOnVolume, int& volumeSerialNumber)
{
    volumeSerialNumber = 0;
    return String::empty;
}

// if this file doesn't exist, find a parent of it that does..
static bool doStatFS (const File* file, struct statfs& result)
{
    File f (*file);

    for (int i = 5; --i >= 0;)
    {
        if (f.exists())
            break;

        f = f.getParentDirectory();
    }

    return statfs (f.getFullPathName().toUTF8(), &result) == 0;
}

int64 File::getBytesFreeOnVolume() const throw()
{
    int64 free_space = 0;

    struct statfs buf;
    if (doStatFS (this, buf))
        // Note: this returns space available to non-super user
        free_space = (int64) buf.f_bsize * (int64) buf.f_bavail;

    return free_space;
}

bool File::isOnCDRomDrive() const throw()
{
    // can't find a list of values for this, but non-zero ones seem to be cds, nfs, etc
    struct statfs buf;
    if (doStatFS (this, buf))
    {
        const String fsType (buf.f_fstypename);
        return fsType.equalsIgnoreCase (T("cd9660"))
                || fsType.equalsIgnoreCase (T("cdfs"));
    }

    // Assume not if this fails for some reason
    return false;
}

bool File::isOnHardDisk() const throw()
{
    // can't find a list of values for this, but non-zero ones seem to be cds, nfs, etc
    // 0x17 seems to be ms-dos
    struct statfs buf;
    if (doStatFS (this, buf))
    {
        const String fsType (buf.f_fstypename);
        return ! (fsType.equalsIgnoreCase (T("nfs"))
                   || fsType.equalsIgnoreCase (T("smbfs"))
                   || fsType.equalsIgnoreCase (T("cd9660"))
                   || fsType.equalsIgnoreCase (T("cdfs")));
    }

    // Assume so if this fails for some reason
    return true;
}

const String SystemStats::getTempPath()
{
    File tmp (T("~/Library/Caches/")
                + File (executableFilename).getFileNameWithoutExtension());

    tmp.createDirectory();
    return tmp.getFullPathName();
}

const String SystemStats::getUserHomeDirectory()
{
    const char* homeDir = getenv ("HOME");

    if (homeDir == 0)
    {
        struct passwd* const pw = getpwuid (getuid());
        if (pw != 0)
            homeDir = pw->pw_dir;
    }

    String home (homeDir);
    if (home.endsWithChar (T('/')))
        home = home.dropLastCharacters (1);

    return home;
}

const String SystemStats::getUserDocumentsDirectory()
{
    return getUserHomeDirectory() + T("/Documents");
}

const String SystemStats::getUserApplicationDataDirectory()
{
    return getUserHomeDirectory() + T("/Library");
}

const String SystemStats::getCurrentExecutableFileName()
{
    return executableFilename;
}

void juce_setCurrentExecutableFileName (const String& filename)
{
    if (! filename.startsWithChar (T('/')))
        executableFilename = File::getCurrentWorkingDirectory().getChildFile (filename).getFullPathName();
    else
        executableFilename = filename;
}

//==============================================================================
const File File::getCurrentWorkingDirectory() throw()
{
    char buf [2048];
    getcwd (buf, sizeof(buf));
    return File (buf);
}

bool File::setAsCurrentWorkingDirectory() const throw()
{
    return chdir (getFullPathName().toUTF8()) == 0;
}

//==============================================================================
struct FindFileStruct
{
    String parentDir, wildCard;
    DIR* dir;

    bool getNextMatch (String& result, bool& isDir)
    {
        const char* const wildCardUTF8 = wildCard.toUTF8();

        for (;;)
        {
            struct dirent* const de = readdir (dir);

            if (de == 0)
                break;

            if (fnmatch (wildCardUTF8, de->d_name, 0) == 0)
            {
                result = String::fromUTF8 ((const uint8*) de->d_name);
                isDir = juce_isDirectory (parentDir + result);

                return true;
            }
        }

        return false;
    }
};

// returns 0 on failure
void* juce_findFileStart (const String& directory,
                          const String& wildCard,
                          String& firstResult,
                          bool& isDir)
{
    DIR* const d = opendir (directory.toUTF8());

    if (d != 0)
    {
        FindFileStruct* const ff = new FindFileStruct();
        ff->parentDir = directory;

        if (!ff->parentDir.endsWithChar (File::separator))
            ff->parentDir += File::separator;

        ff->wildCard = wildCard;
        ff->dir = d;

        if (ff->getNextMatch (firstResult, isDir))
        {
            return ff;
        }
        else
        {
            firstResult = String::empty;
            isDir = false;
            closedir (d);
            delete ff;
        }
    }

    return 0;
}

bool juce_findFileNext (void* handle, String& resultFile, bool& isDir)
{
    FindFileStruct* const ff = (FindFileStruct*) handle;

    if (ff != 0)
        return ff->getNextMatch (resultFile, isDir);

    return false;
}

void juce_findFileClose (void* handle)
{
    FindFileStruct* const ff = (FindFileStruct*)handle;

    if (ff != 0)
    {
        closedir (ff->dir);
        delete ff;
    }
}

//==============================================================================
bool juce_launchExecutable (const String& pathAndArguments)
{
    char* const argv[4] = { "/bin/sh", "-c", (char*) (const char*) pathAndArguments, 0 };

    const int cpid = fork();

    if (cpid == 0)
    {
        // Child process
        if (execve (argv[0], argv, 0) < 0)
            exit (0);
    }
    else
    {
        if (cpid < 0)
            return false;
    }

    return true;
}

bool juce_launchFile (const String& fileName,
                      const String& parameters)
{
    bool ok = false;

    if (fileName.startsWithIgnoreCase (T("http:")))
    {
        CFStringRef urlString = PlatformUtilities::juceStringToCFString (fileName);

        if (urlString != 0)
        {
            CFURLRef url = CFURLCreateWithString (kCFAllocatorDefault,
                                                  urlString, 0);
            CFRelease (urlString);

            if (url != 0)
            {
                ok = (LSOpenCFURLRef (url, 0) == noErr);
                CFRelease (url);
            }
        }
    }
    else
    {
        FSRef ref;
        if (PlatformUtilities::makeFSRefFromPath (&ref, fileName))
        {
            if (juce_isDirectory (fileName) && parameters.isNotEmpty())
            {
                // if we're launching a bundled app with a document..
                StringArray docs;
                docs.addTokens (parameters, true);
                FSRef* docRefs = new FSRef [docs.size()];

                for (int i = 0; i < docs.size(); ++i)
                    PlatformUtilities::makeFSRefFromPath (docRefs + i, docs[i]);

                LSLaunchFSRefSpec ors;
                ors.appRef = &ref;
                ors.numDocs = docs.size();
                ors.itemRefs = docRefs;
                ors.passThruParams = 0;
                ors.launchFlags = kLSLaunchDefaults;
                ors.asyncRefCon = 0;

                FSRef actual;
                ok = (LSOpenFromRefSpec (&ors, &actual) == noErr);

                delete docRefs;
            }
            else
            {
                if (parameters.isNotEmpty())
                    ok = juce_launchExecutable (fileName + T(" ") + parameters);
                else
                    ok = (LSOpenFSRef (&ref, 0) == noErr);
            }
        }
    }

    return ok;
}

//==============================================================================
bool PlatformUtilities::makeFSSpecFromPath (FSSpec* fs, const String& path)
{
    FSRef ref;

    return makeFSRefFromPath (&ref, path)
            && FSGetCatalogInfo (&ref, kFSCatInfoNone, 0, 0, fs, 0) == noErr;
}

bool PlatformUtilities::makeFSRefFromPath (FSRef* destFSRef, const String& path)
{
    return FSPathMakeRef ((const UInt8*) path.toUTF8(), destFSRef, 0) == noErr;
}

const String PlatformUtilities::makePathFromFSRef (FSRef* file)
{
    uint8 path [2048];
    zeromem (path, sizeof (path));

    String result;

    if (FSRefMakePath (file, (UInt8*) path, sizeof (path) - 1) == noErr)
        result = String::fromUTF8 (path);

    return result;
}

//==============================================================================
OSType PlatformUtilities::getTypeOfFile (const String& filename)
{
    FSSpec fs;
    if (makeFSSpecFromPath (&fs, filename))
    {
        FInfo info;
        if (FSpGetFInfo (&fs, &info) == noErr)
            return info.fdType;
    }

    return 0;
}

bool PlatformUtilities::isBundle (const String& filename)
{
    FSSpec fs;
    if (makeFSSpecFromPath (&fs, filename))
    {
        FInfo info;
        if (FSpGetFInfo (&fs, &info) == noErr)
            return (info.fdFlags & kHasBundle) != 0;
    }

    return false;
}


END_JUCE_NAMESPACE
