/*
  ==============================================================================

   This file is part of the JUCE library - "Jules' Utility Class Extensions"
   Copyright 2004-6 by Raw Material Software ltd.

  ------------------------------------------------------------------------------

   JUCE can be redistributed and/or modified under the terms of the
   GNU General Public License, as published by the Free Software Foundation;
   either version 2 of the License, or (at your option) any later version.

   JUCE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with JUCE; if not, visit www.gnu.org/licenses or write to the
   Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
   Boston, MA 02111-1307 USA

  ------------------------------------------------------------------------------

   If you'd like to release a closed-source product which uses JUCE, commercial
   licenses are also available: visit www.rawmaterialsoftware.com/juce for
   more information.

  ==============================================================================
*/

#ifndef __JUCE_ATOMIC_JUCEHEADER__
#define __JUCE_ATOMIC_JUCEHEADER__

// Atomic increment/decrement operations..

//==============================================================================
#if JUCE_MAC && ! DOXYGEN
  #if (MAC_OS_X_VERSION_MAX_ALLOWED >= MAC_OS_X_VERSION_10_4)
    #include <libkern/OSAtomic.h>

    forcedinline void atomicIncrement (int& variable) throw()
    {
        OSAtomicIncrement32 ((int32_t*) &variable);
    }

    forcedinline int atomicIncrementAndReturn (int& variable) throw()
    {
        return OSAtomicIncrement32 ((int32_t*) &variable);
    }

    forcedinline void atomicDecrement (int& variable) throw()
    {
        OSAtomicDecrement32 ((int32_t*) &variable);
    }

    forcedinline int atomicDecrementAndReturn (int& variable) throw()
    {
       return OSAtomicDecrement32 ((int32_t*) &variable);
    }

  #else
    forcedinline void atomicIncrement (int& variable) throw()
    {
       OTAtomicAdd32 (1, (SInt32*) &variable);
    }

    forcedinline int atomicIncrementAndReturn (int& variable) throw()
    {
       return OTAtomicAdd32 (1, (SInt32*) &variable);
    }

    forcedinline void atomicDecrement (int& variable) throw()
    {
       OTAtomicAdd32 (-1, (SInt32*) &variable);
    }

    forcedinline int atomicDecrementAndReturn (int& variable) throw()
    {
       return OTAtomicAdd32 (-1, (SInt32*) &variable);
    }
  #endif
#else
#ifdef __GNUC__

    /** Increments an integer in a thread-safe way. */
    forcedinline void atomicIncrement (int& variable) throw()
    {
        __asm__ __volatile (
            "lock incl (%%ecx)"
            :
            : "c" (&variable));
    }

    /** Increments an integer in a thread-safe way and returns the incremented value. */
    forcedinline int atomicIncrementAndReturn (int& variable) throw()
    {
        int result;

        __asm__ __volatile (
            "lock xaddl %%eax, (%%ebx)  \n\
             incl %%eax"
            : "=a" (result)
            : "b" (&variable), "a" (1)
            : "cc", "memory");

        return result;
    }

    /** Decrememts an integer in a thread-safe way. */
    forcedinline void atomicDecrement (int& variable) throw()
    {
        __asm__ __volatile (
            "lock decl (%%ecx)"
            :
            : "c" (&variable));
    }

    /** Decrememts an integer in a thread-safe way and returns the incremented value. */
    forcedinline int atomicDecrementAndReturn (int& variable) throw()
    {
        int result;

        __asm__ __volatile (
            "lock xaddl %%eax, (%%ebx)  \n\
             decl %%eax"
            : "=a" (result)
            : "b" (&variable), "a" (-1)
            : "cc", "memory");

        return result;
    }

#else

    /** Increments an integer in a thread-safe way. */
    inline_assembly void __fastcall atomicIncrement (int& variable) throw()
    {
        __asm {
            mov ecx, dword ptr [variable]
            lock inc dword ptr [ecx]
        }
    }

    /** Increments an integer in a thread-safe way and returns the incremented value. */
    inline_assembly int __fastcall atomicIncrementAndReturn (int& variable) throw()
    {
        int result;

        __asm {
            mov ecx, dword ptr [variable]
            mov eax, 1
            lock xadd dword ptr [ecx], eax
            inc eax
            mov result, eax
        }

        return result;
    }

    /** Decrememts an integer in a thread-safe way. */
    inline_assembly void __fastcall atomicDecrement (int& variable) throw()
    {
        __asm {
            mov ecx, dword ptr [variable]
            lock dec dword ptr [ecx]
        }
    }

    /** Decrememts an integer in a thread-safe way and returns the incremented value. */
    inline_assembly int __fastcall atomicDecrementAndReturn (int& variable) throw()
    {
        int result;

        __asm {
            mov ecx, dword ptr [variable]
            mov eax, -1
            lock xadd dword ptr [ecx], eax
            dec eax
            mov result, eax
        }

        return result;
    }
#endif
#endif

#endif   // __JUCE_ATOMIC_JUCEHEADER__
