/*
  ==============================================================================

   This file is part of the JUCE library - "Jules' Utility Class Extensions"
   Copyright 2004-6 by Raw Material Software ltd.

  ------------------------------------------------------------------------------

   JUCE can be redistributed and/or modified under the terms of the
   GNU General Public License, as published by the Free Software Foundation;
   either version 2 of the License, or (at your option) any later version.

   JUCE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with JUCE; if not, visit www.gnu.org/licenses or write to the
   Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
   Boston, MA 02111-1307 USA

  ------------------------------------------------------------------------------

   If you'd like to release a closed-source product which uses JUCE, commercial
   licenses are also available: visit www.rawmaterialsoftware.com/juce for
   more information.

  ==============================================================================
*/

#ifndef __JUCE_PLATFORMDEFS_JUCEHEADER__
#define __JUCE_PLATFORMDEFS_JUCEHEADER__


//==============================================================================
// set up platform definitions..
#ifdef _WIN32
  #define       JUCE_WIN32 1
  #define       JUCE_WINDOWS 1
#else
  #ifdef LINUX
    #define     JUCE_LINUX 1
  #else
    #define     JUCE_MAC 1
  #endif
#endif

//==============================================================================
#ifdef JUCE_WINDOWS
  #ifdef _MSC_VER
    #ifdef _WIN64
      #define JUCE_64BIT 1
    #else
      #define JUCE_32BIT 1
    #endif
  #endif

  #ifdef _DEBUG
    #define JUCE_DEBUG 1
  #endif

  /** If defined, this indicates that the processor is little-endian. */
  #define JUCE_LITTLE_ENDIAN 1

  #define JUCE_INTEL 1
#endif

//==============================================================================
#ifdef JUCE_MAC

  #include <CoreServices/CoreServices.h>
  #include <wchar.h>

  #ifndef NDEBUG
    #define JUCE_DEBUG 1
  #endif

  #ifdef __LITTLE_ENDIAN__
    #define JUCE_LITTLE_ENDIAN 1
  #else
    #define JUCE_BIG_ENDIAN 1
  #endif

  #if defined (__ppc__) || defined (__ppc64__)
    #define JUCE_PPC 1
  #else
    #define JUCE_INTEL 1
  #endif

  #ifdef __LP64__
    #define JUCE_64BIT 1
  #else
    #define JUCE_32BIT 1
  #endif
#endif

//==============================================================================
#ifdef JUCE_LINUX

  #ifdef _DEBUG
    #define JUCE_DEBUG 1
  #endif

  #include <wchar.h>

  // Allow override for big-endian Linux platforms
  #ifndef JUCE_BIG_ENDIAN
    #define JUCE_LITTLE_ENDIAN 1
  #endif

  #if defined (__LP64__) || defined (_LP64)
    #define JUCE_64BIT 1
  #else
    #define JUCE_32BIT 1
  #endif

  #define JUCE_INTEL 1
#endif

//==============================================================================
#if defined (__GNUC__) || defined (__MWERKS__)
    /** A platform-independent 64-bit integer type. */
    typedef long long           int64;
    /** A platform-independent 64-bit unsigned integer type. */
    typedef unsigned long long  uint64;
    /** A platform-independent unicode character type. */
    typedef wchar_t             juce_wchar;

    /** A platform-independent way of forcing an inline function.

      Use the syntax: @code
        forcedinline void myfunction (int x)
      @endcode
    */
    #ifndef JUCE_DEBUG
      #define forcedinline  inline __attribute__((always_inline))
    #else
      #define forcedinline  inline
    #endif

    /** A platform-independent way of stopping the compiler inlining a function.

      Use the syntax: @code
        juce_noinline void myfunction (int x)
      @endcode
    */
    #define juce_noinline __attribute__((noinline))

#else
    //==============================================================================
    /** A platform-independent 64-bit integer type. */
    typedef __int64             int64;
    /** A platform-independent 64-bit unsigned integer type. */
    typedef unsigned __int64    uint64;

    /** A platform-independent unicode character type. */
    typedef wchar_t             juce_wchar;

    /** A platform-independent way of forcing an inline function.

      Use the syntax: @code
        forcedinline void myfunction (int x)
      @endcode
    */
    #ifdef _MSC_VER
      #define forcedinline  __forceinline
    #else
      #define forcedinline  inline
    #endif

    /** A platform-independent way of stopping the compiler inlining a function.

      Use the syntax: @code
        juce_noinline void myfunction (int x)
      @endcode
    */
    #define juce_noinline

#endif

#if JUCE_64BIT
   typedef int64                   pointer_sized_int;
   typedef uint64                  pointer_sized_uint;
#else
  #if _MSC_VER >= 1300
     typedef _W64 int              pointer_sized_int;
     typedef _W64 unsigned int     pointer_sized_uint;
  #else
     typedef int                   pointer_sized_int;
     typedef unsigned int          pointer_sized_uint;
  #endif
#endif


// borland can't handle inline functions containing asm code, so define a
// special type of inline modifier for this kind of function. Sigh..
#ifdef __BORLANDC__
  #define inline_assembly
#else
  #define inline_assembly forcedinline
#endif

//==============================================================================
typedef signed char     int8;
typedef unsigned char   uint8;
typedef signed short    int16;
typedef unsigned short  uint16;
typedef signed int      int32;
typedef unsigned int    uint32;


//==============================================================================
// Debugging macros
//

#ifdef JUCE_DEBUG
  //==============================================================================
  // If debugging is enabled..

  /** Writes a string to the standard error stream.

    This is only compiled in a debug build.

    @see Logger::outputDebugString
  */
  #define DBG(dbgtext)              Logger::outputDebugString (dbgtext);

  /** Printf's a string to the standard error stream.

    This is only compiled in a debug build.

    @see Logger::outputDebugString
  */
  #define DBG_PRINTF(dbgprintf)     Logger::outputDebugPrintf dbgprintf;

  // Assertions..
  #if defined (_MSC_VER) || DOXYGEN
    /** This will always cause an assertion failure.

        This is only compiled in a debug build.

        @see jassert()
    */
    #ifdef __BORLANDC__
      extern void juce_StopInDebugger();
      #define jassertfalse          { juce_StopInDebugger(); }
    #else
      #define jassertfalse          { __asm int 3 }
    #endif
  #elif defined (JUCE_MAC)
    #define jassertfalse            { Debugger(); }
  #elif defined (__GNUC__) || defined (JUCE_LINUX)
    #define jassertfalse            { asm("int $3"); }
  #endif

  //==============================================================================
  /** Platform-independent assertion macro.

    This gets optimised out when not being built with debugging turned on.

    Be careful not to call any functions within its arguments that are vital to
    the behaviour of the program, because these won't get called in the release
    build.

    @see jassertfalse
  */
  #define jassert(a)                { if (! (a)) jassertfalse }

#else
  //==============================================================================
  // If debugging is disabled, disable all the assertions and debugging stuff..

  #define DBG(dbgtext)
  #define DBG_PRINTF(dbgprintf)

  #define jassert(a)        {}
  #define jassertfalse      {}

#endif

//==============================================================================
#ifndef DOXYGEN
 template <bool b> struct JuceStaticAssert;
 template <> struct JuceStaticAssert <true> { static void dummy() {} };
#endif

/** A compile-time assertion macro.

    If the expression parameter is false, the macro will cause a compile error.
*/
#define static_jassert(expression)      JuceStaticAssert<expression>::dummy();


//==============================================================================
#if JUCE_CATCH_UNHANDLED_EXCEPTIONS

  #define JUCE_TRY try

  /** Used in try-catch blocks, this macro will send exceptions to the JUCEApplication
      object so they can be logged by the application if it wants to.
  */
  #define JUCE_CATCH_EXCEPTION \
    catch (const std::exception& e)  \
    { \
        JUCEApplication::sendUnhandledException (&e, __FILE__, __LINE__); \
    } \
    catch (...) \
    { \
        JUCEApplication::sendUnhandledException (0, __FILE__, __LINE__); \
    }

  #define JUCE_CATCH_ALL            catch (...) {}
  #define JUCE_CATCH_ALL_ASSERT     catch (...) { jassertfalse }

#else

  #define JUCE_TRY
  #define JUCE_CATCH_EXCEPTION
  #define JUCE_CATCH_ALL
  #define JUCE_CATCH_ALL_ASSERT

#endif


#endif   // __JUCE_PLATFORMDEFS_JUCEHEADER__
