/*
  ==============================================================================

   This file is part of the JUCE library - "Jules' Utility Class Extensions"
   Copyright 2004-6 by Raw Material Software ltd.

  ------------------------------------------------------------------------------

   JUCE can be redistributed and/or modified under the terms of the
   GNU General Public License, as published by the Free Software Foundation;
   either version 2 of the License, or (at your option) any later version.

   JUCE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with JUCE; if not, visit www.gnu.org/licenses or write to the
   Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
   Boston, MA 02111-1307 USA

  ------------------------------------------------------------------------------

   If you'd like to release a closed-source product which uses JUCE, commercial
   licenses are also available: visit www.rawmaterialsoftware.com/juce for
   more information.

  ==============================================================================
*/

#ifndef __JUCE_SYSTEMSTATS_JUCEHEADER__
#define __JUCE_SYSTEMSTATS_JUCEHEADER__


//==============================================================================
/**
    Contains methods for finding out about the current hardware and OS configuration.
*/
class JUCE_API  SystemStats
{
public:
    //==============================================================================
    /** Returns the current version of JUCE,

        (as if you didn't already know at compile-time.)

        See also the JUCE_VERSION, JUCE_MAJOR_VERSION and JUCE_MINOR_VERSION macros.
    */
    static const String getJUCEVersion();

    //==============================================================================
    /** Returns the type of operating system we're running on.

        @returns one of the strings enumerated below..
    */
    static const String getOSType();

    // Possible types of operating system, as returned by getOSType()

    static const tchar* MacOSX;     /**< @see getOSType() */
    static const tchar* Linux;      /**< @see getOSType() */
    static const tchar* WinXP;      /**< @see getOSType() */
    static const tchar* Win2000;    /**< @see getOSType() */
    static const tchar* Win95;      /**< @see getOSType() */
    static const tchar* Win98;      /**< @see getOSType() */
    static const tchar* WinNT351;   /**< @see getOSType() */
    static const tchar* WinNT40;    /**< @see getOSType() */
    static const tchar* UnknownOS;  /**< @see getOSType() */


    //==============================================================================
    // CPU and memory information..

    /** Returns the approximate CPU speed.

        @returns    the speed in megahertz, e.g. 1500, 2500, 32000 (depending on
                    what year you're reading this...)
    */
    static int getCpuSpeedInMegaherz();

    /** Returns a string to indicate the CPU vendor.

        Might not be known on some systems.
    */
    static const String getCpuVendor();

    /** Checks whether Intel MMX instructions are available. */
    static bool hasMMX();

    /** Checks whether Intel SSE instructions are available. */
    static bool hasSSE();

    /** Checks whether Intel SSE2 instructions are available. */
    static bool hasSSE2();

    /** Checks whether AMD 3DNOW instructions are available. */
    static bool has3DNow();

    /** True if the chip has hyperthreading.

        Probably only uber-geeks will care less about this.
    */
    static bool hasHyperThreading();

    /** Checks whether there are multiple processors in the box.

        @see getNumLogicalCpus
    */
    static int getNumPhysicalCpus();

    /** Counts the number of logical processors.

        May give a different result to getNumPhysicalCpus()...
    */
    static int getNumLogicalCpus();

    /** Returns a bitmask for the physical processors.

        This mask shows which of the logical processors are actually physical.

        @see Thread::setAffinityMask
    */
    static uint32 getPhysicalAffinityMask();

    /** Returns a clock-cycle tick counter, if available.

        If the machine can do it, this will return a tick-count
        where each tick is one cpu clock cycle - used for profiling
        code.

        @returns    the tick count, or zero if not available.
    */
    static int64 getClockCycleCounter();

    //==============================================================================
    /** Finds out how much RAM is in the machine.

        @returns    the approximate number of megabytes of memory, or zero if
                    something goes wrong when finding out.
    */
    static int getMemorySizeInMegabytes();

    /** Returns the system page-size.

        This is only used by programmers with beards.
    */
    static int getPageSize();

    //==============================================================================
    // Standard directory path info..

    /** Returns the path of the system's default temporary file directory */
    static const String getTempPath();

    /** Returns the path of the current user's home directory.

        On Mac/Linux, this is equivalent to "~".
    */
    static const String getUserHomeDirectory();

    /** Returns the path of the user's "documents" folder.

        e.g. "My Documents" on Windows, "~" on linux, "~/Documents" on the mac
    */
    static const String getUserDocumentsDirectory();

    /** Returns the path in which an application should store its data.

        e.g. on Windows, this might be "\Documents and Settings\username\Application Data".
             on the Mac, it might be "~/Library"

        The app should use a subdirectory inside this folder to keep its things tidy.
    */
    static const String getUserApplicationDataDirectory();

    /** Returns the full path of the currently executing application's file. */
    static const String getCurrentExecutableFileName();

    //==============================================================================
    /** Returns a list of MAC addresses found on this machine.

        @param  addresses   an array into which the MAC addresses should be copied
        @param  maxNum      the number of elements in this array
        @returns            the number of MAC addresses that were found
    */
    static int getMACAddresses (int64* addresses, int maxNum);


    //==============================================================================
    // not-for-public-use platform-specific method gets called at startup to initialise things.
    static void initialiseStats();
};


#endif   // __JUCE_SYSTEMSTATS_JUCEHEADER__
