/*
  ==============================================================================

   This file is part of the JUCE library - "Jules' Utility Class Extensions"
   Copyright 2004-6 by Raw Material Software ltd.

  ------------------------------------------------------------------------------

   JUCE can be redistributed and/or modified under the terms of the
   GNU General Public License, as published by the Free Software Foundation;
   either version 2 of the License, or (at your option) any later version.

   JUCE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with JUCE; if not, visit www.gnu.org/licenses or write to the
   Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
   Boston, MA 02111-1307 USA

  ------------------------------------------------------------------------------

   If you'd like to release a closed-source product which uses JUCE, commercial
   licenses are also available: visit www.rawmaterialsoftware.com/juce for
   more information.

  ==============================================================================
*/

#ifndef __JUCE_UUID_JUCEHEADER__
#define __JUCE_UUID_JUCEHEADER__

#include "../text/juce_String.h"
#include "../io/juce_InputStream.h"
#include "../io/juce_OutputStream.h"

static const int UUID_BYTES = 16;


//==============================================================================
/**
    A universally unique 128-bit identifier.
*/
class JUCE_API  Uuid
{
public:
    //==============================================================================
    /** Creates a new unique ID. */
    Uuid();

    /** Creates an ID from an encoded string version.

        @see toString
    */
    Uuid (const String& uuid);

    /** Creates an ID from an encoded 16-byte array.

        @see copyToBytes
    */
    Uuid (const unsigned char* bytes);

    /** Creates a copy of another Uuid. */
    Uuid (const Uuid& other);

    // xxx is this needed as well as operator= ??
    void fromString (const String& uuid);

    bool operator== (const Uuid& other) const;

    // Assignment operators
    Uuid& operator= (const Uuid& other);
    Uuid& operator= (const unsigned char* bytes);
    Uuid& operator= (const String& uuid);

    /** Returns a pointer to the internal binary representation of the ID.

        @see copyToBytes, numBytes
    */
    operator const unsigned char*() const;

    /** Returns the ID as a string.

        @returns a string of the form: xxxxxxxx-xxxx-xxxx-xxxx-xxxxxxxxxxxx
    */
    const String toString() const;

    /** Copies the ID to a 16-byte array.

        @see numBytes
    */
    void copyToBytes (unsigned char* bytes) const;

    /** Returns the number of bytes needed for the binary representation of the ID.

        @see copyToBytes
    */
    int numBytes() const  { return UUID_BYTES; }

    /** Returns true if the ID is invalid. */
    bool isNull() const;

    //==============================================================================
    /** Reads the ID from a stream.

        @see toStream
    */
    bool fromStream (InputStream& in);

    /** Writes the ID to a stream.

        @see fromStream
    */
    void toStream (OutputStream& out) const;

    //==============================================================================
    /** Returns a string version of a new ID

        The string returned is in the form: xxxxxxxx-xxxx-xxxx-xxxx-xxxxxxxxxxxx.

        @see Uuid (const String&)
    */
    static const String generateUuid();


private:
    //==============================================================================
    unsigned char uuidBytes [UUID_BYTES];
};


#endif   // __JUCE_UUID_JUCEHEADER__
