/*
  ==============================================================================

   This file is part of the JUCE library - "Jules' Utility Class Extensions"
   Copyright 2004-6 by Raw Material Software ltd.

  ------------------------------------------------------------------------------

   JUCE can be redistributed and/or modified under the terms of the
   GNU General Public License, as published by the Free Software Foundation;
   either version 2 of the License, or (at your option) any later version.

   JUCE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with JUCE; if not, visit www.gnu.org/licenses or write to the
   Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
   Boston, MA 02111-1307 USA

  ------------------------------------------------------------------------------

   If you'd like to release a closed-source product which uses JUCE, commercial
   licenses are also available: visit www.rawmaterialsoftware.com/juce for
   more information.

  ==============================================================================
*/

#ifndef __JUCE_TEXTFUNCTIONS_JUCEHEADER__
#define __JUCE_TEXTFUNCTIONS_JUCEHEADER__


//==============================================================================
/* The String class can either use 16/32 bit unicode characters, or 8-bit ascii
   for its internal representation. To use unicode, define the JUCE_STRINGS_ARE_UNICODE
   macro in juce_Config.h

   Be sure to use "tchar" for characters rather than "char", and always wrap string
   literals in the T("abcd") macro, so that it all works nicely either way round.
*/

#ifndef JUCE_WIN32
  // need some *nix conversions first..
  #define stricmp       strcasecmp
  #define strnicmp      strncasecmp
  #define _timeb        timeb
  #define _ftime        ftime
  #define _vsnprintf    vsnprintf

  #ifdef JUCE_LINUX
    #define _atoi64     atoll
  #endif
#endif

//==============================================================================
/* It's better to use these string functions rather than the standard ones.
   (And even better just to use the String class).
*/
#if JUCE_STRINGS_ARE_UNICODE

  #define JUCE_T(stringLiteral)     (L##stringLiteral)
  typedef juce_wchar                tchar;
  #define juce_tcharToWideChar(c)   (c)

  #define juce_strncpy              wcsncpy
  #define juce_strlen               wcslen
  #define juce_strcat               wcscat
  #define juce_strcmp               wcscmp
  #define juce_strncmp              wcsncmp
  #ifdef JUCE_WIN32
    #define juce_stricmp            _wcsicmp
    #define juce_strnicmp           _wcsnicmp
    #define juce_strupr             _wcsupr
    #define juce_strlwr             _wcslwr
    #define juce_sprintf            _snwprintf
  #else
    BEGIN_JUCE_NAMESPACE
    int juce_stricmp (const tchar* s1, const tchar* s2);
    int juce_strnicmp (const tchar* s1, const tchar* s2, int numChars);
    void juce_strupr (tchar* t);
    void juce_strlwr (tchar* t);
    END_JUCE_NAMESPACE
    #define juce_sprintf            swprintf
  #endif
  #define juce_strstr               wcsstr
  #define juce_tolower              towlower
  #define juce_toupper              towupper
  #define juce_strftime             wcsftime

#else

  #define JUCE_T(stringLiteral)     (stringLiteral)
  typedef char                      tchar;
  #define juce_tcharToWideChar(c)   ((juce_wchar) (unsigned char) (c))

  #define juce_strncpy              strncpy
  #define juce_strlen               strlen
  #define juce_strcat               strcat
  #define juce_strcmp               strcmp
  #define juce_stricmp              stricmp
  #define juce_strncmp              strncmp
  #define juce_strnicmp             strnicmp
  #define juce_strstr               strstr
  #ifdef JUCE_WIN32
    #define juce_sprintf            _snprintf
  #else
    #define juce_sprintf            snprintf
  #endif
  #define juce_strupr               strupr
  #define juce_strlwr               strlwr
  #define juce_tolower              tolower
  #define juce_toupper              toupper
  #define juce_strftime             strftime

#endif

/** The 'T' macro allows a literal string to be compiled as either ascii or unicode.

    If you write your string literals in the form T("xyz"), this will either be compiled
    as "xyz" for ascii builds, or L"xyz" for unicode builds, depending on whether the
    JUCE_STRINGS_ARE_UNICODE macro has been set in juce_Config.h

    Because the 'T' symbol is occasionally used inside 3rd-party library headers which you
    may need to include after juce.h, you can use the following method to work around this:

    @code
    #include "juce.h"

    #undef T
    #include "problematic_header_file.h"
    #define T(s) JUCE_T(s)
    @endcode
*/
#define T(stringLiteral)            JUCE_T(stringLiteral)


//==============================================================================
/** Used in the same way as the T(text) macro, this will attempt to translate a
    string into a localised version using the LocalisedStrings class.

    @see LocalisedStrings
*/
#define TRANS(stringLiteral)     \
    LocalisedStrings::translateWithCurrentMappings (stringLiteral)



#endif   // __JUCE_TEXTFUNCTIONS_JUCEHEADER__
