/**************************************************************************\
 *
 *  This file is part of the Klimt library.
 *  Copyright (C) 2003 by IMS, Vienna University of Technology.
 *  All rights reserved.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  ("GPL") version 2 as published by the Free Software Foundation.
 *  See the file LICENSE.GPL at the root directory of this source
 *  distribution for additional information about the GNU GPL.
 *  For the full GPL license see
 *  <URL:http://www.gnu.org/copyleft/gpl.html>
 *
 *  For using Klimt with software that can not be combined with the
 *  GNU GPL, and for taking advantage of the additional benefits of
 *  our support services, please contact IMS about acquiring a
 *  Klimt Professional Edition License.
 *
 *  Contact: <mailto:klimt@studierstube.org>
 *  See <URL:http://www.studierstube.org/klimt>
 *  for more information.
 *
 *  Vienna University of Technology
 *  Institute for Software Technology and Interactive Systems
 *  Interactive Media Systems Group
 *  Favoritenstrasse 9-11/188/2
 *  A-1040 Vienna, Austria
 *  <URL:http://www.ims.tuwien.ac.at>.
 *
 **************************************************************************
 *
 * $Header: /cvsroot/klimt/klimt/klimt/src/Base/klArray.h,v 1.2 2004/01/13 11:20:09 drgoldie Exp $
 *
\**************************************************************************/


#ifndef __KLARRAY_HEADERFILE__
#define __KLARRAY_HEADERFILE__


#include "klBase.h"


// Stores ogl like vertex arrays (color, normal, ...)
//
// Notice: klVec3 & klVec4 must be defined (typedef'd before)
//         getVec3 & getVec4 are quite unefficient. if you need to
//         fetch multiple elements use klArrayFetcher (see below).
//
struct klArray
{
	klArray()  {  clientData = 0;  enabled = false;  }

	const GLvoid	*clientData;
	bool			enabled;
	GLint			size;
	GLenum			type;
	GLsizei			stride;
	bool			clientDataChanged;

	void set(GLint nSize, GLenum nType, GLsizei nStride, const GLvoid *nClientData)
	{
		size = nSize;
		type = nType;
		stride = nStride;
		clientData = nClientData;
		clientDataChanged = true;
	}

	inline void getVec3(klVec3& nVec3, int nIdx) const;
	inline void getVec4(klVec4& nVec4, int nIdx) const;
};


// This class lets us fetch data from an array.
// in contrast to klArray::getVecX() this class determines
// as much data as possible at creation time and is therefore
// a lot faster (when fetching multiple elements from the
// same array)
class klArrayFetcher
{
public:
	klArrayFetcher(const klArray& nArray);

	inline void getVec3(klVec3& nVec3, int nIdx) const;
	inline void getVec4(klVec4& nVec4, int nIdx) const;

protected:
	typedef void (klArrayFetcher::* GETVEC3FUNC)(klVec3& nVec3, int nIdx) const;
	typedef void (klArrayFetcher::* GETVEC4FUNC)(klVec4& nVec4, int nIdx) const;

	inline void getVec3UnsignedByte(klVec3& nVec3, int nIdx) const;
	inline void getVec3Fixed(klVec3& nVec3, int nIdx) const;
	inline void getVec3Short(klVec3& nVec3, int nIdx) const;
	inline void getVec3Int(klVec3& nVec3, int nIdx) const;
	inline void getVec3Float(klVec3& nVec3, int nIdx) const;
	inline void getVec3Double(klVec3& nVec3, int nIdx) const;

	inline void getVec4UnsignedByte(klVec4& nVec4, int nIdx) const;
	inline void getVec4Fixed(klVec4& nVec4, int nIdx) const;
	inline void getVec4Short(klVec4& nVec4, int nIdx) const;
	inline void getVec4Int(klVec4& nVec4, int nIdx) const;
	inline void getVec4Float(klVec4& nVec4, int nIdx) const;
	inline void getVec4Double(klVec4& nVec4, int nIdx) const;

	int elementStride;
	const klArray* array;
	GETVEC3FUNC getVec3P;
	GETVEC4FUNC getVec4P;
};


inline void
klArray::getVec3(klVec3& nVec3, int nIdx) const
{
	int elementStride = size*getByteSizeFromDataType(type) + stride;
	int offset = nIdx*elementStride;
	const unsigned char* ptr = reinterpret_cast<const unsigned char*>(clientData)+offset;

	switch(type)
	{
	case GL_FIXED:
		nVec3.setFixed(0, *reinterpret_cast<const int*>(ptr));
		nVec3.setFixed(1, *reinterpret_cast<const int*>(ptr+4));
		nVec3.setFixed(2, size>2 ? *reinterpret_cast<const int*>(ptr+8) : 0);
		break;

	case GL_SHORT:
		nVec3.setInt(0, *reinterpret_cast<const short*>(ptr));
		nVec3.setInt(1, *reinterpret_cast<const short*>(ptr+2));
		nVec3.setInt(2, size>2 ? *reinterpret_cast<const short*>(ptr+4) : 0);
		break;

	case GL_INT:
		nVec3.setInt(0, *reinterpret_cast<const int*>(ptr));
		nVec3.setInt(1, *reinterpret_cast<const int*>(ptr+4));
		nVec3.setInt(2, size>2 ? *reinterpret_cast<const int*>(ptr+8) : 0);
		break;

	case GL_FLOAT:
		nVec3.setFloat(0, *reinterpret_cast<const float*>(ptr));
		nVec3.setFloat(1, *reinterpret_cast<const float*>(ptr+4));
		nVec3.setFloat(2, size>2 ? *reinterpret_cast<const float*>(ptr+8) : 0.0f);
		break;

	case GL_DOUBLE:
		nVec3.setDouble(0, *reinterpret_cast<const double*>(ptr));
		nVec3.setDouble(1, *reinterpret_cast<const double*>(ptr+8));
		nVec3.setDouble(2, size>2 ? *reinterpret_cast<const double*>(ptr+16) : 0.0);
		break;
	}
}


inline void
klArray::getVec4(klVec4& nVec4, int nIdx) const
{
	int elementStride = size*getByteSizeFromDataType(type) + stride;
	int offset = nIdx*elementStride;
	const unsigned char* ptr = reinterpret_cast<const unsigned char*>(clientData)+offset;

	switch(type)
	{
	case GL_FIXED:
		nVec4.setFixed(0, *reinterpret_cast<const int*>(ptr));
		nVec4.setFixed(1, *reinterpret_cast<const int*>(ptr+4));
		nVec4.setFixed(2, size>2 ? *reinterpret_cast<const int*>(ptr+8) : 0);
		nVec4.setFixed(3, size>3 ? *reinterpret_cast<const int*>(ptr+12) : 1<<16);
		break;

	case GL_SHORT:
		nVec4.setInt(0, *reinterpret_cast<const short*>(ptr));
		nVec4.setInt(1, *reinterpret_cast<const short*>(ptr+2));
		nVec4.setInt(2, size>2 ? *reinterpret_cast<const short*>(ptr+4) : 0);
		nVec4.setInt(3, size>3 ? *reinterpret_cast<const short*>(ptr+6) : 1);
		break;

	case GL_INT:
		nVec4.setInt(0, *reinterpret_cast<const int*>(ptr));
		nVec4.setInt(1, *reinterpret_cast<const int*>(ptr+4));
		nVec4.setInt(2, size>2 ? *reinterpret_cast<const int*>(ptr+8) : 0);
		nVec4.setInt(3, size>3 ? *reinterpret_cast<const int*>(ptr+12) : 1);
		break;

	case GL_FLOAT:
		nVec4.setFloat(0, *reinterpret_cast<const float*>(ptr));
		nVec4.setFloat(1, *reinterpret_cast<const float*>(ptr+4));
		nVec4.setFloat(2, size>2 ? *reinterpret_cast<const float*>(ptr+8) : 0.0f);
		nVec4.setFloat(3, size>3 ? *reinterpret_cast<const float*>(ptr+12) : 1.0f);
		break;

	case GL_DOUBLE:
		nVec4.setDouble(0, *reinterpret_cast<const double*>(ptr));
		nVec4.setDouble(1, *reinterpret_cast<const double*>(ptr+8));
		nVec4.setDouble(2, size>2 ? *reinterpret_cast<const double*>(ptr+16) : 0.0);
		nVec4.setDouble(3, size>3 ? *reinterpret_cast<const double*>(ptr+24) : 1.0);
		break;
	}
}


klArrayFetcher::klArrayFetcher(const klArray& nArray)
{
	array = &nArray;
	elementStride = nArray.size*getByteSizeFromDataType(nArray.type) + nArray.stride;

	switch(array->type)
	{
	case GL_UNSIGNED_BYTE:
		getVec3P = &klArrayFetcher::getVec3UnsignedByte;
		getVec4P = &klArrayFetcher::getVec4UnsignedByte;
		break;

	case GL_FIXED:
		getVec3P = &klArrayFetcher::getVec3Fixed;
		getVec4P = &klArrayFetcher::getVec4Fixed;
		break;

	case GL_SHORT:
		getVec3P = &klArrayFetcher::getVec3Short;
		getVec4P = &klArrayFetcher::getVec4Short;
		break;

	case GL_INT:
		getVec3P = &klArrayFetcher::getVec3Int;
		getVec4P = &klArrayFetcher::getVec4Int;
		break;

	case GL_FLOAT:
		getVec3P = &klArrayFetcher::getVec3Float;
		getVec4P = &klArrayFetcher::getVec4Float;
		break;

	case GL_DOUBLE:
		getVec3P = &klArrayFetcher::getVec3Double;
		getVec4P = &klArrayFetcher::getVec4Double;
		break;
	}
}


inline void
klArrayFetcher::getVec3(klVec3& nVec3, int nIdx) const
{
	(this->*getVec3P)(nVec3, nIdx);
}


inline void
klArrayFetcher::getVec4(klVec4& nVec4, int nIdx) const
{
	(this->*getVec4P)(nVec4, nIdx);
}


inline void
klArrayFetcher::getVec3UnsignedByte(klVec3& nVec3, int nIdx) const
{
	int offset = nIdx*elementStride;
	const unsigned char* ptr = reinterpret_cast<const unsigned char*>(array->clientData)+offset;

	nVec3.setFixed(0, *reinterpret_cast<const unsigned char*>(ptr)<<8);
	nVec3.setFixed(1, *reinterpret_cast<const unsigned char*>(ptr+1)<<8);
	nVec3.setFixed(2, array->size>2 ? *reinterpret_cast<const unsigned char*>(ptr+2)<<8 : 0);
}


inline void
klArrayFetcher::getVec3Fixed(klVec3& nVec3, int nIdx) const
{
	int offset = nIdx*elementStride;
	const unsigned char* ptr = reinterpret_cast<const unsigned char*>(array->clientData)+offset;

	nVec3.setFixed(0, *reinterpret_cast<const int*>(ptr));
	nVec3.setFixed(1, *reinterpret_cast<const int*>(ptr+4));
	nVec3.setFixed(2, array->size>2 ? *reinterpret_cast<const int*>(ptr+8) : 0);
}


inline void
klArrayFetcher::getVec3Short(klVec3& nVec3, int nIdx) const
{
	int offset = nIdx*elementStride;
	const unsigned char* ptr = reinterpret_cast<const unsigned char*>(array->clientData)+offset;

	nVec3.setInt(0, *reinterpret_cast<const short*>(ptr));
	nVec3.setInt(1, *reinterpret_cast<const short*>(ptr+2));
	nVec3.setInt(2, array->size>2 ? *reinterpret_cast<const short*>(ptr+4) : 0);
}


inline void
klArrayFetcher::getVec3Int(klVec3& nVec3, int nIdx) const
{
	int offset = nIdx*elementStride;
	const unsigned char* ptr = reinterpret_cast<const unsigned char*>(array->clientData)+offset;

	nVec3.setInt(0, *reinterpret_cast<const int*>(ptr));
	nVec3.setInt(1, *reinterpret_cast<const int*>(ptr+4));
	nVec3.setInt(2, array->size>2 ? *reinterpret_cast<const int*>(ptr+8) : 0);
}


inline void
klArrayFetcher::getVec3Float(klVec3& nVec3, int nIdx) const
{
	int offset = nIdx*elementStride;
	const unsigned char* ptr = reinterpret_cast<const unsigned char*>(array->clientData)+offset;

	nVec3.setFloat(0, *reinterpret_cast<const float*>(ptr));
	nVec3.setFloat(1, *reinterpret_cast<const float*>(ptr+4));
	nVec3.setFloat(2, array->size>2 ? *reinterpret_cast<const float*>(ptr+8) : 0.0f);
}


inline void
klArrayFetcher::getVec3Double(klVec3& nVec3, int nIdx) const
{
	int offset = nIdx*elementStride;
	const unsigned char* ptr = reinterpret_cast<const unsigned char*>(array->clientData)+offset;

	nVec3.setDouble(0, *reinterpret_cast<const double*>(ptr));
	nVec3.setDouble(1, *reinterpret_cast<const double*>(ptr+8));
	nVec3.setDouble(2, array->size>2 ? *reinterpret_cast<const double*>(ptr+16) : 0.0);
}


inline void
klArrayFetcher::getVec4UnsignedByte(klVec4& nVec4, int nIdx) const
{
	int offset = nIdx*elementStride;
	const unsigned char* ptr = reinterpret_cast<const unsigned char*>(array->clientData)+offset;

	nVec4.setFixed(0, *reinterpret_cast<const unsigned char*>(ptr)<<8);
	nVec4.setFixed(1, *reinterpret_cast<const unsigned char*>(ptr+1)<<8);
	nVec4.setFixed(2, array->size>2 ? *reinterpret_cast<const unsigned char*>(ptr+2)<<8 : 0);
	nVec4.setFixed(3, array->size>3 ? *reinterpret_cast<const int*>(ptr+3)<<8 : 1<<16);
}


inline void
klArrayFetcher::getVec4Fixed(klVec4& nVec4, int nIdx) const
{
	int offset = nIdx*elementStride;
	const unsigned char* ptr = reinterpret_cast<const unsigned char*>(array->clientData)+offset;

	nVec4.setFixed(0, *reinterpret_cast<const int*>(ptr));
	nVec4.setFixed(1, *reinterpret_cast<const int*>(ptr+4));
	nVec4.setFixed(2, array->size>2 ? *reinterpret_cast<const int*>(ptr+8) : 0);
	nVec4.setFixed(3, array->size>3 ? *reinterpret_cast<const int*>(ptr+12) : 1<<16);
}


inline void
klArrayFetcher::getVec4Short(klVec4& nVec4, int nIdx) const
{
	int offset = nIdx*elementStride;
	const unsigned char* ptr = reinterpret_cast<const unsigned char*>(array->clientData)+offset;

	nVec4.setInt(0, *reinterpret_cast<const short*>(ptr));
	nVec4.setInt(1, *reinterpret_cast<const short*>(ptr+2));
	nVec4.setInt(2, array->size>2 ? *reinterpret_cast<const short*>(ptr+4) : 0);
	nVec4.setInt(3, array->size>3 ? *reinterpret_cast<const short*>(ptr+6) : 1);
}


inline void
klArrayFetcher::getVec4Int(klVec4& nVec4, int nIdx) const
{
	int offset = nIdx*elementStride;
	const unsigned char* ptr = reinterpret_cast<const unsigned char*>(array->clientData)+offset;

	nVec4.setInt(0, *reinterpret_cast<const int*>(ptr));
	nVec4.setInt(1, *reinterpret_cast<const int*>(ptr+4));
	nVec4.setInt(2, array->size>2 ? *reinterpret_cast<const int*>(ptr+8) : 0);
	nVec4.setInt(3, array->size>3 ? *reinterpret_cast<const int*>(ptr+12) : 1);
}


inline void
klArrayFetcher::getVec4Float(klVec4& nVec4, int nIdx) const
{
	int offset = nIdx*elementStride;
	const unsigned char* ptr = reinterpret_cast<const unsigned char*>(array->clientData)+offset;

	nVec4.setFloat(0, *reinterpret_cast<const float*>(ptr));
	nVec4.setFloat(1, *reinterpret_cast<const float*>(ptr+4));
	nVec4.setFloat(2, array->size>2 ? *reinterpret_cast<const float*>(ptr+8) : 0.0f);
	nVec4.setFloat(3, array->size>3 ? *reinterpret_cast<const float*>(ptr+12) : 1.0f);
}


inline void
klArrayFetcher::getVec4Double(klVec4& nVec4, int nIdx) const
{
	int offset = nIdx*elementStride;
	const unsigned char* ptr = reinterpret_cast<const unsigned char*>(array->clientData)+offset;

		nVec4.setDouble(0, *reinterpret_cast<const double*>(ptr));
		nVec4.setDouble(1, *reinterpret_cast<const double*>(ptr+8));
		nVec4.setDouble(2, array->size>2 ? *reinterpret_cast<const double*>(ptr+16) : 0.0);
		nVec4.setDouble(3, array->size>3 ? *reinterpret_cast<const double*>(ptr+24) : 1.0);
}


#endif //__KLARRAY_HEADERFILE__
