/**************************************************************************\
 *
 *  This file is part of the Klimt library.
 *  Copyright (C) 2003 by IMS, Vienna University of Technology.
 *  All rights reserved.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  ("GPL") version 2 as published by the Free Software Foundation.
 *  See the file LICENSE.GPL at the root directory of this source
 *  distribution for additional information about the GNU GPL.
 *  For the full GPL license see
 *  <URL:http://www.gnu.org/copyleft/gpl.html>
 *
 *  For using Klimt with software that can not be combined with the
 *  GNU GPL, and for taking advantage of the additional benefits of
 *  our support services, please contact IMS about acquiring a
 *  Klimt Professional Edition License.
 *
 *  Contact: <mailto:klimt@studierstube.org>
 *  See <URL:http://www.studierstube.org/klimt>
 *  for more information.
 *
 *  Vienna University of Technology
 *  Institute for Software Technology and Interactive Systems
 *  Interactive Media Systems Group
 *  Favoritenstrasse 9-11/188/2
 *  A-1040 Vienna, Austria
 *  <URL:http://www.ims.tuwien.ac.at>.
 *
 **************************************************************************
 *
 * $Header: /cvsroot/klimt/klimt/klimt/src/Base/klFloat_float.h,v 1.4 2004/02/10 18:19:30 drgoldie Exp $
 *
\**************************************************************************/


#ifndef __KLFLOAT_FLOAT_HEADERFILE__
#define __KLFLOAT_FLOAT_HEADERFILE__

#include <math.h>


// Floating point implementation of klFloat.
// Although this class will be very slow on any hardware
// without an FPU the advantage is high precision.
//
class klFloat_float
{
public:
	klFloat_float()								{}
	//klFloat_float(int nV)							{  setInt(nV);  }  // could be mixed with fixed...
	klFloat_float(float nV		)					{  setFloat(nV);  }
	klFloat_float(double nV)						{  setDouble(nV);  }
	klFloat_float(const klFloat_float& nOther)	{  v = nOther.v;  }

	// custom interface
	//
	void setFixed(int nV)							{  v = nV/65536.0f;  }
	void setInt(int nV)								{  v = (float)nV;  }
	void setFloat(float nV)							{  v = nV;  }
	void setDouble(double nV)						{  v = (float)nV;  }
	void setKlFloat(const klFloat_float& nOther)	{  v = nOther.v;  }

	int getByte() const         {  return (int)(v*255.0f);  }
	int getFixed() const        {  return (int)(v*65536.0f);  }
	int getInt() const			{  return (int)v;  }
	float getFloat() const		{  return v;  }
	double getDouble() const	{  return (double)v;  }

	void inverse(const klFloat_float& nOther)		{  v = 1.0f/nOther.v;  }
	void inverse()									{  v = 1.0f/v;  }

	void inverseSqrt(const klFloat_float& nOther)	{  v = 1.0f/(float)sqrt(nOther.v);  }
	void inverseSqrt()								{  v = 1.0f/(float)sqrt(v);  }

	void multiplyBy255()							{  v *= 255.0f;  }


	// some standard math.h routines applied to this class
	//
	friend inline const klFloat_float sqrt(const klFloat_float& nV);
	friend inline const klFloat_float cos(const klFloat_float& nV);
	friend inline const klFloat_float sin(const klFloat_float& nV);
	friend inline const klFloat_float fabs(const klFloat_float& nV);
	friend inline const klFloat_float ceil(const klFloat_float& nV);


	// overloaded operators
	//
	klFloat_float& operator=(int nV)						{  setInt(nV);  return *this; }
	klFloat_float& operator=(float nV)						{  setFloat(nV);  return *this; }
	klFloat_float& operator=(double nV)					{  setDouble(nV);  return *this; }
	klFloat_float& operator=(const klFloat_float& nOther)	{  v = nOther.v;  return *this;  }

	klFloat_float operator-() const  {  klFloat_float w;  w.v = -v;  return w;  }

	klFloat_float& operator+=(int nV)	{  v+=(float)nV;  return *this;  }
	klFloat_float& operator-=(int nV)	{  v-=(float)nV;  return *this;  }
	klFloat_float& operator*=(int nV)	{  v*=(float)nV;  return *this;  }
	klFloat_float& operator/=(int nV)	{  v/=(float)nV;  return *this;  }

	klFloat_float& operator+=(float nV)	{  v+=nV;  return *this;  }
	klFloat_float& operator-=(float nV)	{  v-=nV;  return *this;  }
	klFloat_float& operator*=(float nV)	{  v*=nV;  return *this;  }
	klFloat_float& operator/=(float nV)	{  v/=nV;  return *this;  }

	klFloat_float& operator+=(double nV)	{  v+=(float)nV;  return *this;  }
	klFloat_float& operator-=(double nV)	{  v-=(float)nV;  return *this;  }
	klFloat_float& operator*=(double nV)	{  v*=(float)nV;  return *this;  }
	klFloat_float& operator/=(double nV)	{  v/=(float)nV;  return *this;  }

	klFloat_float& operator+=(const klFloat_float& nOther)	{  v+=nOther.v;  return *this;  }
	klFloat_float& operator-=(const klFloat_float& nOther)	{  v-=nOther.v;  return *this;  }
	klFloat_float& operator*=(const klFloat_float& nOther)	{  v*=nOther.v;  return *this;  }
	klFloat_float& operator/=(const klFloat_float& nOther)	{  v/=nOther.v;  return *this;  }

	klFloat_float& operator>>=(int nBits)	{  int tmp=1<<nBits;	v/=tmp;  return *this;  }
	klFloat_float& operator<<=(int nBits)	{  int tmp=1<<nBits;	v*=tmp;  return *this;  }

	bool operator==(const klFloat_float& nOther) const		{  return v==nOther.v;  }
	bool operator!=(const klFloat_float& nOther) const		{  return v!=nOther.v;  }
	bool operator<=(const klFloat_float& nOther) const		{  return v<=nOther.v;  }
	bool operator>=(const klFloat_float& nOther) const		{  return v>=nOther.v;  }
	bool operator<(const klFloat_float& nOther) const		{  return v<nOther.v;  }
	bool operator>(const klFloat_float& nOther) const		{  return v>nOther.v;  }

	bool operator==(int nOther) const		{  return v==(float)nOther;  }
	bool operator!=(int nOther) const		{  return v!=(float)nOther;  }
	bool operator<=(int nOther) const		{  return v<=(float)nOther;  }
	bool operator>=(int nOther) const		{  return v>=(float)nOther;  }
	bool operator<(int nOther) const		{  return v< (float)nOther;  }
	bool operator>(int nOther) const		{  return v> (float)nOther;  }

	bool operator==(float nOther) const		{  return v==nOther;  }
	bool operator!=(float nOther) const		{  return v!=nOther;  }
	bool operator<=(float nOther) const		{  return v<=nOther;  }
	bool operator>=(float nOther) const		{  return v>=nOther;  }
	bool operator<(float nOther) const		{  return v<nOther;  }
	bool operator>(float nOther) const		{  return v>nOther;  }

	friend inline const klFloat_float operator+(const klFloat_float& left, const klFloat_float& right);
	friend inline const klFloat_float operator-(const klFloat_float& left, const klFloat_float& right);
	friend inline const klFloat_float operator*(const klFloat_float& left, const klFloat_float& right);
	friend inline const klFloat_float operator/(const klFloat_float& left, const klFloat_float& right);

	friend inline const klFloat_float operator+(const klFloat_float& left, const klFloat_float& right);
	friend inline const klFloat_float operator+(const klFloat_float& left, float right);
	friend inline const klFloat_float operator+(float left, const klFloat_float& right);

	friend inline const klFloat_float operator-(const klFloat_float& left, const klFloat_float& right);
	friend inline const klFloat_float operator-(const klFloat_float& left, float right);
	friend inline const klFloat_float operator-(float left, const klFloat_float& right);

	friend inline const klFloat_float operator*(const klFloat_float& left, const klFloat_float& right);
	friend inline const klFloat_float operator*(const klFloat_float& left, float right);
	friend inline const klFloat_float operator*(float left, const klFloat_float& right);

	friend inline const klFloat_float operator/(const klFloat_float& left, const klFloat_float& right);
	friend inline const klFloat_float operator/(const klFloat_float& left, float right);
	friend inline const klFloat_float operator/(float left, const klFloat_float& right);

protected:
	float v;
};


// binary operator +
//
inline const klFloat_float
operator+(const klFloat_float& left, const klFloat_float& right)
{
	return klFloat_float(left.v+right.v);
}

inline const klFloat_float
operator+(const klFloat_float& left, float right)
{
	return klFloat_float(left.v+right);
}

inline const klFloat_float
operator+(float left, const klFloat_float& right)
{
	return klFloat_float(left+right.v);
}



// binary operator -
//
inline const klFloat_float
operator-(const klFloat_float& left, const klFloat_float& right)
{
	return klFloat_float(left.v-right.v);
}

inline const klFloat_float
operator-(const klFloat_float& left, float right)
{
	return klFloat_float(left.v-right);
}

inline const klFloat_float
operator-(float left, const klFloat_float& right)
{
	return klFloat_float(left-right.v);
}


// binary operator *
//
inline const klFloat_float
operator*(const klFloat_float& left, const klFloat_float& right)
{
	return klFloat_float(left.v*right.v);
}

inline const klFloat_float
operator*(const klFloat_float& left, float right)
{
	return klFloat_float(left.v*right);
}

inline const klFloat_float
operator*(float left, const klFloat_float& right)
{
	return klFloat_float(left*right.v);
}


// binary operator /
//
inline const klFloat_float
operator/(const klFloat_float& left, const klFloat_float& right)
{
	return klFloat_float(left.v/right.v);
}

inline const klFloat_float
operator/(const klFloat_float& left, float right)
{
	return klFloat_float(left.v/right);
}

inline const klFloat_float
operator/(float left, const klFloat_float& right)
{
	return klFloat_float(left/right.v);
}



// math.h methods
//
inline const klFloat_float
cos(const klFloat_float& nV)
{
	return klFloat_float(::cos(nV.v));
}


inline const klFloat_float
sin(const klFloat_float& nV)
{
	return klFloat_float(::sin(nV.v));
}


inline const klFloat_float
fabs(const klFloat_float& nV)
{
	return klFloat_float(::fabs(nV.v));
}


inline const klFloat_float
sqrt(const klFloat_float& nV)
{
	return klFloat_float(::sqrt(nV.v));
}


inline const klFloat_float
ceil(const klFloat_float& nV)
{
	return klFloat_float(::ceil(nV.v));
}


#endif //__KLFLOAT_FLOAT_HEADERFILE__
