#include "ZSurface.h"
#include <stdlib.h>
#include <qglobal.h>
#include <qrect.h>
#include "ZFont.h"

#ifndef USE_C      

/* ASM stuff  - the "aligned - non aligned" postfix doesn't mean anything - it is an artifax from the early design phase */
extern "C" 
{

   void flipNormal(void *dst, void *src);              
   void flipLandscape1(void *dst, void *src);          
   void flipLandscape2(void *dst, void *src);          

   void blitNormalAligned(void *blitData);

   void blitNormalBlendAligned(void *blitData);
   int  blitNormalBlend50Aligned(void *blitData);

   void fillSurface(void *memory, int length, int color);
   void fillRectAligned(void *blitData);
   void fillRectBlend50Aligned(void *blitData);
   void fillRectBlendAligned(void *blitData);



   void blitKeyedAligned(void *blitData);
   void blitKeyedBlend50Aligned(void *blitData);
   void blitKeyedBlend(void *blitData);

   int blitAlpha(void *blitData);


   void fillKeyedAligned(void *blitData);
   void fillKeyedBlend50Aligned(void *blitData);
   void fillKeyedBlend(void *blitData);



}
#endif

ZSurface::ZROTATION ZSurface::zrot=Z0;


ZSurface::ZSurface()
{
   initSurface();   
}

ZSurface::ZSurface(int width, int height, bool alpha)
{
   initSurface();   
   create(width,height,alpha);    

}

ZSurface::ZSurface(const QImage &i, bool alpha)
{
   initSurface();   
   create(i,alpha);

}

ZSurface::ZSurface(const QPixmap &p, bool alpha)
{
   initSurface();
   create(p,alpha);
}

// Copy constructor

ZSurface::ZSurface(const ZSurface &s)
{
   copySurface(s);
}

ZSurface & ZSurface::operator=(const ZSurface &s)
{
   if ( this!=&s )
   {
      copySurface(s);         
   }
   return *this;      
}

void ZSurface::copySurface(const ZSurface &s)
{
   create(s.width(),s.height(),s.hasAlpha());                 
   if ( data )
   {
      memcpy(data,s.data,s.size());
   }
   if ( adata )
   {
      memcpy(adata,s.adata,s.iTW*s.iH*1);
   }
}


ZSurface::~ZSurface()
{
   if ( data )
   {
      free(data);
   }
   if ( adata )
   {
      free(adata);
   }
}

// main method for allocating/setting up memory for the surface
// Create new databuffer 

bool ZSurface::create(int width, int height, bool alpha)
{
   int oldSize=size();
   iTW=iW=width;     
   iH=height;
   if ( iW&1 )
      iTW++;

   if ( oldSize!=size() )                             // if size has changed - we need to allocate new buffer
   {
      if ( data )
      {
         free (data);
         data=0;
      }
      if ( adata )
      {
         free (adata);
         adata=0;
      }
      if ( !size() )                                  // if size==0 , make this an invalid surface
      {
         initSurface(); 
         return true; 
      }
      if ( !(data=(unsigned char*)malloc(iTW*iH*iD)) )
         return false;
      if ( alpha )
      {
         adata=(unsigned char*)malloc(iTW*iH*iD);   // if alpha alloc fails - will simply won't use - don't report it as an error                  
         if ( adata )
            memset(adata,255,iTW*iH*iD);              // set everything to opaque
      }
   }
   return true;
}


bool  ZSurface::create(const QImage &i, bool alpha)
{
   if ( i.depth()==32 )
   {
      if ( create(i.width(),i.height(),alpha) )
      {
         initFromImage(i);
         return true;
      }
   }
   return false;
}

bool  ZSurface::create(const QPixmap &p, bool alpha)
{
   return create(p.convertToImage(), alpha);   
}


void ZSurface::fill(QColor clr)
{
   if ( !isValid() )
      return;
#ifdef USE_C   
   unsigned short val=qt_convRgbTo16(clr.rgb());  
   register ushort *p = (ushort *)data;   
   ushort *end = p + iTW*iH;
   int cnt=0;
   while ( p < end )
   {
      *p++ = val;
      cnt++;
   }
#else
   fillSurface(data,iTW*iH, qt_convRgbTo16(clr.rgb()));
#endif
}


// Draws a text on a surface using default ZFont image glyphs and optional blending using static alpha value (0-32)

int ZSurface::drawText(int dx, int dy,const QString &s, ZFont *f,int opacity)
{
   const char *text=s.latin1();
   int   posX=0;

   for ( int i=0;i<s.length();i++ )
   {
      if(opacity==32)
         ZSurface::bitBlitKeyed(this,dx+posX,dy,f->backgroundColor(),(ZSurface*)f,f->x(text[i]),f->y(),f->charWidth(text[i]),f->height()); 
      else
         ZSurface::bitBlitKeyedBlend(opacity,this,dx+posX,dy,f->backgroundColor(),(ZSurface*)f,f->x(text[i]),f->y(),f->charWidth(text[i]),f->height());
      posX+=f->charWidth(text[i])+f->spacing();
   }
   return posX;
}


// Draws a text on a surface using color "c" and optional blending using static alpha value (0-32)

int ZSurface::drawText(int dx, int dy,const QString &s, ZFont *f,QColor c, int opacity)
{
   const char *text=s.latin1();
   int   posX=0;

   for ( int i=0;i<s.length();i++ )
   {

      if(opacity==32)
         ZSurface::fillBlitKeyed(this,dx+posX,dy,f->backgroundColor(),c,(ZSurface*)f,f->x(text[i]),f->y(),f->charWidth(text[i]),f->height());
      else
         ZSurface::fillBlitKeyedBlend(opacity,this,dx+posX,dy,f->backgroundColor(),c,(ZSurface*)f,f->x(text[i]),f->y(),f->charWidth(text[i]),f->height());
      posX+=f->charWidth(text[i])+f->spacing();
   }

   return posX;
}


// Private stuff

void ZSurface::initSurface()
{
   data=0;
   adata=0;
   iD=2;

}

// Init surface using QImage ( QImage MUST be 32 bit)


void ZSurface::initFromImage(const QImage &i)
{
   bool hasAlpha=(i.hasAlphaBuffer() && adata);

   unsigned short *dt=(unsigned short *)data;
   unsigned short  *ad=(unsigned short *)adata;
   int x,y;
   int oneMore=iTW-iW;
   QRgb pixel;

   for ( y=0;y<i.height();y++ )
   {
      for ( x=0;x<i.width();x++ )
      {
         pixel=i.pixel(x,y);
         if ( hasAlpha )
         {
            *ad++=(unsigned short)(qAlpha(pixel)>>2);            
         }
         *dt++=qt_convRgbTo16(pixel);
      }
      dt+=oneMore;
      if ( hasAlpha )
         ad+=oneMore;
   }

}


// Bunch of code to perform line -> rectangle clipping

unsigned int   ZSurface::getCode (int x, int y, QRect r)
{
   unsigned int c = 0;
   if ( y > r.bottom() )
      c |= TOP;
   else if ( y < r.top() )
      c |= BOTTOM;
   if ( x > r.right() )
      c |= RIGHT;
   else if ( x < r.left() )
      c |= LEFT;
   return c;
}

bool  ZSurface::clipLine (int &x0, int &y0, int &x1, int &y1, QRect clip)
{
   unsigned int C0, C1, C;
   int x, y;

   C0 = getCode (x0, y0, clip);
   C1 = getCode (x1, y1, clip);

   for ( ;; )
   {
      if ( (C0 | C1) == 0 )
      {
         return true;
      }

      if ( (C0 & C1) != 0 )
         return false;

      C = C0 ? C0 : C1;
      if ( C & TOP )
      {
         x = x0 + (x1 - x0) * (clip.bottom() - y0) / (y1 - y0);
         y = clip.bottom();
      }
      else if ( C & BOTTOM )
      {
         x = x0 + (x1 - x0) * (clip.top() - y0) / (y1 - y0);
         y = clip.top();
      }
      else if ( C & RIGHT )
      {
         x = clip.right();
         y = y0 + (y1 - y0) * (clip.right() - x0) / (x1 - x0);
      }
      else
      {
         x = clip.left();
         y = y0 + (y1 - y0) * (clip.left() - x0) / (x1 - x0);
      }

      if ( C == C0 )
      {
         x0 = x; y0 = y;
         C0 = getCode (x0, y0, clip);
      }
      else
      {
         x1 = x; y1 = y;
         C1 = getCode (x1, y1, clip);
      }
   }
}

// Draws an alpha blended line on the surface using color "color" and static alpha value "opacity" (0-32)

bool  ZSurface::drawLineBlend(int x1, int y1, int x2, int y2, QColor color, int opacity)
{

   QRect clip(0,0,iW,iH);

   if ( !clipLine(x1,y1,x2,y2,clip) )
      return false;

   int             x, y;
   int             dx, dy;
   int             incx, incy;
   int             balance;

   if ( x2 >= x1 )
   {
      dx = x2 - x1;
      incx = 1;
   }
   else
   {
      dx = x1 - x2;
      incx = -1;
   }

   if ( y2 >= y1 )
   {
      dy = y2 - y1;
      incy = 1;
   }
   else
   {
      dy = y1 - y2;
      incy = -1;
   }

   x = x1;
   y = y1;

   unsigned short *dest=(unsigned short*)data;
   unsigned short pixel=qt_convRgbTo16(color.rgb());

   unsigned short pixel1=pixel & 0xf81f;
   unsigned short pixel2=pixel & 0x07e0;


   dest=dest+(y*iTW)+x;

   int yOffset=incy*iTW;

   unsigned short dwAlphaRBtemp;
   unsigned short dwAlphaGtemp;
   unsigned short dw6bitOpacity = ((opacity*8) >> 2);

   unsigned short backPixel;

   if ( dx >= dy )
   {
      dy <<= 1;
      balance = dy - dx;
      dx <<= 1;      

      while ( x != x2 )
      {
         backPixel=*dest;
         dwAlphaRBtemp = (backPixel & 0xf81f);
         dwAlphaGtemp = (backPixel & 0x07e0);
         *dest=((dwAlphaRBtemp + ((((pixel1) - dwAlphaRBtemp) * dw6bitOpacity) >> 6)) & 0xf81f) |
               ((dwAlphaGtemp + ((((pixel2) - dwAlphaGtemp) * dw6bitOpacity) >> 6)) & 0x07e0);            
         if ( balance >= 0 )
         {
            dest+=yOffset;
            balance -= dx;
         }
         balance += dy;
         x += incx;
         dest+=incx;
      }       
      backPixel=*dest;
      dwAlphaRBtemp = (backPixel & 0xf81f);
      dwAlphaGtemp = (backPixel & 0x07e0);
      *dest=((dwAlphaRBtemp + ((((pixel1) - dwAlphaRBtemp) * dw6bitOpacity) >> 6)) & 0xf81f) |
            ((dwAlphaGtemp + ((((pixel2) - dwAlphaGtemp) * dw6bitOpacity) >> 6)) & 0x07e0); 
   }
   else
   {
      dx <<= 1;
      balance = dx - dy;
      dy <<= 1;


      while ( y != y2 )
      {
         backPixel=*dest;
         dwAlphaRBtemp = (backPixel & 0xf81f);
         dwAlphaGtemp = (backPixel & 0x07e0);
         *dest=((dwAlphaRBtemp + ((((pixel1) - dwAlphaRBtemp) * dw6bitOpacity) >> 6)) & 0xf81f) |
               ((dwAlphaGtemp + ((((pixel2) - dwAlphaGtemp) * dw6bitOpacity) >> 6)) & 0x07e0); 
         if ( balance >= 0 )
         {
            dest+=incx;
            balance -= dy;
         }
         balance += dx;
         y += incy;

         dest+=yOffset;
      } 

      backPixel=*dest;
      dwAlphaRBtemp = (backPixel & 0xf81f);
      dwAlphaGtemp = (backPixel & 0x07e0);
      *dest=((dwAlphaRBtemp + ((((pixel1) - dwAlphaRBtemp) * dw6bitOpacity) >> 6)) & 0xf81f) |
            ((dwAlphaGtemp + ((((pixel2) - dwAlphaGtemp) * dw6bitOpacity) >> 6)) & 0x07e0); 
   }

   return true;

}

// Draws a line on the surface using color "color" 

bool  ZSurface::drawLine(int x1, int y1, int x2, int y2, QColor color)
{

   QRect clip(0,0,iW,iH);

   if ( !clipLine(x1,y1,x2,y2,clip) )
      return false;

   int             x, y;
   int             dx, dy;
   int             incx, incy;
   int             balance;

   if ( x2 >= x1 )
   {
      dx = x2 - x1;
      incx = 1;
   }
   else
   {
      dx = x1 - x2;
      incx = -1;
   }
   if ( y2 >= y1 )
   {
      dy = y2 - y1;
      incy = 1;
   }
   else
   {
      dy = y1 - y2;
      incy = -1;
   }

   x = x1;
   y = y1;

   unsigned short *dest=(unsigned short*)data;
   unsigned short pixel=qt_convRgbTo16(color.rgb());

   dest=dest+(y*iTW)+x;

   int yOffset=incy*iTW;

   if ( dx >= dy )
   {
      dy <<= 1;
      balance = dy - dx;
      dx <<= 1;      

      while ( x != x2 )
      {
         *dest=pixel;            
         if ( balance >= 0 )
         {
            dest+=yOffset;
            balance -= dx;
         }
         balance += dy;
         x += incx;
         dest+=incx;
      }       
      *dest=pixel;
   }
   else
   {
      dx <<= 1;
      balance = dx - dy;
      dy <<= 1;

      while ( y != y2 )
      {
         *dest=pixel;
         if ( balance >= 0 )
         {
            dest+=incx;
            balance -= dy;
         }
         balance += dx;
         y += incy;
         dest+=yOffset;
      } 
      *dest=pixel;      
   }

   return true;
}

// Blit a source sourface to a destination surface using the key color as a transparency indicator.
// Pixels matching key will not be transgfered to the destination surface.
// Uses the color value in fillColor instead of source pixels to perform a color fill.

bool ZSurface::fillBlitKeyed(ZSurface *dst, int dx, int dy,const QColor &key,const QColor &fillColor, const ZSurface *src, int sx, int sy, int sw, int sh)
{

   BLIT_PARS bpar;     
   bpar.key=(int)qt_convRgbTo16(key.rgb());
   bpar.color=(int)qt_convRgbTo16(fillColor.rgb());

   if ( !computeBlitPars(&bpar,dst,dx,dy,src,sx,sy,sw,sh) )
      return false;
#ifdef USE_C
   unsigned short *dest=(unsigned short*)bpar.dst;
   unsigned short *source=(unsigned short*)bpar.src;
   unsigned short pixel=(unsigned short)bpar.key;
   unsigned short pixel2=(unsigned short)bpar.color;
   int   srcOffset=bpar.ss/bpar.depth;
   int   dstOffset=bpar.ds/bpar.depth;
   int   sWidth=bpar.sw;

   for ( int i=0;i<bpar.sh;i++ )
      while ( bpar.sh )
      {
         while ( sWidth )
         {
            if ( *source!=pixel )
               *dest=pixel2;
            dest++;
            source++;
            sWidth--;
         }
         dest+=dstOffset;
         source+=srcOffset;
         bpar.sh--;
         sWidth=bpar.sw;
      }
#else
   fillKeyedAligned(&bpar);
#endif

   return true;
}

// Blit a source sourface to a destination surface using the key color as a transparency indicator.
// Pixels matching key will not be transgfered to the destination surface.
// Uses the color value in fillColor instead of source pixels to perform a color fill.
// Performs alpha blending using single alpha value (0-32)


bool ZSurface::fillBlitKeyedBlend(int opacity,ZSurface *dst, int dx, int dy,const QColor &key,const QColor &fillColor, const ZSurface *src,
                                  int sx=0, int sy=0, int sw=-1, int sh=-1)
{
   BLIT_PARS bpar;     
   bpar.key=(int)qt_convRgbTo16(key.rgb());
   bpar.color=(int)qt_convRgbTo16(fillColor.rgb());

   if ( !computeBlitPars(&bpar,dst,dx,dy,src,sx,sy,sw,sh) )
      return false;

#ifndef USE_C
   bpar.alpha=opacity;

   if ( opacity==16 )
      fillKeyedBlend50Aligned(&bpar);
   else
      fillKeyedBlend(&bpar);
#else

   unsigned short *dest=(unsigned short*)bpar.dst;
   unsigned short *source=(unsigned short*)bpar.src;
   unsigned short pixel=(unsigned short)bpar.key;
   unsigned short pixel2=(unsigned short)bpar.color;
   int   srcOffset=bpar.ss/bpar.depth;
   int   dstOffset=bpar.ds/bpar.depth;
   int   sWidth=bpar.sw;

   unsigned short dwAlphaRBtemp;
   unsigned short dwAlphaGtemp;
   unsigned short dw6bitOpacity = ((opacity*8) >> 2);

   unsigned short pixelA=pixel2 & 0xf81f;
   unsigned short pixelB=pixel2 & 0x07e0;
   unsigned short backPixel;

   while ( bpar.sh )
   {
      while ( sWidth )
      {
         if ( *source!=pixel )
         {
            backPixel=*dest;
            dwAlphaRBtemp = (backPixel & 0xf81f);
            dwAlphaGtemp = (backPixel & 0x07e0);
            *dest=((dwAlphaRBtemp + ((((pixelA) - dwAlphaRBtemp) * dw6bitOpacity) >> 6)) & 0xf81f) |
                  ((dwAlphaGtemp + ((((pixelB) - dwAlphaGtemp) * dw6bitOpacity) >> 6)) & 0x07e0); 
         }
         dest++;
         source++;
         sWidth--;
      }
      dest+=dstOffset;
      source+=srcOffset;
      bpar.sh--;
      sWidth=bpar.sw;
   }
#endif
   return true;

}
// Blit a source sourface to a destination surface using the key color as a transparency indicator.
// Pixels matching key will not be transgfered to the destination surface.

bool ZSurface::bitBlitKeyed(ZSurface *dst, int dx, int dy, const QColor &key, const ZSurface *src, int sx, int sy, int sw, int sh)
{
   BLIT_PARS bpar;    
   bpar.key=(int)qt_convRgbTo16(key.rgb());
   if ( !computeBlitPars(&bpar,dst,dx,dy,src,sx,sy,sw,sh) )
      return false;

#ifdef USE_C

   unsigned short *dest=(unsigned short*)bpar.dst;
   unsigned short *source=(unsigned short*)bpar.src;
   unsigned short pixel=(unsigned short)bpar.key;
   int   srcOffset=bpar.ss/bpar.depth;
   int   dstOffset=bpar.ds/bpar.depth;
   int   sWidth=bpar.sw;

   while ( bpar.sh )
   {
      while ( sWidth )
      {
         if ( *source!=pixel )
            *dest=*source;
         dest++;
         source++;
         sWidth--;
      }
      dest+=dstOffset;
      source+=srcOffset;
      bpar.sh--;
      sWidth=bpar.sw;
   }
#else
   blitKeyedAligned(&bpar);
#endif


   return true;
}

// Blit a source sourface to a destination surface using the key color as a transparency indicator.
// Pixels matching key will not be transgfered to the destination surface.
// Perform alpha blending using single alpha value (0-32)

bool ZSurface::bitBlitKeyedBlend(int opacity, ZSurface *dst, int dx, int dy,const QColor &key, const ZSurface *src, int sx=0, int sy=0, int sw=-1, int sh=-1)
{
   BLIT_PARS bpar;    
   bpar.key=(int)qt_convRgbTo16(key.rgb());

   if ( !computeBlitPars(&bpar,dst,dx,dy,src,sx,sy,sw,sh) )
      return false;

#ifndef USE_C 
   bpar.alpha=opacity;

   if ( opacity==16 )
      blitKeyedBlend50Aligned(&bpar);
   else
   {
      blitKeyedBlend(&bpar);      
   }
   return true;
#else

   unsigned short *dest=(unsigned short*)bpar.dst;
   unsigned short *source=(unsigned short*)bpar.src;
   unsigned short keyPixel=(unsigned short)bpar.key;
   int   srcOffset=bpar.ss/bpar.depth;
   int   dstOffset=bpar.ds/bpar.depth;
   int   sWidth=bpar.sw;

   unsigned short dwAlphaRBtemp;
   unsigned short dwAlphaGtemp;
   unsigned short dw6bitOpacity = ((opacity*8) >> 2);
   unsigned short backPixel;
   unsigned short pixel;

   while ( bpar.sh )
   {
      while ( sWidth )
      {
         pixel=*source;
         if ( pixel!=keyPixel )
         {
            backPixel=*dest;            
            dwAlphaRBtemp = (backPixel & 0xf81f);
            dwAlphaGtemp = (backPixel & 0x07e0);
            *dest=((dwAlphaRBtemp + ((((pixel & 0xf81f) - dwAlphaRBtemp) * dw6bitOpacity) >> 6)) & 0xf81f) |
                  ((dwAlphaGtemp + ((((pixel & 0x07e0) - dwAlphaGtemp) * dw6bitOpacity) >> 6)) & 0x07e0); 

         }

         dest++;
         source++;
         sWidth--;
      }
      dest+=dstOffset;
      source+=srcOffset;
      bpar.sh--;
      sWidth=bpar.sw;
   }
#endif
   return true;
}



// Blit a source sourface to a destination surface
bool ZSurface::bitBlit(ZSurface *dst, int dx, int dy, const ZSurface *src, int sx, int sy, int sw, int sh)
{
   BLIT_PARS bpar;    

   if ( !computeBlitPars(&bpar,dst,dx,dy,src,sx,sy,sw,sh) )
      return false;

#ifdef USE_C 

   int depth=dst->iD;
   int destOffset=(bpar.sw*depth)+(bpar.ds);
   int srcOffset=(bpar.sw*depth)+(bpar.ss);

   int swidth=bpar.sw*depth;

   for ( int i=0;i<bpar.sh;i++ )
   {
      memcpy(bpar.dst,bpar.src,swidth);
      bpar.dst+=destOffset;
      bpar.src+=srcOffset;
   }
#else   
   blitNormalAligned(&bpar);
#endif
   return true;
}


// Blit a source sourface to a destination surface using a single alpha value (0-32)

bool ZSurface::bitBlitBlend(int opacity, ZSurface *dst, int dx, int dy, const ZSurface *src, int sx, int sy, int sw, int sh)
{
   if(!opacity)
      return true;   
   if(opacity==32)
      return  bitBlit(dst,dx,dy,src,sx,sy,sw,sh);

   BLIT_PARS bpar;    
   if ( !computeBlitPars(&bpar,dst,dx,dy,src,sx,sy,sw,sh) )
      return false;

#ifndef USE_C 


   bpar.alpha=opacity;

   int res;

   if ( opacity==16 )
      blitNormalBlend50Aligned(&bpar);
   else   
      blitNormalBlendAligned(&bpar);
   

#else
   unsigned short *dest=(unsigned short*)bpar.dst;
   unsigned short *source=(unsigned short*)bpar.src;
   int   srcOffset=bpar.ss/bpar.depth;
   int   dstOffset=bpar.ds/bpar.depth;
   int   sWidth=bpar.sw;

   unsigned short dwAlphaRBtemp;
   unsigned short dwAlphaGtemp;
   unsigned short dw6bitOpacity = (opacity*8 >> 2);

   unsigned short backPixel;
   unsigned short pixel;

   while ( bpar.sh )
   {
      while ( sWidth )
      {
         backPixel=*dest;
         pixel=*source++;
         dwAlphaRBtemp = (backPixel & 0xf81f);
         dwAlphaGtemp = (backPixel & 0x07e0);
         *dest++=((dwAlphaRBtemp + ((((pixel & 0xf81f) - dwAlphaRBtemp) * dw6bitOpacity) >> 6)) & 0xf81f) |
                 ((dwAlphaGtemp + ((((pixel & 0x07e0) - dwAlphaGtemp) * dw6bitOpacity) >> 6)) & 0x07e0); 
         sWidth--;
      }
      dest+=dstOffset;
      source+=srcOffset;
      bpar.sh--;
      sWidth=bpar.sw;
   }
#endif
   return true;
}



//  Blit a source sourface to a destination  surface using an src alpha information. 
// If the source surface doesn't have alpha channel defined then use standard opaque blit instead.

bool ZSurface::bitBlitAlpha(ZSurface *dst, int dx, int dy, const ZSurface *src, int sx, int sy, int sw, int sh)
{
   if ( src && !src->adata )
   {
      return bitBlit(dst,dx,dy,src,sx,sy,sw,sh);      // if there is no alpha channel then just do standard blit
   }


   BLIT_PARS bpar;    

   if ( !computeBlitPars(&bpar,dst,dx,dy,src,sx,sy,sw,sh) )
      return false;

#ifndef USE_C 
   blitAlpha(&bpar);
#else
   unsigned short *dest=(unsigned short*)bpar.dst;
   unsigned short *source=(unsigned short*)bpar.src;
   unsigned short *asource=(unsigned short*)bpar.asrc;
   int   srcOffset=bpar.ss/bpar.depth;
   int   dstOffset=bpar.ds/bpar.depth;
   int   sWidth=bpar.sw;

   unsigned short dwAlphaRBtemp;
   unsigned short dwAlphaGtemp;
   unsigned short dw6bitOpacity; 

   unsigned short backPixel;
   unsigned short pixel;

   while ( bpar.sh )
   {
      while ( sWidth )
      {
         dw6bitOpacity=(*asource++);
         if ( dw6bitOpacity )
         {
            if ( dw6bitOpacity==63 )
               *dest=*source;
            else
            {
               backPixel=*dest;
               pixel=*source;
               dwAlphaRBtemp = (backPixel & 0xf81f);
               dwAlphaGtemp = (backPixel & 0x07e0);
               *dest=((dwAlphaRBtemp + ((((pixel & 0xf81f) - dwAlphaRBtemp) * dw6bitOpacity) >> 6)) & 0xf81f) |
                     ((dwAlphaGtemp + ((((pixel & 0x07e0) - dwAlphaGtemp) * dw6bitOpacity) >> 6)) & 0x07e0); 
            }

         }
         dest++;
         source++;                   
         sWidth--;
      }
      dest+=dstOffset;
      source+=srcOffset;
      asource+=srcOffset;
      bpar.sh--;
      sWidth=bpar.sw;
   }

#endif

   return true;
}

//  Draw filled rectangle

bool ZSurface::fillRect(ZSurface *dst, int dx, int dy,int rw, int rh,const QColor &clr)
{
   BLIT_PARS bpar;     
   bpar.key=(int)qt_convRgbTo16(clr.rgb());

   if ( !computeBlitPars(&bpar,dst,dx,dy,rw,rh) )
      return false;

#ifdef USE_C 
   unsigned short *dest=(unsigned short*)bpar.dst;
   unsigned short pixel=(unsigned short)bpar.key;
   int   dstOffset=bpar.ds/bpar.depth;
   for ( int i=0;i<bpar.sh;i++ )
   {
      for ( int j=0;j<bpar.sw;j++ )
      {
         *dest++=pixel;
      }
      dest+=dstOffset;
   }
#else
   fillRectAligned(&bpar);
#endif
   return true;
}



//  Draw filled rect blending the fill color with a destination surface pixels ( 32 levels of alpha)

bool ZSurface::fillRectBlend(int opacity,ZSurface *dst, int dx, int dy,int rw, int rh,const QColor &clr)
{
   if(!opacity)
      return true;   
   if(opacity==32)
      return  fillRect(dst,dx,dy,rw,rh,clr);

   BLIT_PARS bpar;     
   bpar.key=(int)qt_convRgbTo16(clr.rgb());

   if (!computeBlitPars(&bpar,dst,dx,dy,rw,rh) )  
      return false;

#ifndef USE_C 
   bpar.alpha=opacity;

   if ( bpar.alpha==16 )
   {
      fillRectBlend50Aligned(&bpar);
      return true;
   }
   else
   {
      fillRectBlendAligned(&bpar);
      return true;
   }
#else

   unsigned short dwAlphaRBtemp;
   unsigned short dwAlphaGtemp;
   unsigned short dw6bitOpacity = ((opacity*8) >> 2);

   unsigned short backPixel;
   unsigned short pixel=(unsigned short)bpar.key;


   unsigned short pixel1=pixel & 0xf81f;
   unsigned short pixel2=pixel & 0x07e0;


   unsigned short *dest=(unsigned short*)bpar.dst;

   int   dstOffset=bpar.ds/bpar.depth;
   for ( int i=0;i<bpar.sh;i++ )
   {
      for ( int j=0;j<bpar.sw;j++ )
      {
         backPixel=*dest;
         dwAlphaRBtemp = (backPixel & 0xf81f);
         dwAlphaGtemp = (backPixel & 0x07e0);
         *dest++=((dwAlphaRBtemp + ((((pixel1) - dwAlphaRBtemp) * dw6bitOpacity) >> 6)) & 0xf81f) |
                 ((dwAlphaGtemp + ((((pixel2) - dwAlphaGtemp) * dw6bitOpacity) >> 6)) & 0x07e0); 
      }
      dest+=dstOffset;
   }
#endif

   return true;
}


// Generic clipping function used for population BLIT_PARS struct with info as expected by asm code
// Returns false if the target rectangle for the operation does not intersect with the destination surface rectangle 
// Used by every blit with the exception of fillRect ....


bool ZSurface::computeBlitPars( BLIT_PARS *bpar,ZSurface *dst, int dx, int dy, const ZSurface *src, int sx, int sy, int sw, int sh)
{
   if ( (!dst || !dst->isValid()) || (!src || !src->isValid()) )
      return false;


   if ( sw < 0 ) 
      sw = src->width();
   if ( sh < 0 ) 
      sh = src->height();
   if ( sx < 0 )
   {
      dx -= sx; sw += sx; sx = 0;
   }
   if ( sy < 0 )
   {
      dy -= sy; sh += sy; sy = 0;
   }
   if ( dx < 0 )
   {
      sx -= dx; sw += dx; dx = 0;
   }
   if ( dy < 0 )
   {
      sy -= dy; sh += dy; dy = 0;
   }
   if ( sx + sw > src->width() ) 
      sw = src->width() - sx;
   if ( sy + sh > src->height() ) 
      sh = src->height() - sy;
   if ( dx + sw > dst->width() ) 
      sw = dst->width() - dx;
   if ( dy + sh > dst->height() ) 
      sh = dst->height() - dy;
   if ( sw <= 0 || sh <= 0 ) 
      return false;
   if ( (dst == src) && dx==sx && dy==sy ) 
      return false;


   bpar->depth=src->iD;
   bpar->sw=sw;
   bpar->sh=sh;
   bpar->ss=(src->iTW-bpar->sw)*src->iD;            
   bpar->ds=(dst->iTW-sw)*dst->iD;    

   bpar->src=src->data+(sy*(src->iTW*src->iD))+(sx*src->iD);
   bpar->dst=dst->data+(dy*(dst->iTW*dst->iD))+(dx*dst->iD);
   if ( src->adata )
   {
      bpar->asrc=src->adata+(sy*(src->iTW*src->iD))+(sx*src->iD);

   }
   else
      bpar->asrc=0;

   return true;
}


// Generic clipping function used for population BLIT_PARS struct with info as expected by asm code
// Returns false if the target rectangle for the operation does not intersect with the destination surface rectangle 
// Used only for fills etc ...

bool ZSurface::computeBlitPars( BLIT_PARS *bpar,ZSurface *dst, int dx, int dy,int sw, int sh)
{

   if ( !dst || !dst->isValid() )
      return false;


   if ( sw < 0 ) 
      sw = dst->width();
   if ( sh < 0 ) 
      sh = dst->height();
   if ( dx < 0 )
   {
      sw += dx; dx = 0;
   }
   if ( dy < 0 )
   {
      sh += dy; dy = 0;
   }
   if ( dx + sw > dst->width() ) 
      sw = dst->width() - dx;
   if ( dy + sh > dst->height() ) 
      sh = dst->height() - dy;
   if ( sw <= 0 || sh <= 0 ) 
      return false;


   bpar->depth=dst->iD;
   bpar->sw=sw;
   bpar->sh=sh;            
   bpar->ds=(dst->iTW-sw)*dst->iD;    
   bpar->dst=dst->data+(dy*(dst->iTW*dst->iD))+(dx*dst->iD);


   return true;
}



// Here we transfer our backbuffer ( which MUST be of the same size as the framebuffer (screen))
// TODO: next step will be to allow for partial backbuffer transfers - right it is an entire screen or nothing

bool ZSurface::bitBlit(QDirectPainter *dst, const ZSurface *src,ZROTATION r)
{
   if ( !dst || (!src || !src->isValid()) )
   {
      return false;
   }

   int orient=dst->transformOrientation ();

#ifdef USE_C 

   // we have to fool qvfb to update entire screen
   dst->fillRect(0,0,240,320, QColor(0,0,0));
   memcpy(dst->frameBuffer(),src->data,320*240*2);//src->width()*src->height()*src->depth());

#else
   if ( orient==0 )                       // no rotation 
   {
       flipLandscape1(dst->frameBuffer(),src->data);      // just plain memcpy
   }
   else if ( orient==3 )                // 270 degrees -Zaurus 5500
   {
      switch ( r )
      {
         case  Z0:
            flipNormal(dst->frameBuffer(),src->data);
            break;
         case   Z90:
            flipLandscape2(dst->frameBuffer(),src->data);
            break;
         case   Z270:
            flipLandscape1(dst->frameBuffer(),src->data);   // just plain memcpy
            break;
         default:
            return false;
      }       
   }
   else
   {
      return false;
   }

#endif
   return true;
}

