/**************************************************************************\
 *
 *  This file is part of the Klimt library.
 *  Copyright (C) 2003 by IMS, Vienna University of Technology.
 *  All rights reserved.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  ("GPL") version 2 as published by the Free Software Foundation.
 *  See the file LICENSE.GPL at the root directory of this source
 *  distribution for additional information about the GNU GPL.
 *  For the full GPL license see
 *  <URL:http://www.gnu.org/copyleft/gpl.html>
 *
 *  For using Klimt with software that can not be combined with the
 *  GNU GPL, and for taking advantage of the additional benefits of
 *  our support services, please contact IMS about acquiring a
 *  Klimt Professional Edition License.
 *
 *  Contact: <mailto:klimt@studierstube.org>
 *  See <URL:http://www.studierstube.org/klimt>
 *  for more information.
 *
 *  Vienna University of Technology
 *  Institute for Software Technology and Interactive Systems
 *  Interactive Media Systems Group
 *  Favoritenstrasse 9-11/188/2
 *  A-1040 Vienna, Austria
 *  <URL:http://www.ims.tuwien.ac.at>.
 *
 **************************************************************************
 *
 * $Header: /cvsroot/klimt/klimt/klimt/src/RasterizerSW565/klRSW565_HelperStructs.h,v 1.3 2004/02/10 18:19:31 drgoldie Exp $
 *
\**************************************************************************/

//
//class klRSW565 {
//

//
// klRasterizerSW565 internal helper structs
//


// A vertex for internal usage
//
struct DeviceVertex
{
	klVec3 pos;		// 3D position on the screen
	klColor4 col;	// color as RGBA
	klVec3 tex;		// texture coordinate (should be klVec2 actually)
	klFloat texZ;	// z-value for texturing
	klFloat fog;	// fog density value
};


// Mipmap level data
//
struct MipMapLevel : public klMipMapLevel_Info
{
	MipMapLevel() : pixels(NULL)  {}
	~MipMapLevel()
	{
		if(pixels)
			delete pixels;
	}

	TEX_PIXELTYPE*	pixels;			// have pixels in 32-bits: [X8-A8-RGB565]
};

typedef MipMapLevel*  MipMapLevelPtr;


// Texture object data
struct Texture
{
	enum
	{
		MAX_MIPMAPLEVELS = 10
	};

	Texture()
		{	for(int i=0; i<MAX_MIPMAPLEVELS; i++)  mmLevels[i] = NULL;	}

	~Texture()
		{	for(int i=0; i<MAX_MIPMAPLEVELS; i++)  if(mmLevels[i]) delete mmLevels[i];	}

	MipMapLevelPtr	mmLevels[MAX_MIPMAPLEVELS];
};


// Gradients are used for texturing only...
//
struct TexGradients {
	void set(const DeviceVertex *pVertices)
	{
		int Counter;

		enum { SSHIFT = 6 };				// allows us to move numbers into a good range for fixed point.
											// for highest precision we apply it back as late as possible (see below)

		klFloat	dx02 = (pVertices[0].pos[0] - pVertices[2].pos[0]);  dx02>>=SSHIFT;
		klFloat	dx12 = (pVertices[1].pos[0] - pVertices[2].pos[0]);  dx12>>=SSHIFT;
		klFloat	dy02 = (pVertices[0].pos[1] - pVertices[2].pos[1]);  dy02>>=SSHIFT;
		klFloat	dy12 = (pVertices[1].pos[1] - pVertices[2].pos[1]);  dy12>>=SSHIFT;
		klFloat OneOverdX = (dx12 * dy02) - (dx02 * dy12);
		OneOverdX.inverse();

		for(Counter = 0;Counter < 3;Counter++) {
			klFloat OneOverZ = pVertices[Counter].texZ;
			OneOverZ.inverse();

			aOneOverZ[Counter] = OneOverZ;
			aUOverZ[Counter] = pVertices[Counter].tex[0] * OneOverZ;
			aVOverZ[Counter] = pVertices[Counter].tex[1] * OneOverZ;
		}

		dOneOverZdX =  OneOverdX * (((aOneOverZ[1] - aOneOverZ[2]) * dy02) - ((aOneOverZ[0] - aOneOverZ[2]) * dy12));
		dOneOverZdX >>= SSHIFT;
		dOneOverZdY =  OneOverdX * (((aOneOverZ[0] - aOneOverZ[2]) * dx12) - ((aOneOverZ[1] - aOneOverZ[2]) * dx02));
		dOneOverZdY >>= SSHIFT;

		dUOverZdX =  OneOverdX * (((aUOverZ[1]  -  aUOverZ[2]) * dy02)  -  ((aUOverZ[0]  -  aUOverZ[2]) * dy12));
		dUOverZdX >>= SSHIFT;
		dUOverZdY =  OneOverdX * (((aUOverZ[0]  -  aUOverZ[2]) * dx12)  -  ((aUOverZ[1]  -  aUOverZ[2]) * dx02));
		dUOverZdY >>= SSHIFT;

		dVOverZdX =  OneOverdX * (((aVOverZ[1]  -  aVOverZ[2]) * dy02)  -  ((aVOverZ[0]  -  aVOverZ[2]) * dy12));
		dVOverZdX >>= SSHIFT;
		dVOverZdY =  OneOverdX * (((aVOverZ[0]  -  aVOverZ[2]) * dx12)  -  ((aVOverZ[1]  -  aVOverZ[2]) * dx02));
		dVOverZdY >>= SSHIFT;
	}


	klFloat	aOneOverZ[3];				// 1/z for each vertex
	klFloat	aUOverZ[3];					// u/z for each vertex
	klFloat	aVOverZ[3];					// v/z for each vertex
	klFloat	dOneOverZdX, dOneOverZdY,	// d(1/z)/dX, d(1/z)/dY
			dUOverZdX, dUOverZdY,		// d(u/z)/dX, d(u/z)/dY
			dVOverZdX, dVOverZdY;		// d(v/z)/dX, d(v/z)/dY
};


// tx_EDGE stores triangle edges plus all gradients and steps
//
struct Edge {
	Edge(const TexGradients *nGradients, const DeviceVertex *pVertices, int Top, int Bottom, int nEdgeOptions)
	{
		Y = ceil(pVertices[Top].pos[1]).getInt();
		int YEnd = ceil(pVertices[Bottom].pos[1]).getInt();
		Height = YEnd - Y;

		if(Height==0)
			return;

		klFloat XPrestep, YPrestep;
		YPrestep.setInt(Y);
		YPrestep -= pVertices[Top].pos[1];

		klFloat		RealWidth = pVertices[Bottom].pos[0] - pVertices[Top].pos[0],
					RealHeight = pVertices[Bottom].pos[1] - pVertices[Top].pos[1];

		klFloat		OneOverRealHeight = RealHeight;
		OneOverRealHeight.inverse();
		klFloat		YPrestepOverRealHeight = YPrestep*OneOverRealHeight;

		X = RealWidth * YPrestepOverRealHeight + pVertices[Top].pos[0];
		XStep = RealWidth*OneOverRealHeight;
		XPrestep = X - pVertices[Top].pos[0];

		if(nEdgeOptions&OPT_DEPTH)
		{
			klFloat RealDepth = pVertices[Bottom].pos[2] - pVertices[Top].pos[2];

			Z = RealDepth * YPrestepOverRealHeight + pVertices[Top].pos[2];
			ZStep = RealDepth*OneOverRealHeight;
		}
		else
		{
			Z.setInt(0);	ZStep.setInt(0);
		}

		if(nEdgeOptions&OPT_SHADING_RGB)
		{
			klFloat	RealR = pVertices[Bottom].col[0] - pVertices[Top].col[0],
					RealG = pVertices[Bottom].col[1] - pVertices[Top].col[1],
					RealB = pVertices[Bottom].col[2] - pVertices[Top].col[2];

			// TODO: should we skip the prestep for small triangles?
			//
			R = RealR * YPrestepOverRealHeight + pVertices[Top].col[0];
			RStep = RealR*OneOverRealHeight;

			G = RealG * YPrestepOverRealHeight + pVertices[Top].col[1];
			GStep = RealG*OneOverRealHeight;

			B = RealB * YPrestepOverRealHeight + pVertices[Top].col[2];
			BStep = RealB*OneOverRealHeight;
		}
		else
		{
			R.setInt(0);	RStep.setInt(0);
			G.setInt(0);	GStep.setInt(0);
			B.setInt(0);	BStep.setInt(0);
		}

		if(nEdgeOptions&OPT_SHADING_ALPHA)
		{
			klFloat	RealA = pVertices[Bottom].col[3] - pVertices[Top].col[3];

			A = RealA * YPrestepOverRealHeight + pVertices[Top].col[3];
			AStep = RealA*OneOverRealHeight;
		}
		else
		{
			A.setInt(255);	AStep.setInt(0);
		}

		if(nEdgeOptions&OPT_FOG)
		{
			klFloat	RealF = pVertices[Bottom].fog - pVertices[Top].fog;

			Fog = RealF * YPrestepOverRealHeight + pVertices[Top].fog;
			FogStep = RealF*OneOverRealHeight;
		}
		else
		{
			Fog.setInt(0);	FogStep.setInt(0);
		}


		if(nGradients)
		{
			OneOverZ = nGradients->aOneOverZ[Top]  +  YPrestep * nGradients->dOneOverZdY  +  XPrestep * nGradients->dOneOverZdX;
			OneOverZStep = XStep * nGradients->dOneOverZdX  +  nGradients->dOneOverZdY;

			UOverZ = nGradients->aUOverZ[Top]  +  YPrestep * nGradients->dUOverZdY  +  XPrestep * nGradients->dUOverZdX;
			UOverZStep = XStep * nGradients->dUOverZdX  +  nGradients->dUOverZdY;

			VOverZ = nGradients->aVOverZ[Top]  +  YPrestep * nGradients->dVOverZdY  +  XPrestep * nGradients->dVOverZdX;
			VOverZStep = XStep * nGradients->dVOverZdX  +  nGradients->dVOverZdY;
		}
		else
		if(nEdgeOptions&OPT_AFFINE_TEX)
		{
			klFloat	RealU = pVertices[Bottom].tex[0] - pVertices[Top].tex[0],
					RealV = pVertices[Bottom].tex[1] - pVertices[Top].tex[1];

			// skip this for speed (these are small triangles anyhow...)
			//
			U = /*RealU * YPrestepOverRealHeight +*/ pVertices[Top].tex[0];
			UStep = RealU*OneOverRealHeight;

			V = /*RealV * YPrestepOverRealHeight +*/ pVertices[Top].tex[1];
			VStep = RealV*OneOverRealHeight;
		}
	}


	inline int stepFlat()
	{
		Y++;
		Height--;
		X += XStep;
		Z += ZStep;
		Fog += FogStep;
		return Height;
	}

	inline int stepShade()
	{
		Y++;
		Height--;
		X += XStep;
		Z += ZStep;
		Fog += FogStep;
		R += RStep;
		G += GStep;
		B += BStep;
		A += AStep;
		return Height;
	}

	inline int stepTex()
	{
		Y++;
		Height--;
		X += XStep;
		Z += ZStep;
		Fog += FogStep;
		UOverZ += UOverZStep;
		VOverZ += VOverZStep;
		OneOverZ += OneOverZStep;
		return Height;
	}

	inline int stepAffineTex()
	{
		Y++; Height--;
		X += XStep;
		Z += ZStep;
		Fog += FogStep;
		U += UStep;
		V += VStep;
		return Height;
	}

	inline int stepTexShade()
	{
		Y++;
		Height--;
		X += XStep;
		Z += ZStep;
		Fog += FogStep;
		R += RStep;
		G += GStep;
		B += BStep;
		A += AStep;
		UOverZ += UOverZStep;
		VOverZ += VOverZStep;
		OneOverZ += OneOverZStep;
		return Height;
	}

	inline int stepAffineTexShade()
	{
		Y++;
		Height--;
		X += XStep;
		Z += ZStep;
		Fog += FogStep;
		R += RStep;
		G += GStep;
		B += BStep;
		A += AStep;
		U += UStep;
		V += VStep;
		return Height;
	}


	klFloat		X, XStep;					// fractional x and dX/dY
	klFloat		Z, ZStep;					// fractional x and dX/dY
	int			Y, Height;					// current y and vert count
	klFloat		OneOverZ, OneOverZStep,		// 1/z and step
				UOverZ, UOverZStep,			// u/z and step
				VOverZ, VOverZStep;			// v/z and step
	klFloat		R,G,B,A, RStep,GStep,BStep,AStep;
	klFloat		Fog, FogStep;

	klFloat		U,V, UStep,VStep;
};


//
//}  class klRSW565
//
