/**************************************************************************\
 *
 *  This file is part of the Klimt library.
 *  Copyright (C) 2003 by IMS, Vienna University of Technology.
 *  All rights reserved.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  ("GPL") version 2 as published by the Free Software Foundation.
 *  See the file LICENSE.GPL at the root directory of this source
 *  distribution for additional information about the GNU GPL.
 *  For the full GPL license see
 *  <URL:http://www.gnu.org/copyleft/gpl.html>
 *
 *  For using Klimt with software that can not be combined with the
 *  GNU GPL, and for taking advantage of the additional benefits of
 *  our support services, please contact IMS about acquiring a
 *  Klimt Professional Edition License.
 *
 *  Contact: <mailto:klimt@studierstube.org>
 *  See <URL:http://www.studierstube.org/klimt>
 *  for more information.
 *
 *  Vienna University of Technology
 *  Institute for Software Technology and Interactive Systems
 *  Interactive Media Systems Group
 *  Favoritenstrasse 9-11/188/2
 *  A-1040 Vienna, Austria
 *  <URL:http://www.ims.tuwien.ac.at>.
 *
 **************************************************************************
 *
 * $Header: /cvsroot/klimt/klimt/klimt/src/RasterizerSW565/klRSW565_Scanline_PixelWrite.h,v 1.2 2004/01/13 15:40:57 drgoldie Exp $
 *
\**************************************************************************/


//
// class klRSW565 {
//


// writes pixels into RGB565 video memory.
// simply copies pixels over. existing data is overwritten.
//
void drawScanLine_PixelWrite_One_Zero(const BUF_PIXELTYPE* nSrc, PIXELTYPE* nDst)
{
	for(int len,rlIdx=0;;)
	{
		// copy visible pixels
		//
		len = scanlineRunLengths[rlIdx++];

		while(len--)
			*(nDst++) = (PIXELTYPE)(*nSrc++);					// just overwrite existing pixels

		if(rlIdx>=numScanlineRunLengths)
			break;

		// skip invisible pixels (only in nDst, since
		// we didn't create them in nSrc anyhow...)
		len = scanlineRunLengths[rlIdx++];
		nDst += len;

		if(rlIdx>=numScanlineRunLengths)
			break;
	}
}


// writes pixels into RGB565 video memory.
// adds new pixels to existing pixels component-wise
//   with a constant factor of one.
//
void drawScanLine_PixelWrite_One_One(const BUF_PIXELTYPE* nSrc, PIXELTYPE* nDst)
{
	register unsigned short newCol;

	for(int len,rlIdx=0;;)
	{
		// copy visible pixels
		//
		len = scanlineRunLengths[rlIdx++];

		while(len--)
		{
			// this method allows us to combine all four color channels with a single
			// add operation. the trade-off is that we pollute the next component's LSB.
			//
			newCol = (unsigned short)( (((*nSrc)>>1) & RASTER_ADD_SHIFTMASK) + (((*nDst)>>1) & RASTER_ADD_SHIFTMASK) );

			if (newCol & 0x8410)						// check if any component has to be saturated
			{
				if(newCol&0x8000)    newCol|=0x7800;	// red
				if(newCol&0x0400)    newCol|=0x03E0;	// green
				if(newCol&0x0010)    newCol|=0x000F;	// blue
			}
			newCol <<= 1;
			*nDst++ = newCol;
			nSrc++;
		}

		if(rlIdx>=numScanlineRunLengths)
			break;

		// skip invisible pixels (only in nDst, since
		// we didn't create them in nSrc anyhow...)
		len = scanlineRunLengths[rlIdx++];
		nDst += len;

		if(rlIdx>=numScanlineRunLengths)
			break;
	}
}


// writes pixels into RGB565 video memory.
// new pixels are blended with existing pixels by using
//   newCol*SrcAlpha + oldCol*OneMinusSrcAlpha
//
void drawScanLine_PixelWrite_SrcAlpha_OneMinusSrcAlpha(const BUF_PIXELTYPE* nSrc, PIXELTYPE* nDst)
{
	register unsigned int src;

	for(int len,rlIdx=0;;)
	{
		// copy visible pixels
		//
		len = scanlineRunLengths[rlIdx++];

		while(len--)
		{
			src = *nSrc;
			*nDst = blendPixel16_6((unsigned short)src, (unsigned short)*nDst, src>>18);		// "src>>18" is our 6-bits alpha value 
			nSrc++;
			nDst++;
		}

		if(rlIdx>=numScanlineRunLengths)
			break;

		// skip invisible pixels (only in nDst, since
		// we didn't create them in nSrc anyhow...)
		len = scanlineRunLengths[rlIdx++];
		nDst += len;

		if(rlIdx>=numScanlineRunLengths)
			break;
	}
}


// does color modulation with zero
//
void drawScanLine_PixelWrite_Modulate_Zero(PIXELTYPE* nDst)
{
	for(int len,rlIdx=0;;)
	{
		// copy visible pixels
		//
		len = scanlineRunLengths[rlIdx++];

		while(len--)
			*(nDst++) = 0;

		if(rlIdx>=numScanlineRunLengths)
			break;

		// skip invisible pixels (only in nDst, since
		// we didn't create them in nSrc anyhow...)
		len = scanlineRunLengths[rlIdx++];
		nDst += len;

		if(rlIdx>=numScanlineRunLengths)
			break;
	}
}


// does color modulation with one
//
void drawScanLine_PixelWrite_Modulate_One(PIXELTYPE* nDst, const BUF_PIXELTYPE* nSrcCol)
{
	for(int len,rlIdx=0;;)
	{
		// copy visible pixels
		//
		len = scanlineRunLengths[rlIdx++];

		while(len--)
			*(nDst++) = (unsigned short)*(nSrcCol++);

		if(rlIdx>=numScanlineRunLengths)
			break;

		// skip invisible pixels (only in nDst, since
		// we didn't create them in nSrc anyhow...)
		len = scanlineRunLengths[rlIdx++];
		nDst += len;

		if(rlIdx>=numScanlineRunLengths)
			break;
	}
}


// does color modulation with color (version with 32-bits nSrcFact)
//
template <class COLTYPE, class FACTYPE>
void drawScanLine_PixelWrite_Modulate_Color(PIXELTYPE* nDst,
										    const COLTYPE* nSrcCol, const FACTYPE* nSrcFact)
{
	register unsigned int c0,c1,tmp;

	for(int len,rlIdx=0;;)
	{
		// copy visible pixels
		//
		len = scanlineRunLengths[rlIdx++];

		while(len--)
		{
			c0 = *nSrcCol++;
			c1 = *nSrcFact++;

			tmp = (c0&REDBLUE_MASK) * (c1&REDBLUE_MASK);
			*(nDst++) =	(unsigned short)(((tmp>>16)&RED_MASK) | 
										 ((((c0&GREEN_MASK) * (c1&GREEN_MASK))>>11)&GREEN_MASK) |
										 ((tmp>>5)&BLUE_MASK)
										);
		}

		if(rlIdx>=numScanlineRunLengths)
			break;

		// skip invisible pixels (only in nDst, since
		// we didn't create them in nSrc anyhow...)
		len = scanlineRunLengths[rlIdx++];
		nDst += len;
		if(sizeof(COLTYPE)==2)		// only skip it if we are reading from video memory
			nSrcCol += len;
		if(sizeof(FACTYPE)==2)		// only skip it if we are reading from video memory
			nSrcFact += len;

		if(rlIdx>=numScanlineRunLengths)
			break;
	}
}


// does color modulation with one minus color (version with 32-bits nSrcFact)
//
template <class COLTYPE, class FACTYPE>
void drawScanLine_PixelWrite_Modulate_OneMinusColor(PIXELTYPE* nDst,
													const COLTYPE* nSrcCol, const FACTYPE* nSrcFact)
{
	register unsigned int c0,c1,tmp;

	for(int len,rlIdx=0;;)
	{
		// copy visible pixels
		//
		len = scanlineRunLengths[rlIdx++];

		while(len--)
		{
			c0 = *nSrcCol++;
			c1 = ~(*nSrcFact++);			// do one-minus the quick way...

			tmp = (c0&REDBLUE_MASK) * (c1&REDBLUE_MASK);
			*(nDst++) =	((tmp>>16)&RED_MASK) | 
						((((c0&GREEN_MASK) * (c1&GREEN_MASK))>>11)&GREEN_MASK) |
						((tmp>>5)&BLUE_MASK);
		}

		if(rlIdx>=numScanlineRunLengths)
			break;

		// skip invisible pixels (only in nDst, since
		// we didn't create them in nSrc anyhow...)
		len = scanlineRunLengths[rlIdx++];
		nDst += len;
		if(sizeof(COLTYPE)==2)		// only skip it if we are reading from video memory
			nSrcCol += len;
		if(sizeof(FACTYPE)==2)		// only skip it if we are reading from video memory
			nSrcFact += len;

		if(rlIdx>=numScanlineRunLengths)
			break;
	}
}


// does color modulation with alpha
//
template <class COLTYPE, class FACTYPE>
void drawScanLine_PixelWrite_Modulate_Alpha(PIXELTYPE* nDst,
											const COLTYPE* nSrcCol, const FACTYPE* nSrcFact)
{
	register unsigned int c0;
	register unsigned short op,rb,g;

	for(int len,rlIdx=0;;)
	{
		// copy visible pixels
		//
		len = scanlineRunLengths[rlIdx++];

		while(len--)
		{
			c0 = *nSrcCol++;
			op = (*nSrcFact++) >> 18;

			rb = (unsigned short)( ((c0&REDBLUE_MASK) * (op>>1)) >> 5 );
			g  = (unsigned short)( ((c0&GREEN_MASK) * op) >> 6 );

			*(nDst++) = (rb&REDBLUE_MASK) | (g&GREEN_MASK);
		}

		if(rlIdx>=numScanlineRunLengths)
			break;

		// skip invisible pixels (only in nDst, since
		// we didn't create them in nSrc anyhow...)
		len = scanlineRunLengths[rlIdx++];
		nDst += len;
		if(sizeof(COLTYPE)==2)		// only skip it if we are reading from video memory
			nSrcCol += len;
		if(sizeof(FACTYPE)==2)		// only skip it if we are reading from video memory
			nSrcFact += len;

		if(rlIdx>=numScanlineRunLengths)
			break;
	}
}


// does color modulation with one minus alpha
//
template <class COLTYPE, class FACTYPE>
void drawScanLine_PixelWrite_Modulate_OneMinusAlpha(PIXELTYPE* nDst,
													const COLTYPE* nSrcCol, const FACTYPE* nSrcFact)
{
	register unsigned int c0;
	register unsigned short op,rb,g;


	for(int len,rlIdx=0;;)
	{
		// copy visible pixels
		//
		len = scanlineRunLengths[rlIdx++];

		while(len--)
		{
			c0 = *nSrcCol++;
			op = 63-((*nSrcFact++) >> 18);

			rb = (unsigned short)( ((c0&REDBLUE_MASK) * (op>>1)) >> 5 );
			g  = (unsigned short)( ((c0&GREEN_MASK) * op) >> 6 );

			*(nDst++) = (rb&REDBLUE_MASK) | (g&GREEN_MASK);
		}

		if(rlIdx>=numScanlineRunLengths)
			break;

		// skip invisible pixels (only in nDst, since
		// we didn't create them in nSrc anyhow...)
		len = scanlineRunLengths[rlIdx++];
		nDst += len;
		if(sizeof(COLTYPE)==2)		// only skip it if we are reading from video memory
			nSrcCol += len;
		if(sizeof(FACTYPE)==2)		// only skip it if we are reading from video memory
			nSrcFact += len;

		if(rlIdx>=numScanlineRunLengths)
			break;
	}
}


void drawScanLine_BlitRGB(PIXELTYPE* nSrc, PIXELTYPE* nDst)
{
	register unsigned short newCol;

	for(int len,rlIdx=0;;)
	{
		// copy visible pixels
		//
		len = scanlineRunLengths[rlIdx++];

		while(len--)
		{
			// this method allows us to combine all four color channels with a single
			// add operation. the trade-off is that we pollute the next component's LSB.
			//
			newCol = (unsigned short)( (((*nSrc)>>1) & RASTER_ADD_SHIFTMASK) + (((*nDst)>>1) & RASTER_ADD_SHIFTMASK) );

			if (newCol & 0x8410)						// check if any component has to be saturated
			{
				if(newCol&0x8000)    newCol|=0x7800;	// red
				if(newCol&0x0400)    newCol|=0x03E0;	// green
				if(newCol&0x0010)    newCol|=0x000F;	// blue
			}
			newCol <<= 1;

			*nDst++ = newCol;
			nSrc++;
		}

		if(rlIdx>=numScanlineRunLengths)
			break;

		// skip invisible pixels (only in nDst, since
		// we didn't create them in nSrc anyhow...)
		len = scanlineRunLengths[rlIdx++];
		nDst += len;
		nSrc += len;

		if(rlIdx>=numScanlineRunLengths)
			break;
	}
}


void drawScanLine_BlitMasked(PIXELTYPE* nSrc, PIXELTYPE* nDst)
{
	register unsigned short newCol,oldCol,rgbMask=0,rgbMaskInv;

	if(settings.colorMask&COLOR_MASK_RED)		rgbMask |= RED_MASK;
	if(settings.colorMask&COLOR_MASK_GREEN)		rgbMask |= GREEN_MASK;
	if(settings.colorMask&COLOR_MASK_BLUE)		rgbMask |= BLUE_MASK;
	rgbMaskInv = (unsigned short)~rgbMask;

	for(int len,rlIdx=0;;)
	{
		// copy visible pixels
		//
		len = scanlineRunLengths[rlIdx++];

		while(len--)
		{
			// this method allows us to combine all four color channels with a single
			// add operation. the trade-off is that we pollute the next component's LSB.
			//
			oldCol = *nDst;
			newCol = (unsigned short)( (((*nSrc)>>1) & RASTER_ADD_SHIFTMASK) + (((oldCol)>>1) & RASTER_ADD_SHIFTMASK) );

			if (newCol & 0x8410)						// check if any component has to be saturated
			{
				if(newCol&0x8000)    newCol|=0x7800;	// red
				if(newCol&0x0400)    newCol|=0x03E0;	// green
				if(newCol&0x0010)    newCol|=0x000F;	// blue
			}
			newCol <<= 1;

			*nDst++ = (unsigned short)((newCol&rgbMask) | (oldCol&rgbMaskInv));
			nSrc++;
		}

		if(rlIdx>=numScanlineRunLengths)
			break;

		// skip invisible pixels (only in nDst, since
		// we didn't create them in nSrc anyhow...)
		len = scanlineRunLengths[rlIdx++];
		nDst += len;
		nSrc += len;

		if(rlIdx>=numScanlineRunLengths)
			break;
	}
}


void drawScanLine_PixelWrite(int nSrcBlend, int nDstBlend, BUF_PIXELTYPE* nSrc, PIXELTYPE* nDst)
{
	// handle some common cases directy with a specialized function
	//
	if(pixelWriteFunc)
	{
		(this->*pixelWriteFunc)(nSrc, nDst);
		return;
	}
	// if we did not succed to do the blending with a
	// specialized function we have to go the hard way...


	// first modulate source
	//
	switch(nSrcBlend)
	{
	case GL_ZERO:
		drawScanLine_PixelWrite_Modulate_Zero(scanLineSrcBlend);
		break;

	case GL_ONE:
		drawScanLine_PixelWrite_Modulate_One(scanLineSrcBlend, nSrc);
		break;

	case GL_SRC_COLOR:
		drawScanLine_PixelWrite_Modulate_Color(scanLineSrcBlend, nSrc, nSrc);
		break;

	case GL_ONE_MINUS_SRC_COLOR:
		drawScanLine_PixelWrite_Modulate_OneMinusColor(scanLineSrcBlend, nSrc, nSrc);
		break;

	case GL_DST_COLOR:
		drawScanLine_PixelWrite_Modulate_Color(scanLineSrcBlend, nSrc, nDst);
		break;

	case GL_ONE_MINUS_DST_COLOR:
		drawScanLine_PixelWrite_Modulate_OneMinusColor(scanLineSrcBlend, nSrc, nDst);
		break;

	case GL_SRC_ALPHA:
		drawScanLine_PixelWrite_Modulate_Alpha(scanLineSrcBlend, nSrc, nSrc);
		break;

	case GL_ONE_MINUS_SRC_ALPHA:
		drawScanLine_PixelWrite_Modulate_OneMinusAlpha(scanLineSrcBlend, nSrc, nSrc);
		break;
	}


	// next modulate destination
	//
	switch(nDstBlend)
	{
	case GL_ZERO:
		drawScanLine_PixelWrite_Modulate_Zero(nDst);
		break;

	case GL_ONE:
		// nothing to do...
		break;

	case GL_SRC_COLOR:
		drawScanLine_PixelWrite_Modulate_Color(nDst, nDst, nSrc);
		break;

	case GL_ONE_MINUS_SRC_COLOR:
		drawScanLine_PixelWrite_Modulate_OneMinusColor(nDst, nDst, nSrc);
		break;

	case GL_DST_COLOR:
		drawScanLine_PixelWrite_Modulate_Color(nDst, nDst, nDst);
		break;

	case GL_ONE_MINUS_DST_COLOR:
		drawScanLine_PixelWrite_Modulate_OneMinusColor(nDst, nDst, nDst);
		break;

	case GL_SRC_ALPHA:
		drawScanLine_PixelWrite_Modulate_Alpha(nDst, nDst, nSrc);
		break;

	case GL_ONE_MINUS_SRC_ALPHA:
		drawScanLine_PixelWrite_Modulate_OneMinusAlpha(nDst, nDst, nSrc);
		break;
	}

	// finally sum the two buffers doing saturation...
	//
	if((settings.colorMask&COLOR_MASK_RGB)==COLOR_MASK_RGB)		// write into all channels? (RGB only at the moment)
		drawScanLine_BlitRGB(scanLineSrcBlend, nDst);
	else
		drawScanLine_BlitMasked(scanLineSrcBlend, nDst);
}


//
// } class klRSW565
//
