 /**************************************************************************\
 *
 *  This file is part of the Klimt library.
 *  Copyright (C) 2003 by IMS, Vienna University of Technology.
 *  All rights reserved.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  ("GPL") version 2 as published by the Free Software Foundation.
 *  See the file LICENSE.GPL at the root directory of this source
 *  distribution for additional information about the GNU GPL.
 *  For the full GPL license see
 *  <URL:http://www.gnu.org/copyleft/gpl.html>
 *
 *  For using Klimt with software that can not be combined with the
 *  GNU GPL, and for taking advantage of the additional benefits of
 *  our support services, please contact IMS about acquiring a
 *  Klimt Professional Edition License.
 *
 *  Contact: <mailto:klimt@studierstube.org>
 *  See <URL:http://www.studierstube.org/klimt>
 *  for more information.
 *
 *  Vienna University of Technology
 *  Institute for Software Technology and Interactive Systems
 *  Interactive Media Systems Group
 *  Favoritenstrasse 9-11/188/2
 *  A-1040 Vienna, Austria
 *  <URL:http://www.ims.tuwien.ac.at>.
 *
 **************************************************************************
 *
 * $Header: /cvsroot/klimt/klimt/klimt/src/klContext.h,v 1.6 2004/02/10 18:19:30 drgoldie Exp $
 *
\**************************************************************************/


#ifndef __KLCONTEXT_HEADERFILE__
#define __KLCONTEXT_HEADERFILE__


#include "klImageConverter.h"
#include "klDisplayList.h"
#include "klDisplayList_Record.h"


// The main Klimt context. The whole class is written
// as inline code. Since this class has a very large
// interface, all methods are stored in seperate
// header files (see include statements below)
//
class klContext
{
public:
	enum
	{
		MAX_MODELVIEW_STACK = 32,
		MAX_PROJECTION_STACK = 8,
		MAX_TEXTURE_STACK = 8,

		DISPLAYLIST_MAXHASH = 32,

		MAX_ATTRIBUTE_STACK = 32,
		MAX_CLIENTATTRIBUTE_STACK = 32,

		MAX_PRIMITIVE_LENGTH = 128,
		MAX_LIGHTS = 8,
	};


protected:
	// internal temporary vertex storage
	//
	klVertex	tmpVertices[MAX_PRIMITIVE_LENGTH+4],		// vertex storage for temporal use
				clipTmpVertices[MAX_PRIMITIVE_LENGTH+4];	// (max. until jumping back to application code)


	// matrix stacks
	//
	klStack<klMatrix>		modelviewStack,				// modelview matrix stack
							projectionStack,			// projection matrix stack
							textureStack,				// texture matrix stack
							*curStack;					// currently selected statck
	bool					modelProjMatrixDirty;		// dirty flag showing modelProjMatrix needs to be updated
	klMatrix				modelProjMatrix,			// this stores the combined modelview*projection matrix
							modelProjMatrixInvTrans;	// this stores the inverted & transposed of modelProjMatrix


	// enable bits
	//
	bool		lightingEnabled,			// global lighting on/off switch
				cullingEnabled,				// global culling on/off switch
				texturingEnabled,			// global texturing on/off switch
				depthWriteEnabled,			// global z-value writing on/off switch
				depthTestEnabled,			// global z-test on/off switch
				smoothShadingEnabled,		// global smooth shading (gouraud interpolation) on/off switch
				colorMaterialEnabled,		// global color material on/off switch
				normalizeNormalsEnabled;	// global normals normalizing on/off switch

	RASTER_MODE_ZTEST	depthFunc;							// z-test current function
	GLenum				textureApplyMode;					// how do we apply the texture to the current fragment
	GLenum				cullMode;							// whether to cull front, back or both
	GLenum				frontPolyMode, backPolyMode;		// point,line or fill mode for front/back side
	GLenum				frontFace;							// sets whether CW or CCW is the front face side
	unsigned int		colorMask;							// stores which color channels will be written to
	klColor4			clearColor;							// stores the color used for clearing the color buffer
	klFloat				clearDepth;							// stores the value used for clearing the depth buffer
	int					largestActiveLightIdx;				// largest index of an active light
	bool				isFrontSideVisible;					// true if the front side of the polygon is visible

	// fog
	//
	struct FOG {
		FOG() : color(0.0f, 0.0f, 0.0f, 0.0f)
			{  start = 0.0f;  end = 1.0f;  enabled = false;  }

		klColor4	color;
		klFloat		start, end,
					invDFog;								// needed internally
		bool		enabled;

		void update()
		{
			if(end-start<=0)
				end = start + 0.001f;
			invDFog = end - start;
			invDFog.inverse();
		}
	} fog;

	
	// front & back materials
	//
	struct MATERIAL {
		MATERIAL () : ambient(0.2f,0.2f,0.2f,1.0f), diffuse(0.8f,0.8f,0.8f,1.0f),
					  specular(0.0f,0.0f,0.0f,1.0f), emissive(0.0f,0.0f,0.0f,1.0f)
			{ shininess = 0.0f;  colorMaterialMode = GL_AMBIENT_AND_DIFFUSE; polyMode = GL_FILL;  }

		klColor4	ambient, diffuse, specular, emissive;
		klFloat		shininess;
		GLenum		colorMaterialMode;						// if color material is enabled this defines which mode to use
		GLenum		polyMode;								// polygon mode (FILL, LINE, POINT)
		int			idx;
	} materialFront, materialBack;


	// lights
	//
	struct LIGHT {
		LIGHT() : ambient(0.0f,0.0f,0.0f,1.0f), diffuse(0.0f,0.0f,0.0f,0.0f), specular(0.0f,0.0f,0.0f,0.0f),
				  position(0.0f,0.0f,0.0f,1.0f)
			{  enabled = false;  }

		klColor4	ambient, diffuse, specular;
		klVec4		position;
		bool		enabled;
		int			idx;
	} lights[MAX_LIGHTS];


	// lighting model
	//
	struct LIGHTMODEL {
		LIGHTMODEL() : ambient(0.2f, 0.2f, 0.2f, 1.0f) { localViewer = true;  twoSided = false;  }

		klColor4	ambient;							// global ambient light
		bool		localViewer,
					twoSided;							// TODO: twosided materials not supported yet
	} lightModel;


	// this structure caches lighting attributes
	//
	struct LIGHTINGCACHE {
		void updateModelAmbient(const MATERIAL& nMaterial, const LIGHTMODEL& nLightModel);
		void updateLightAmbient(const MATERIAL& nMaterial, const LIGHT& nLight);
		void updateLightAmbient(const MATERIAL& nMaterial, const LIGHT* nLights);
		void updateLightDiffuse(const MATERIAL& nMaterial, const LIGHT& nLight);
		void updateLightDiffuse(const MATERIAL& nMaterial, const LIGHT* nLights);
		void updateLightSpecular(const MATERIAL& nMaterial, const LIGHT& nLight);
		void updateLightSpecular(const MATERIAL& nMaterial, const LIGHT* nLights);

		klColor4	modelAmbient[2],
					lightAmbient[MAX_LIGHTS][2],
					lightDiffuse[MAX_LIGHTS][2],
					lightSpecular[MAX_LIGHTS][2];
	} lightingCache;

	// scissor test
	//
	struct SCISSOR {
		GLint		x,y;
		GLsizei		width, height;
		bool		enabled;
	} scissor;


	// blending settings
	//
	struct BLEND {
		GLenum		src, dst;							// source and destination blending factors
		bool		enabled;							// blending on/off switch
	} blend;


	// vertex arrays
	//
	klArray	arrayVertex,							// stores all infos about vertex array client data
			arrayNormal,							// stores all infos about normal array client data
			arrayColor,								// stores all infos about color array client data
			arrayTexCoord;							// stores all infos about texture coordinate array client data


	// opengl begin/end stuff
	//
	struct BEGINEND_STATE {
		BEGINEND_STATE() : currentColor(1.0f, 1.0f, 1.0f, 1.0f)
			{  numVertices = 0;  primType=GL_NONE;  }
		
		klVec4		vertices[MAX_PRIMITIVE_LENGTH],
					texCoords[MAX_PRIMITIVE_LENGTH], currentTexCoord;
		klVec3		normals[MAX_PRIMITIVE_LENGTH], currentNormal;
		klColor4	colors[MAX_PRIMITIVE_LENGTH], currentColor;
		int			numVertices;
		GLenum		primType;

		void fillVertex()								// creates a new vertex from currently set values
		{
			if(primType==GL_NONE)
				return;
			texCoords[numVertices] = currentTexCoord;
			normals[numVertices] = currentNormal;
			colors[numVertices] = currentColor;
			numVertices++;
		}
	} beginendState;


	// some often needed constants
	//
	struct CONSTANTS {
		CONSTANTS() : one(1.0f), zero(0.0f)
			{}

		klFloat one, zero;
	} constant;


	// tweak settings
	//
	struct TWEAK {
		bool	textureMatrixEnabled,			// allows to turn off texture matrix usage
				doSpecularLighting,				// allows to turn off speculuar light calculation
				allowNormalization,				// false turns off normalization of normals, even if enabled with glEnable()
				forceFlatShading,				// if true always flat shading will be used...
				forceCulling;					// if true culling will be done, even if it is disabled with glDisable()
	} tweak;


	// attribute stack
	//
	#include "Base/klAttributes.h"

	klStack<klAttributes>		attributeStack;
	klStack<klClientAttributes>	clientAttributeStack;


	// rasterizer instance
	//
	klRasterizer* rasterizer;


	// display lists stuff
	//
	struct DISPLAYLIST {
		klPtrList<klDisplayList>	lists;
		klDisplayList*				recording;
		bool						executeCommands;		// if false, commands are not executed but only recorded
		int							listBase,
									listIdx;
		klPtrList<klDisplayList>	nameHashTable[DISPLAYLIST_MAXHASH];
	} displaylist;


public:
	// internal methods
	//
	#include "klContext_Init.h"
	#include "klContext_PrimitiveAssembler.h"
	#include "klContext_Clipping.h"


	// OpenGL(|ES) API methods
	//
	#include "klContext_oglArray.h"
	#include "klContext_oglAttributeStack.h"
	#include "klContext_oglBeginEnd.h"
	#include "klContext_oglDisplayList.h"
	#include "klContext_oglLight.h"
	#include "klContext_oglMaterial.h"
	#include "klContext_oglMatrix.h"
	#include "klContext_oglPixels.h"
	#include "klContext_oglSetup.h"
	#include "klContext_oglTexture.h"


	// Klimt API methods
	//
	#include "klContext_kl.h"
};


#include "klContext_Internal.h"
#include "klDisplayList_Commands.h"


#endif //__KLCONTEXT_HEADERFILE__
