/**************************************************************************\
 *
 *  This file is part of the Klimt library.
 *  Copyright (C) 2003 by IMS, Vienna University of Technology.
 *  All rights reserved.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  ("GPL") version 2 as published by the Free Software Foundation.
 *  See the file LICENSE.GPL at the root directory of this source
 *  distribution for additional information about the GNU GPL.
 *  For the full GPL license see
 *  <URL:http://www.gnu.org/copyleft/gpl.html>
 *
 *  For using Klimt with software that can not be combined with the
 *  GNU GPL, and for taking advantage of the additional benefits of
 *  our support services, please contact IMS about acquiring a
 *  Klimt Professional Edition License.
 *
 *  Contact: <mailto:klimt@studierstube.org>
 *  See <URL:http://www.studierstube.org/klimt>
 *  for more information.
 *
 *  Vienna University of Technology
 *  Institute for Software Technology and Interactive Systems
 *  Interactive Media Systems Group
 *  Favoritenstrasse 9-11/188/2
 *  A-1040 Vienna, Austria
 *  <URL:http://www.ims.tuwien.ac.at>.
 *
 **************************************************************************
 *
 * $Header: /cvsroot/klimt/klimt/klimt/src/klContext_PrimitiveAssembler.h,v 1.8 2004/02/11 14:15:50 drgoldie Exp $
 *
\**************************************************************************/



/**
class klContext
{
*/


//
// This file contains all methods that create
// primitives from the current state settings
// and pass them on to the rasterizer
//

bool
checkAllCulledAway(GLenum nPrimType)
{
	return cullingEnabled && cullMode==GL_FRONT_AND_BACK &&
		   nPrimType!=GL_POINTS && nPrimType!=GL_LINES && nPrimType!=GL_LINE_STRIP && nPrimType!=GL_LINE_LOOP;
}


void
checkModelProjMatrix()
{
	if(modelProjMatrixDirty)
	{
		modelProjMatrix.multiply(projectionStack.getTop(), modelviewStack.getTop());
		modelProjMatrixDirty = false;

		//
		// TODO: we should get rid of this in future
		//       (currently needed to transform normals, but
		//		  lights could be transformed instead of normals)
		//
		modelProjMatrixInvTrans.invert(modelviewStack.getTop());
		modelProjMatrixInvTrans.transpose();

		// modify the matrix not to change the normals lengths...
		//
		//
		// TODO: a generally better way to transform the normals should be found
		//
		klVec3 vUnit(1.0f, 0.0f, 0.0f), vTransformed;
		
		vTransformed.transform(modelProjMatrixInvTrans, vUnit);
		klFloat len = vTransformed.getLength();

		if(len<0.95f || len>1.05f)
		{
			klMatrix mat;

			mat.makeIdent();
			len.inverse();

			mat[0] = mat[5] = mat[10] = len;
			modelProjMatrixInvTrans *= mat;
		}
	}
}


void
renderArray(GLenum nPrimType, GLsizei nCount, GLenum nIndexType, const GLvoid* nIndices)
{
	if(checkAllCulledAway(nPrimType))
		return;

	checkModelProjMatrix();
	configureRasterizer();

	GLsizei	i=0, p, primLen = getPrimitiveLength(nPrimType, nCount);
	klVec3	vec3, vec3_;
	klVec4	position;
	klArrayFetcher positionFetcher(arrayVertex),			// although the array allows direct access to its data,
					normalFetcher(arrayNormal),				// using the fetcher classes is a lot faster for
					colorFetcher(arrayColor),				// multiple accesses...
					texCoordFetcher(arrayTexCoord);

	while(i<nCount)
	{
		for(p=0; p<primLen; p++)
		{
			int		idx = getIndex(nIndexType, i++, nIndices);

			// generate vertex position
			//
			positionFetcher.getVec4(position, idx);
			tmpVertices[p].pos.transform(modelProjMatrix, position);

			// generate vertex color
			//
			if(lightingEnabled)
			{
				if(arrayNormal.enabled)
					normalFetcher.getVec3(vec3_, idx);
				else
					vec3_ = beginendState.currentNormal;					// this is not due to the OGLES specs, but in OGL its ok...

				vec3.transform(modelProjMatrixInvTrans, vec3_);

				if(smoothShadingEnabled || p==0)
					calculateVertexLighting(tmpVertices[p], vec3, true);	// TODO: check which side is visible to implement double sided lighting (this means to postpone lighting until we transformed at least 3 vertices....)
				else
					tmpVertices[p].col = tmpVertices[0].col;
			}
			else
			{
				if(arrayColor.enabled)
					colorFetcher.getVec4(tmpVertices[p].col, idx);
				else
					tmpVertices[p].col = beginendState.currentColor;		// this is not due to the OGLES specs, but in OGL its ok...
			}

			// generate texture coordinate
			//
			if(texturingEnabled)
			{
				klVec4 texCoord;

				if(arrayTexCoord.enabled)
					texCoordFetcher.getVec4(texCoord, idx);
				else
					texCoord = beginendState.currentTexCoord;				// this is not due to the OGLES specs, but in OGL its ok...

				if(tweak.textureMatrixEnabled)
					tmpVertices[p].texCoord.transform(textureStack.getTop(), texCoord);
				else
					tmpVertices[p].texCoord = texCoord;
			}

			// calculate fog density
			//
			if(fog.enabled)
			{
				klFloat eyeZ = -klVec4::transformZ(modelviewStack.getTop(), position);
				klFloat fogf = (fog.end - eyeZ) * fog.invDFog;
				if(fogf>constant.one)
					fogf = constant.one;
				if(fogf<constant.zero)
					fogf = constant.zero;
				tmpVertices[p].fogDensity = fogf;
			}
		}

		renderPrimitive(nPrimType, tmpVertices, primLen);
	}
}


void
renderBeginEndPrimitive()
{
	if(checkAllCulledAway(beginendState.primType))
		return;

	//checkModelProjMatrix();
	//configureRasterizer();

	GLsizei	idx=0, p, primLen = getPrimitiveLength(beginendState.primType, beginendState.numVertices);
	klVec3 vec3;

	while(idx<beginendState.numVertices)
	{
		for(p=0; p<primLen; p++,idx++)
		{
			// generate vertex position
			//
			tmpVertices[p].pos.transform(modelProjMatrix, beginendState.vertices[idx]);

			// generate vertex color
			//
			if(lightingEnabled)
			{
				if(smoothShadingEnabled || p==0)
				{
					vec3.transform(modelProjMatrixInvTrans, beginendState.normals[idx]);
					calculateVertexLighting(tmpVertices[p], vec3, true);	// TODO: check which side is visible to implement double sided lighting (this means to postpone lighting until we transformed at least 3 vertices....)
				}
				else
					tmpVertices[p].col = tmpVertices[0].col;
			}
			else
				tmpVertices[p].col = beginendState.colors[idx];

			// generate texture coordinate
			//
			if(texturingEnabled)											// TODO: check for cases where lighting is not used (wireframe, lines, etc...)
			{
				if(tweak.textureMatrixEnabled)
					tmpVertices[p].texCoord.transform(textureStack.getTop(), beginendState.texCoords[idx]);
				else
					tmpVertices[p].texCoord = beginendState.texCoords[idx];
			}

			// calculate fog density
			//
			if(fog.enabled)
			{
				klFloat eyeZ = -klVec4::transformZ(modelviewStack.getTop(), beginendState.vertices[idx]);
				klFloat fogf = (fog.end - eyeZ) * fog.invDFog;
				if(fogf>constant.one)
					fogf = constant.one;
				if(fogf<constant.zero)
					fogf = constant.zero;
				tmpVertices[p].fogDensity = fogf;
			}
		}

		renderPrimitive(beginendState.primType, tmpVertices, primLen);
	}

	beginendState.numVertices = 0;
}


void
renderPrimitive(GLenum nPrimType, klVertex* nVertices, GLsizei nPrimLen)
{
	klVertex tmpV[9];		// temporal vertex storage for strips and fans
	int i;

	switch(nPrimType)
	{
	case GL_POINTS:
	case GL_LINE_STRIP:
	case GL_LINE_LOOP:
	case GL_LINES:
		clipPrimitve(nVertices, nPrimLen);
		if(nPrimLen==0)
			return;

		divideByW(nVertices, nPrimLen);

		rasterizer->renderPrimitive(nPrimType, nVertices, nPrimLen);
		break;

	case GL_TRIANGLES:
	case GL_QUADS:
	case GL_POLYGON:
		clipPrimitve(nVertices, nPrimLen);
		if(nPrimLen==0)
			return;

		divideByW(nVertices, nPrimLen);
		if(!checkCulling(nVertices, nPrimLen))
			return;

		rasterizer->renderPrimitive(nPrimType, nVertices, nPrimLen);
		break;

	case GL_TRIANGLE_FAN:
		for(i=1; i<nPrimLen-1; i++)						// for triangle fans we have clip each triangle on its own
		{
			int num = 3;
			tmpV[0] = nVertices[0];
			tmpV[1] = nVertices[i];
			tmpV[2] = nVertices[i+1];

			clipPrimitve(tmpV, num);
			if(num<3)
				continue;

			divideByW(tmpV, num);
			if(!checkCulling(tmpV, num))
				continue;

			rasterizer->renderPrimitive(nPrimType, tmpV, num);
		}
		break;

	case GL_QUAD_STRIP:
		for(i=0; i<nPrimLen-2; i+=2)						// for quad strips we have clip each quad on its own
		{
			int num = 4;
			tmpV[0] = nVertices[i];
			tmpV[1] = nVertices[i+1];
			tmpV[2] = nVertices[i+3];
			tmpV[3] = nVertices[i+2];

			clipPrimitve(tmpV, num);
			if(num<3)
				continue;

			divideByW(tmpV, num);
			if(!checkCulling(tmpV, num))
				continue;

			rasterizer->renderPrimitive(nPrimType, tmpV, num);
		}
		break;

	case GL_TRIANGLE_STRIP:
		for(i=0; i<nPrimLen-2; i++)						// for quad strips we have clip each quad on its own
		{
			int num = 3, d = i&1;
			tmpV[0] = nVertices[i];
			tmpV[1] = nVertices[i+1+d];			// even tris are normal, odd tris are reversed
			tmpV[2] = nVertices[i+2-d];

			clipPrimitve(tmpV, num);
			divideByW(tmpV, num);
			if(!checkCulling(tmpV, num))
				continue;

			rasterizer->renderPrimitive(nPrimType, tmpV, num);
		}
		break;
	}
}


void
divideByW(klVertex* nVertices, GLsizei nNum)
{
	for(int i=0; i<nNum; i++)
	{
		nVertices[i].unprojectedZ = nVertices[i].pos[2];
		nVertices[i].pos.divideByW();
	}
}


bool
checkCulling(const klVertex* nVertices, int nNum)
{
	if(nNum<3 || !cullingEnabled)
		return true;

	int idx0=0,idx1,idx2;
	klFloat	dx10,dy10, dx20,dy20, z,zpos,zmin;
	zmin.setFixed(6554);		// 0.1f

	do
	{
		idx1 = idx0+1;	if(idx1>=nNum)	idx1-=nNum;
		idx2 = idx0+2;	if(idx2>=nNum)	idx2-=nNum;
		
		dx10 = nVertices[idx1].pos[0] - nVertices[idx0].pos[0];
		dy10 = nVertices[idx1].pos[1] - nVertices[idx0].pos[1];
		dx20 = nVertices[idx2].pos[0] - nVertices[idx0].pos[0];
		dy20 = nVertices[idx2].pos[1] - nVertices[idx0].pos[1];

		// these are normalized coordinates [-1.0 to 1.0]
		// so we scale them up to make them more rubust for fixed point
		dx10 <<= 6;		dy10 <<= 6;
		dx20 <<= 6;		dy20 <<= 6;

		klFloat z1 = dx10*dy20,
				z2 = dy10*dx20;
		z = dx10*dy20 - dy10*dx20;
		zpos = z<0 ? -z : z;

		idx0++;
	} while(zpos<zmin && idx0<nNum-1);

	isFrontSideVisible = (frontFace==GL_CCW) ? (z>0) : (z<0);

	if(isFrontSideVisible)
		return cullMode==GL_BACK;
	else
		return cullMode==GL_FRONT;
}


void
configureRasterizer()
{
	const MATERIAL& mat = materialFront;		// TODO: need to use background material here if two-sided...
	klRasterizer_Settings& set = rasterizer->getSettings();

	set.ztest = depthTestEnabled ? depthFunc : RASTER_ZTEST_ALWAYS;
	set.zwrite = depthWriteEnabled ? RASTER_ZWRITE_ON : RASTER_ZWRITE_OFF;
	set.colorMask = colorMask;
	set.numtexstages = texturingEnabled ? 1 : 0;
	set.texenv = (textureApplyMode==GL_MODULATE) ? RASTER_TEXENV_MODULATE : RASTER_TEXENV_DECAL;
	set.fog = fog.enabled ? RASTER_FOG_ON : RASTER_FOG_OFF;
	set.blendEnabled = blend.enabled;
	set.blendSrc = blend.src;
	set.blendDst = blend.dst;

	switch(mat.polyMode)
	{
	case GL_POINT:
		set.polyMode = RASTER_POINTS;
		break;
	case GL_LINE:
		set.polyMode = RASTER_LINES;
		break;
	case GL_FILL:
		set.polyMode = RASTER_FILL;
		break;
	}

	rasterizer->applySettings(set);
}


void
calculateVertexLighting(klVertex& nVertex, const klVec3& nVertexNormal, bool nFrontSide)
{
	const MATERIAL&	mat = nFrontSide ? materialFront : materialBack;
	klColor4		tmpCol, diffuse, specular;


	// set base color value from ambient lighting
	//
	nVertex.col = lightingCache.modelAmbient[mat.idx];


	// walk through all lights...
	//
	for(int li=0; li<=largestActiveLightIdx; li++)
		if(lights[li].enabled)
		{
			const LIGHT& light = lights[li];
			klVec3 lightDir, h;

			// get light direction
			//
			if(lights[li].position[3]==0)						// directional light ?
				lightDir = light.position;
			else												// or positional light ?
			{
				lightDir[0] = light.position[0] - nVertex.pos[0];
				lightDir[1] = light.position[1] - nVertex.pos[1];
				lightDir[2] = light.position[2] - nVertex.pos[2];

				lightDir.normalize();
			}


			// ambient light
			//
			nVertex.col += lightingCache.lightAmbient[li][mat.idx];


			// diffuse light
			//
			klFloat dot = lightDir.dot(nVertexNormal);			// TODO: we could use a mul-lookup table (3 lookups) for this...
			if(!nFrontSide)
				dot = -dot;

			if(dot>0)
			{
				diffuse = lightingCache.lightDiffuse[li][mat.idx];
				diffuse *= dot;

				nVertex.col += diffuse;
			}


			if(tweak.doSpecularLighting)
			{
				// specular light
				//
				if(lightModel.localViewer)			// eye point at (0,0,0) ?
				{									// TODO: need to check if the whole localViewer is done correctly in Klimt at all...
					h = -klVec3(nVertex.pos);

					h.normalize();
					h += lightDir;
					h *= 0.5f;
				}
				else						// exe point at (0,0,oo) ?
				{
					h[0] = lightDir[0];
					h[1] = lightDir[1];
					h[2] = (lightDir[2] - 1.0f);
					h *= 0.5f;
				}

				klFloat dot = -h.dot(nVertexNormal);
				if(dot>0)
				{
					//specular.multiplyComonentWise(light.specular, mat.specular);
					specular = lightingCache.lightDiffuse[li][mat.idx];
					specular *= dot;

					nVertex.col += specular;
				}
			}
		}


	// do color clamping
	//
	if(nVertex.col[0]>1)	nVertex.col[0].setInt(1);
	if(nVertex.col[1]>1)	nVertex.col[1].setInt(1);
	if(nVertex.col[2]>1)	nVertex.col[2].setInt(1);
	if(nVertex.col[3]>1)	nVertex.col[3].setInt(1);
}


/**
}  // class klContext
*/
