/**************************************************************************\
 *
 *  This file is part of the Klimt library.
 *  Copyright (C) 2003 by IMS, Vienna University of Technology.
 *  All rights reserved.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  ("GPL") version 2 as published by the Free Software Foundation.
 *  See the file LICENSE.GPL at the root directory of this source
 *  distribution for additional information about the GNU GPL.
 *  For the full GPL license see
 *  <URL:http://www.gnu.org/copyleft/gpl.html>
 *
 *  For using Klimt with software that can not be combined with the
 *  GNU GPL, and for taking advantage of the additional benefits of
 *  our support services, please contact IMS about acquiring a
 *  Klimt Professional Edition License.
 *
 *  Contact: <mailto:klimt@studierstube.org>
 *  See <URL:http://www.studierstube.org/klimt>
 *  for more information.
 *
 *  Vienna University of Technology
 *  Institute for Software Technology and Interactive Systems
 *  Interactive Media Systems Group
 *  Favoritenstrasse 9-11/188/2
 *  A-1040 Vienna, Austria
 *  <URL:http://www.ims.tuwien.ac.at>.
 *
 **************************************************************************
 *
 * $Header: /cvsroot/klimt/klimt/klimt/src/klContext_oglMatrix.h,v 1.3 2004/02/06 14:14:53 drgoldie Exp $
 *
\**************************************************************************/



/**
class klContext
{
*/


//
// This file holds all opengl functions that deal with
//
//               matrix stack manipulation
//


void glMatrixMode(GLenum nMode)
{
	RECORD_DISPLAYLISTS1(MatrixMode, nMode)
	if(!displaylist.executeCommands)
		return;

	switch(nMode)
	{
	case GL_MODELVIEW:
		curStack = &modelviewStack;
		break;

	case GL_PROJECTION:
		curStack = &projectionStack;
		break;

	case GL_TEXTURE:
		curStack = &textureStack;
		break;
	}
}


void glLoadMatrixf(const GLfloat *nMatrix)
{
	klMatrix mat;
	mat.setFloat(nMatrix);

	RECORD_DISPLAYLISTS1(LoadMatrix, mat)
	if(!displaylist.executeCommands)
		return;

	curStack->getTop() = mat;
	modelProjMatrixDirty = true;
}


void glLoadMatrixd(const GLdouble *nMatrix)
{
	klMatrix mat;
	mat.setDouble(nMatrix);

	RECORD_DISPLAYLISTS1(LoadMatrix, mat)
	if(!displaylist.executeCommands)
		return;

	curStack->getTop() = mat;
	modelProjMatrixDirty = true;
}


void glLoadMatrixx(const GLfixed *nMatrix)
{
	klMatrix mat;
	mat.setFixed(nMatrix);

	RECORD_DISPLAYLISTS1(LoadMatrix, mat)
	if(!displaylist.executeCommands)
		return;

	curStack->getTop() = mat;
	modelProjMatrixDirty = true;
}


void glLoadMatrix(const klMatrix& nMatrix)
{
	curStack->getTop() = nMatrix;
	modelProjMatrixDirty = true;
}


void glMultMatrixf(const GLfloat *nMatrix)
{
	klMatrix mat;
	mat.setFloat(nMatrix);

	RECORD_DISPLAYLISTS1(MultMatrix, mat)
	if(!displaylist.executeCommands)
		return;

	curStack->getTop() *= mat;
	modelProjMatrixDirty = true;
}


void glMultMatrixd(const GLdouble *nMatrix)
{
	klMatrix mat;
	mat.setDouble(nMatrix);

	RECORD_DISPLAYLISTS1(MultMatrix, mat)
	if(!displaylist.executeCommands)
		return;

	curStack->getTop() *= mat;
	modelProjMatrixDirty = true;
}


void glMultMatrixx(const GLfixed *nMatrix)
{
	klMatrix mat;
	mat.setFixed(nMatrix);

	RECORD_DISPLAYLISTS1(MultMatrix, mat)
	if(!displaylist.executeCommands)
		return;

	curStack->getTop() *= mat;
	modelProjMatrixDirty = true;
}


void glMultMatrix(const klMatrix& nMatrix)
{
	curStack->getTop()*=nMatrix;
	modelProjMatrixDirty = true;
}


void glPushMatrix()
{
	RECORD_DISPLAYLISTS0(PushMatrix)
	if(!displaylist.executeCommands)
		return;

	curStack->pushCurrent();
}


void glPopMatrix()
{
	RECORD_DISPLAYLISTS0(PopMatrix)
	if(!displaylist.executeCommands)
		return;

	curStack->pop();
	modelProjMatrixDirty = true;
}


void glLoadIdentity()
{
	RECORD_DISPLAYLISTS0(LoadIdentity)
	if(!displaylist.executeCommands)
		return;

	curStack->getTop().makeIdent();
	modelProjMatrixDirty = true;
}


void glTranslatef(GLfloat nX, GLfloat nY, GLfloat nZ)
{
	klFloat x,y,z;
	x.setFloat(nX);
	y.setFloat(nY);
	z.setFloat(nZ);

	RECORD_DISPLAYLISTS3(Translate, x,y,z)
	if(!displaylist.executeCommands)
		return;

	curStack->getTop().translate(x,y,z);
	modelProjMatrixDirty = true;
}


void glTranslated(GLdouble nX, GLdouble nY, GLdouble nZ)
{
	klFloat x,y,z;
	x.setDouble(nX);
	y.setDouble(nY);
	z.setDouble(nZ);

	RECORD_DISPLAYLISTS3(Translate, x,y,z)
	if(!displaylist.executeCommands)
		return;

	curStack->getTop().translate(x,y,z);
	modelProjMatrixDirty = true;
}


void glTranslatex(GLfixed nX, GLfixed nY, GLfixed nZ)
{
	klFloat x,y,z;
	x.setFixed(nX);
	y.setFixed(nY);
	z.setFixed(nZ);

	RECORD_DISPLAYLISTS3(Translate, x,y,z)
	if(!displaylist.executeCommands)
		return;

	curStack->getTop().translate(x,y,z);
	modelProjMatrixDirty = true;
}


void glTranslate(const klFloat& nX, const klFloat& nY, const klFloat& nZ)
{
	curStack->getTop().translate(nX,nY,nZ);
	modelProjMatrixDirty = true;
}


void glTranslatefv(const GLfloat* nTrans)
{
	klFloat x,y,z;
	x.setFloat(nTrans[0]);
	y.setFloat(nTrans[1]);
	z.setFloat(nTrans[2]);

	RECORD_DISPLAYLISTS3(Translate, x,y,z)
	if(!displaylist.executeCommands)
		return;

	curStack->getTop().translate(x,y,z);
	modelProjMatrixDirty = true;
}


void
glScalef(GLfloat nX, GLfloat nY, GLfloat nZ)
{
	klFloat x,y,z;
	x.setFloat(nX);
	y.setFloat(nY);
	z.setFloat(nZ);

	RECORD_DISPLAYLISTS3(Scale, x,y,z)
	if(!displaylist.executeCommands)
		return;

	curStack->getTop().scale(x,y,z);
	modelProjMatrixDirty = true;
}


void
glScalex(GLfixed nX, GLfixed nY, GLfixed nZ)
{
	klFloat x,y,z;
	x.setFixed(nX);
	y.setFixed(nY);
	z.setFixed(nZ);

	RECORD_DISPLAYLISTS3(Scale, x,y,z)
	if(!displaylist.executeCommands)
		return;

	curStack->getTop().scale(x,y,z);
	modelProjMatrixDirty = true;
}


void
glScale(const klFloat& nX, const klFloat& nY, const klFloat& nZ)
{
	curStack->getTop().scale(nX,nY,nZ);
	modelProjMatrixDirty = true;
}


void glRotatef(GLfloat nAngle, GLfloat nX, GLfloat nY, GLfloat nZ)
{
	klFloat a,x,y,z;
	a.setFloat(nAngle);
	x.setFloat(nX);
	y.setFloat(nY);
	z.setFloat(nZ);

	RECORD_DISPLAYLISTS4(Rotate, a,x,y,z)
	if(!displaylist.executeCommands)
		return;

	curStack->getTop().rotate(nAngle, nX,nY,nZ);
	modelProjMatrixDirty = true;
}


void glRotatex(GLfixed nAngle, GLfixed nX, GLfixed nY, GLfixed nZ)
{
	klFloat a,x,y,z;
	a.setFixed(nAngle);
	x.setFixed(nX);
	y.setFixed(nY);
	z.setFixed(nZ);

	RECORD_DISPLAYLISTS4(Rotate, a,x,y,z)
	if(!displaylist.executeCommands)
		return;

	curStack->getTop().rotate(a, x,y,z);
	modelProjMatrixDirty = true;
}


void glRotate(const klFloat& nAngle, const klFloat& nX, const klFloat& nY, const klFloat& nZ)
{
	curStack->getTop().rotate(nAngle, nX,nY,nZ);
	modelProjMatrixDirty = true;
}


void
glOrtho(GLdouble nLeft, GLdouble nRight, GLdouble nBottom, GLdouble nTop, GLdouble nNearval, GLdouble nFarval)
{
	klFloat l,r,b,t,n,f;
	klMatrix m;

	l.setDouble(nLeft);
	r.setDouble(nRight);
	b.setDouble(nBottom);
	t.setDouble(nTop);
	n.setDouble(nNearval);
	f.setDouble(nFarval);

	RECORD_DISPLAYLISTS6(Ortho, l,r,b,t,n,f)
	if(!displaylist.executeCommands)
		return;

	m.makeOrtho(l,r,b,t,n,f);
	curStack->getTop() *= m;
	modelProjMatrixDirty = true;
}


void
glOrthox(GLfixed nLeft, GLfixed nRight, GLfixed nBottom, GLfixed nTop, GLfixed nNearval, GLfixed nFarval)
{
	klFloat l,r,b,t,n,f;
	klMatrix m;

	l.setFixed(nLeft);
	r.setFixed(nRight);
	b.setFixed(nBottom);
	t.setFixed(nTop);
	n.setFixed(nNearval);
	f.setFixed(nFarval);

	RECORD_DISPLAYLISTS6(Ortho, l,r,b,t,n,f)
	if(!displaylist.executeCommands)
		return;

	m.makeOrtho(l,r,b,t,n,f);
	curStack->getTop() *= m;
	modelProjMatrixDirty = true;
}


void
glOrtho(const klFloat& nLeft, const klFloat& nRight, const klFloat& nBottom, const klFloat& nTop, const klFloat& nNearval, const klFloat& nFarval)
{
	klMatrix m;

	m.makeOrtho(nLeft, nRight, nBottom, nTop, nNearval, nFarval);
	curStack->getTop() *= m;
	modelProjMatrixDirty = true;
}


void
glFrustum(GLdouble nLeft, GLdouble nRight, GLdouble nBottom, GLdouble nTop, GLdouble nNearval, GLdouble nFarval)
{
	klFloat l,r,b,t,n,f;
	klMatrix m;

	l.setDouble(nLeft);
	r.setDouble(nRight);
	b.setDouble(nBottom);
	t.setDouble(nTop);
	n.setDouble(nNearval);
	f.setDouble(nFarval);

	RECORD_DISPLAYLISTS6(Frustum, l,r,b,t,n,f)
	if(!displaylist.executeCommands)
		return;

	m.makeFrustum(l,r,b,t,n,f);
	curStack->getTop() *= m;
	modelProjMatrixDirty = true;
}


void
glFrustumx(GLfixed nLeft, GLfixed nRight, GLfixed nBottom, GLfixed nTop, GLfixed nNearval, GLfixed nFarval)
{
	klFloat l,r,b,t,n,f;
	klMatrix m;

	l.setFixed(nLeft);
	r.setFixed(nRight);
	b.setFixed(nBottom);
	t.setFixed(nTop);
	n.setFixed(nNearval);
	f.setFixed(nFarval);

	RECORD_DISPLAYLISTS6(Frustum, l,r,b,t,n,f)
	if(!displaylist.executeCommands)
		return;

	m.makeFrustum(l,r,b,t,n,f);
	curStack->getTop() *= m;
	modelProjMatrixDirty = true;
}


void
glFrustum(const klFloat& nLeft, const klFloat& nRight, const klFloat& nBottom, const klFloat& nTop, const klFloat& nNearval, const klFloat& nFarval)
{
	klMatrix m;

	m.makeFrustum(nLeft, nRight, nBottom, nTop, nNearval, nFarval);
	curStack->getTop() *= m;
	modelProjMatrixDirty = true;
}


/**
}  // class klContext
*/
