/**************************************************************************\
 *
 *  This file is part of the Klimt library.
 *  Copyright (C) 2003 by IMS, Vienna University of Technology.
 *  All rights reserved.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  ("GPL") version 2 as published by the Free Software Foundation.
 *  See the file LICENSE.GPL at the root directory of this source
 *  distribution for additional information about the GNU GPL.
 *  For the full GPL license see
 *  <URL:http://www.gnu.org/copyleft/gpl.html>
 *
 *  For using Klimt with software that can not be combined with the
 *  GNU GPL, and for taking advantage of the additional benefits of
 *  our support services, please contact IMS about acquiring a
 *  Klimt Professional Edition License.
 *
 *  Contact: <mailto:klimt@studierstube.org>
 *  See <URL:http://www.studierstube.org/klimt>
 *  for more information.
 *
 *  Vienna University of Technology
 *  Institute for Software Technology and Interactive Systems
 *  Interactive Media Systems Group
 *  Favoritenstrasse 9-11/188/2
 *  A-1040 Vienna, Austria
 *  <URL:http://www.ims.tuwien.ac.at>.
 *
 **************************************************************************
 *
 * $Id: klDisplayList_Commands.h,v 1.4 2004/01/14 16:50:18 drgoldie Exp $
 *
\**************************************************************************/

//
// In this file all of DisplayList's store and execute
// functions are implemented.
//

#define ALLOC_COMMAND(STRUCT_TYPE, PTR) STRUCT_TYPE* PTR = (STRUCT_TYPE*)commands.getNewMemory(sizeof(STRUCT_TYPE));
#define GET_COMMAND(STRUCT_TYPE, RAW, CMD, SIZE) const STRUCT_TYPE* CMD = (const STRUCT_TYPE*)RAW; int size = sizeof(STRUCT_TYPE);
#define GET_SIZE(STRUCT_TYPE, SIZE) int size = sizeof(STRUCT_TYPE);



///////////////////////////////////////////////////////////
//
//                      MatrixMode
//
struct CMD_MatrixMode {
	DPL_EXECCOMMAND	func;
	GLenum			mode;
};


int
exec_MatrixMode(const void* nData, klContext* nContext)
{
	GET_COMMAND(CMD_MatrixMode, nData, cmd, size);

	nContext->glMatrixMode(cmd->mode);
	return size;
}

inline void
klDisplayList::store_MatrixMode(GLenum nMode)
{
	ALLOC_COMMAND(CMD_MatrixMode, cmd);

	cmd->func = exec_MatrixMode;
	cmd->mode = nMode;
}


///////////////////////////////////////////////////////////
//
//                      LoadMatrix
//
struct CMD_LoadMatrix {
	DPL_EXECCOMMAND	func;
	klMatrix		matrix;
};

int
exec_LoadMatrix(const void* nData, klContext* nContext)
{
	GET_COMMAND(CMD_LoadMatrix, nData, cmd, size);

	nContext->glLoadMatrix(cmd->matrix);
	return size;
}

inline void
klDisplayList::store_LoadMatrix(const klMatrix& nMatrix)
{
	ALLOC_COMMAND(CMD_LoadMatrix, cmd);

	cmd->func = exec_LoadMatrix;
	cmd->matrix = nMatrix;
}


///////////////////////////////////////////////////////////
//
//                      MultMatrix
//
struct CMD_MultMatrix {
	DPL_EXECCOMMAND	func;
	klMatrix		matrix;
};

int
exec_MultMatrix(const void* nData, klContext* nContext)
{
	GET_COMMAND(CMD_MultMatrix, nData, cmd, size);

	nContext->glMultMatrix(cmd->matrix);
	return size;
}

inline void
klDisplayList::store_MultMatrix(const klMatrix& nMatrix)
{
	ALLOC_COMMAND(CMD_MultMatrix, cmd);

	cmd->func = exec_MultMatrix;
	cmd->matrix = nMatrix;
}


///////////////////////////////////////////////////////////
//
//                      PushMatrix
//
struct CMD_PushMatrix {
	DPL_EXECCOMMAND	func;
};

int
exec_PushMatrix(const void* /*nData*/, klContext* nContext)
{
	GET_SIZE(CMD_PushMatrix, size);

	nContext->glPushMatrix();
	return size;
}

inline void
klDisplayList::store_PushMatrix()
{
	ALLOC_COMMAND(CMD_PushMatrix, cmd);

	cmd->func = exec_PushMatrix;
}


///////////////////////////////////////////////////////////
//
//                      PopMatrix
//
struct CMD_PopMatrix {
	DPL_EXECCOMMAND	func;
};

int
exec_PopMatrix(const void* /*nData*/, klContext* nContext)
{
	GET_SIZE(CMD_PopMatrix, size);

	nContext->glPopMatrix();
	return size;
}

inline void
klDisplayList::store_PopMatrix()
{
	ALLOC_COMMAND(CMD_PopMatrix, cmd);

	cmd->func = exec_PopMatrix;
}


///////////////////////////////////////////////////////////
//
//                     LoadIdentity
//
struct CMD_LoadIdentity {
	DPL_EXECCOMMAND	func;
};

int
exec_LoadIdentity(const void* /*nData*/, klContext* nContext)
{
	GET_SIZE(CMD_LoadIdentity, size);

	nContext->glLoadIdentity();
	return size;
}

inline void
klDisplayList::store_LoadIdentity()
{
	ALLOC_COMMAND(CMD_LoadIdentity, cmd);

	cmd->func = exec_LoadIdentity;
}


///////////////////////////////////////////////////////////
//
//                      Translate
//
struct CMD_Translate {
	DPL_EXECCOMMAND	func;
	klFloat			x,y,z;
};

int
exec_Translate(const void* nData, klContext* nContext)
{
	GET_COMMAND(CMD_Translate, nData, cmd, size);

	nContext->glTranslate(cmd->x, cmd->y, cmd->z);
	return size;
}

inline void
klDisplayList::store_Translate(const klFloat& nX, const klFloat& nY, const klFloat& nZ)
{
	ALLOC_COMMAND(CMD_Translate, cmd);

	cmd->func = exec_Translate;
	cmd->x = nX;
	cmd->y = nY;
	cmd->z = nZ;
}


///////////////////////////////////////////////////////////
//
//                      Scale
//
struct CMD_Scale {
	DPL_EXECCOMMAND	func;
	klFloat			x,y,z;
};

int
exec_Scale(const void* nData, klContext* nContext)
{
	GET_COMMAND(CMD_Scale, nData, cmd, size);

	nContext->glScale(cmd->x, cmd->y, cmd->z);
	return size;
}

inline void
klDisplayList::store_Scale(const klFloat& nX, const klFloat& nY, const klFloat& nZ)
{
	ALLOC_COMMAND(CMD_Scale, cmd);

	cmd->func = exec_Scale;
	cmd->x = nX;
	cmd->y = nY;
	cmd->z = nZ;
}


///////////////////////////////////////////////////////////
//
//                     Rotate
//
struct CMD_Rotate {
	DPL_EXECCOMMAND	func;
	klFloat			a, x,y,z;
};

int
exec_Rotate(const void* nData, klContext* nContext)
{
	GET_COMMAND(CMD_Rotate, nData, cmd, size);

	nContext->glRotate(cmd->a, cmd->x, cmd->y, cmd->z);
	return size;
}

inline void
klDisplayList::store_Rotate(const klFloat& nA, const klFloat& nX, const klFloat& nY, const klFloat& nZ)
{
	ALLOC_COMMAND(CMD_Rotate, cmd);

	cmd->func = exec_Rotate;
	cmd->a = nA;
	cmd->x = nX;
	cmd->y = nY;
	cmd->z = nZ;
}


///////////////////////////////////////////////////////////
//
//                     Ortho
//
struct CMD_Ortho {
	DPL_EXECCOMMAND	func;
	klFloat			left,right, bottom,top, nearval,farval;
};

int
exec_Ortho(const void* nData, klContext* nContext)
{
	GET_COMMAND(CMD_Ortho, nData, cmd, size);

	nContext->glOrtho(cmd->left,cmd->right, cmd->bottom,cmd->top, cmd->nearval,cmd->farval);
	return size;
}

inline void
klDisplayList::store_Ortho(const klFloat& nLeft, const klFloat& nRight, const klFloat& nBottom, const klFloat& nTop, const klFloat& nNearval, const klFloat& nFarval)
{
	ALLOC_COMMAND(CMD_Ortho, cmd);

	cmd->func = exec_Ortho;
	cmd->left = nLeft;
	cmd->right = nRight;
	cmd->bottom = nBottom;
	cmd->top = nTop;
	cmd->nearval = nNearval;
	cmd->farval = nFarval;
}


///////////////////////////////////////////////////////////
//
//                    Frustum
//
struct CMD_Frustum {
	DPL_EXECCOMMAND	func;
	klFloat			left,right, bottom,top, nearval,farval;
};

int
exec_Frustum(const void* nData, klContext* nContext)
{
	GET_COMMAND(CMD_Frustum, nData, cmd, size);

	nContext->glFrustum(cmd->left,cmd->right, cmd->bottom,cmd->top, cmd->nearval,cmd->farval);
	return size;
}

inline void
klDisplayList::store_Frustum(const klFloat& nLeft, const klFloat& nRight, const klFloat& nBottom, const klFloat& nTop, const klFloat& nNearval, const klFloat& nFarval)
{
	ALLOC_COMMAND(CMD_Frustum, cmd);

	cmd->func = exec_Frustum;
	cmd->left = nLeft;
	cmd->right = nRight;
	cmd->bottom = nBottom;
	cmd->top = nTop;
	cmd->nearval = nNearval;
	cmd->farval = nFarval;
}


///////////////////////////////////////////////////////////
//
//                    ClearColor
//
struct CMD_ClearColor {
	DPL_EXECCOMMAND	func;
	klVec4			color;
};

int
exec_ClearColor(const void* nData, klContext* nContext)
{
	GET_COMMAND(CMD_ClearColor, nData, cmd, size);

	nContext->glClearColor(cmd->color);
	return size;
}

inline void
klDisplayList::store_ClearColor(const klVec4& nColor)
{
	ALLOC_COMMAND(CMD_ClearColor, cmd);

	cmd->func = exec_ClearColor;
	cmd->color = nColor;
}


///////////////////////////////////////////////////////////
//
//                    ClearDepth
//
struct CMD_ClearDepth {
	DPL_EXECCOMMAND	func;
	klFloat			depth;
};

int
exec_ClearDepth(const void* nData, klContext* nContext)
{
	GET_COMMAND(CMD_ClearDepth, nData, cmd, size);

	nContext->glClearDepth(cmd->depth);
	return size;
}

inline void
klDisplayList::store_ClearDepth(const klFloat& nDepth)
{
	ALLOC_COMMAND(CMD_ClearDepth, cmd);

	cmd->func = exec_ClearDepth;
	cmd->depth = nDepth;
}


///////////////////////////////////////////////////////////
//
//                      Clear
//
struct CMD_Clear {
	DPL_EXECCOMMAND	func;
	int				flags;
};

int
exec_Clear(const void* nData, klContext* nContext)
{
	GET_COMMAND(CMD_Clear, nData, cmd, size);

	nContext->glClear(cmd->flags);
	return size;
}

inline void
klDisplayList::store_Clear(int nFlags)
{
	ALLOC_COMMAND(CMD_Clear, cmd);

	cmd->func = exec_Clear;
	cmd->flags = nFlags;
}


///////////////////////////////////////////////////////////
//
//                    Colormask
//
struct CMD_ColorMask {
	DPL_EXECCOMMAND	func;
	GLboolean		red, green, blue, alpha;
};

int
exec_ColorMask(const void* nData, klContext* nContext)
{
	GET_COMMAND(CMD_ColorMask, nData, cmd, size);

	nContext->glColorMask(cmd->red, cmd->green, cmd->blue, cmd->alpha);
	return size;
}

inline void
klDisplayList::store_ColorMask(GLboolean nRed, GLboolean nGreen, GLboolean nBlue, GLboolean nAlpha)
{
	ALLOC_COMMAND(CMD_ColorMask, cmd);

	cmd->func = exec_ColorMask;
	cmd->red = nRed;
	cmd->green = nGreen;
	cmd->blue = nBlue;
	cmd->alpha = nAlpha;
}


///////////////////////////////////////////////////////////
//
//                    Enable
//
struct CMD_Enable {
	DPL_EXECCOMMAND	func;
	GLenum			select;
};

int
exec_Enable(const void* nData, klContext* nContext)
{
	GET_COMMAND(CMD_Enable, nData, cmd, size);

	nContext->glEnable(cmd->select);
	return size;
}

inline void
klDisplayList::store_Enable(GLenum nSelect)
{
	ALLOC_COMMAND(CMD_Enable, cmd);

	cmd->func = exec_Enable;
	cmd->select = nSelect;
}


///////////////////////////////////////////////////////////
//
//                    Disable
//
struct CMD_Disable {
	DPL_EXECCOMMAND	func;
	GLenum			select;
};

int
exec_Disable(const void* nData, klContext* nContext)
{
	GET_COMMAND(CMD_Disable, nData, cmd, size);

	nContext->glDisable(cmd->select);
	return size;
}

inline void
klDisplayList::store_Disable(GLenum nSelect)
{
	ALLOC_COMMAND(CMD_Disable, cmd);

	cmd->func = exec_Disable;
	cmd->select = nSelect;
}


///////////////////////////////////////////////////////////
//
//                    DepthMask
//
struct CMD_DepthMask {
	DPL_EXECCOMMAND	func;
	GLboolean		set;
};

int
exec_DepthMask(const void* nData, klContext* nContext)
{
	GET_COMMAND(CMD_DepthMask, nData, cmd, size);

	nContext->glDepthMask(cmd->set);
	return size;
}

inline void
klDisplayList::store_DepthMask(GLboolean nSet)
{
	ALLOC_COMMAND(CMD_DepthMask, cmd);

	cmd->func = exec_DepthMask;
	cmd->set = nSet;
}


///////////////////////////////////////////////////////////
//
//                    CullFace
//
struct CMD_CullFace {
	DPL_EXECCOMMAND	func;
	GLenum			cull;
};

int
exec_CullFace(const void* nData, klContext* nContext)
{
	GET_COMMAND(CMD_CullFace, nData, cmd, size);

	nContext->glCullFace(cmd->cull);
	return size;
}

inline void
klDisplayList::store_CullFace(GLenum nCull)
{
	ALLOC_COMMAND(CMD_CullFace, cmd);

	cmd->func = exec_CullFace;
	cmd->cull = nCull;
}


///////////////////////////////////////////////////////////
//
//                    FrontFace
//
struct CMD_FrontFace {
	DPL_EXECCOMMAND	func;
	GLenum			face;
};

int
exec_FrontFace(const void* nData, klContext* nContext)
{
	GET_COMMAND(CMD_FrontFace, nData, cmd, size);

	nContext->glFrontFace(cmd->face);
	return size;
}

inline void
klDisplayList::store_FrontFace(GLenum nFace)
{
	ALLOC_COMMAND(CMD_FrontFace, cmd);

	cmd->func = exec_FrontFace;
	cmd->face = nFace;
}


///////////////////////////////////////////////////////////
//
//                      Scissor
//
struct CMD_Scissor {
	DPL_EXECCOMMAND	func;
	GLint			x,y;
	GLsizei			width,height;
};

int
exec_Scissor(const void* nData, klContext* nContext)
{
	GET_COMMAND(CMD_Scissor, nData, cmd, size);

	nContext->glScissor(cmd->x, cmd->y, cmd->width, cmd->height);
	return size;
}

inline void
klDisplayList::store_Scissor(GLint x, GLint y, GLsizei width, GLsizei height)
{
	ALLOC_COMMAND(CMD_Scissor, cmd);

	cmd->func = exec_Scissor;
	cmd->x = x;
	cmd->y = y;
	cmd->width = width;
	cmd->height = height;
}


///////////////////////////////////////////////////////////
//
//                      Fog
//
struct CMD_Fog {
	DPL_EXECCOMMAND	func;
	GLenum			pname;
	klFloat			param;
};

int
exec_Fog(const void* nData, klContext* nContext)
{
	GET_COMMAND(CMD_Fog, nData, cmd, size);

	nContext->glFog(cmd->pname, cmd->param);
	return size;
}

inline void
klDisplayList::store_Fog(GLenum pname, const klFloat& param)
{
	ALLOC_COMMAND(CMD_Fog, cmd);

	cmd->func = exec_Fog;
	cmd->pname = pname;
	cmd->param = param;
}


///////////////////////////////////////////////////////////
//
//                      FogColor
//
struct CMD_FogColor {
	DPL_EXECCOMMAND	func;
	klColor4		color;
};

int
exec_FogColor(const void* nData, klContext* nContext)
{
	GET_COMMAND(CMD_FogColor, nData, cmd, size);

	nContext->glFogColor(cmd->color);
	return size;
}

inline void
klDisplayList::store_FogColor(const klColor4& nColor)
{
	ALLOC_COMMAND(CMD_FogColor, cmd);

	cmd->func = exec_FogColor;
	cmd->color = nColor;
}


///////////////////////////////////////////////////////////
//
//                      BlendFunc
//
struct CMD_BlendFunc {
	DPL_EXECCOMMAND	func;
	GLenum			sfactor,dfactor;
};

int
exec_BlendFunc(const void* nData, klContext* nContext)
{
	GET_COMMAND(CMD_BlendFunc, nData, cmd, size);

	nContext->glBlendFunc(cmd->sfactor, cmd->dfactor);
	return size;
}

inline void
klDisplayList::store_BlendFunc(GLenum sfactor, GLenum dfactor)
{
	ALLOC_COMMAND(CMD_BlendFunc, cmd);

	cmd->func = exec_BlendFunc;
	cmd->sfactor = sfactor;
	cmd->dfactor = dfactor;
}


///////////////////////////////////////////////////////////
//
//                      DepthFunc
//
struct CMD_DepthFunc {
	DPL_EXECCOMMAND	func;
	GLenum			param;
};

int
exec_DepthFunc(const void* nData, klContext* nContext)
{
	GET_COMMAND(CMD_DepthFunc, nData, cmd, size);

	nContext->glDepthFunc(cmd->param);
	return size;
}

inline void
klDisplayList::store_DepthFunc(GLenum nFunc)
{
	ALLOC_COMMAND(CMD_DepthFunc, cmd);

	cmd->func = exec_DepthFunc;
	cmd->param = nFunc;
}


///////////////////////////////////////////////////////////
//
//                      Lightv
//
struct CMD_Lightv {
	DPL_EXECCOMMAND	func;
	GLenum			light;
	GLenum			select;
	klVec4			values;
};

int
exec_Lightv(const void* nData, klContext* nContext)
{
	GET_COMMAND(CMD_Lightv, nData, cmd, size);

	nContext->glLightv(cmd->light, cmd->select, cmd->values);
	return size;
}

inline void
klDisplayList::store_Lightv(GLenum nLight, GLenum nSelect, const klVec4& nValues)
{
	ALLOC_COMMAND(CMD_Lightv, cmd);

	cmd->func = exec_Lightv;
	cmd->light = nLight;
	cmd->select = nSelect;
	cmd->values = nValues;
}


///////////////////////////////////////////////////////////
//
//                      LightModel
//
struct CMD_LightModel {
	DPL_EXECCOMMAND	func;
	GLenum			param;
	klFloat			value;
};

int
exec_LightModel(const void* nData, klContext* nContext)
{
	GET_COMMAND(CMD_LightModel, nData, cmd, size);

	nContext->glLightModel(cmd->param, cmd->value);
	return size;
}

inline void
klDisplayList::store_LightModel(GLenum nParam, const klFloat& nValue)
{
	ALLOC_COMMAND(CMD_LightModel, cmd);

	cmd->func = exec_LightModel;
	cmd->param = nParam;
	cmd->value = nValue;
}


///////////////////////////////////////////////////////////
//
//                      LightModelv
//
struct CMD_LightModelv {
	DPL_EXECCOMMAND	func;
	GLenum			param;
	klVec4			values;
};

int
exec_LightModelv(const void* nData, klContext* nContext)
{
	GET_COMMAND(CMD_LightModelv, nData, cmd, size);

	nContext->glLightModelv(cmd->param, cmd->values);
	return size;
}

inline void
klDisplayList::store_LightModelv(GLenum nParam, const klVec4& nValues)
{
	ALLOC_COMMAND(CMD_LightModelv, cmd);

	cmd->func = exec_LightModelv;
	cmd->param = nParam;
	cmd->values = nValues;
}


///////////////////////////////////////////////////////////
//
//                      ShadeModel
//
struct CMD_ShadeModel {
	DPL_EXECCOMMAND	func;
	GLenum			model;
};

int
exec_ShadeModel(const void* nData, klContext* nContext)
{
	GET_COMMAND(CMD_ShadeModel, nData, cmd, size);

	nContext->glShadeModel(cmd->model);
	return size;
}

inline void
klDisplayList::store_ShadeModel(GLenum nModel)
{
	ALLOC_COMMAND(CMD_ShadeModel, cmd);

	cmd->func = exec_ShadeModel;
	cmd->model = nModel;
}


///////////////////////////////////////////////////////////
//
//                      Material
//
struct CMD_Material {
	DPL_EXECCOMMAND	func;
	GLenum			side;
	GLenum			select;
	klFloat			value;
};

int
exec_Material(const void* nData, klContext* nContext)
{
	GET_COMMAND(CMD_Material, nData, cmd, size);

	nContext->glMaterial(cmd->side, cmd->select, cmd->value);
	return size;
}

inline void
klDisplayList::store_Material(GLenum nSide, GLenum nSelect, const klFloat& nValue)
{
	ALLOC_COMMAND(CMD_Material, cmd);

	cmd->func = exec_Material;
	cmd->side = nSide;
	cmd->select = nSelect;
	cmd->value = nValue;
}


///////////////////////////////////////////////////////////
//
//                      Materialv
//
struct CMD_Materialv {
	DPL_EXECCOMMAND	func;
	GLenum			side;
	GLenum			select;
	klVec4			values;
};

int
exec_Materialv(const void* nData, klContext* nContext)
{
	GET_COMMAND(CMD_Materialv, nData, cmd, size);

	nContext->glMaterialv(cmd->side, cmd->select, cmd->values);
	return size;
}

inline void
klDisplayList::store_Materialv(GLenum nSide, GLenum nSelect, const klVec4& nValues)
{
	ALLOC_COMMAND(CMD_Materialv, cmd);

	cmd->func = exec_Materialv;
	cmd->side = nSide;
	cmd->select = nSelect;
	cmd->values = nValues;
}


///////////////////////////////////////////////////////////
//
//                      ColorMaterial
//
struct CMD_ColorMaterial {
	DPL_EXECCOMMAND	func;
	GLenum			side;
	GLenum			mode;
};

int
exec_ColorMaterial(const void* nData, klContext* nContext)
{
	GET_COMMAND(CMD_ColorMaterial, nData, cmd, size);

	nContext->glColorMaterial(cmd->side, cmd->mode);
	return size;
}

inline void
klDisplayList::store_ColorMaterial(GLenum nSide, GLenum nMode)
{
	ALLOC_COMMAND(CMD_ColorMaterial, cmd);

	cmd->func = exec_ColorMaterial;
	cmd->side = nSide;
	cmd->mode = nMode;
}


///////////////////////////////////////////////////////////
//
//                      PolygonMode
//
struct CMD_PolygonMode {
	DPL_EXECCOMMAND	func;
	GLenum			side;
	GLenum			poly;
};

int
exec_PolygonMode(const void* nData, klContext* nContext)
{
	GET_COMMAND(CMD_PolygonMode, nData, cmd, size);

	nContext->glPolygonMode(cmd->side, cmd->poly);
	return size;
}

inline void
klDisplayList::store_PolygonMode(GLenum nSide, GLenum nPoly)
{
	ALLOC_COMMAND(CMD_PolygonMode, cmd);

	cmd->func = exec_PolygonMode;
	cmd->side = nSide;
	cmd->poly = nPoly;
}


///////////////////////////////////////////////////////////
//
//                      TexEnv
//
struct CMD_TexEnv {
	DPL_EXECCOMMAND	func;
	GLint			param;
};

int
exec_TexEnv(const void* nData, klContext* nContext)
{
	GET_COMMAND(CMD_TexEnv, nData, cmd, size);

	nContext->glTexEnvi(0,0, cmd->param);			// first two parameters are fixed and klimt does not check them anyway...
	return size;
}

inline void
klDisplayList::store_TexEnv(GLint nParam)
{
	ALLOC_COMMAND(CMD_TexEnv, cmd);

	cmd->func = exec_TexEnv;
	cmd->param = nParam;
}


///////////////////////////////////////////////////////////
//
//                      PushAttrib
//
struct CMD_PushAttrib {
	DPL_EXECCOMMAND	func;
	GLbitfield		mask;
};

int
exec_PushAttrib(const void* nData, klContext* nContext)
{
	GET_COMMAND(CMD_PushAttrib, nData, cmd, size);

	nContext->glPushAttrib(cmd->mask);
	return size;
}

inline void
klDisplayList::store_PushAttrib(GLbitfield nMask)
{
	ALLOC_COMMAND(CMD_PushAttrib, cmd);

	cmd->func = exec_PushAttrib;
	cmd->mask = nMask;
}


///////////////////////////////////////////////////////////
//
//                      PopAttrib
//
struct CMD_PopAttrib {
	DPL_EXECCOMMAND	func;
};

int
exec_PopAttrib(const void* /*nData*/, klContext* nContext)
{
	GET_SIZE(CMD_PopAttrib, size);

	nContext->glPopAttrib();
	return size;
}

inline void
klDisplayList::store_PopAttrib()
{
	ALLOC_COMMAND(CMD_PopAttrib, cmd);

	cmd->func = exec_PopAttrib;
}


///////////////////////////////////////////////////////////
//
//                      BindTexture
//
struct CMD_BindTexture {
	DPL_EXECCOMMAND	func;
	GLuint			texture;
};

int
exec_BindTexture(const void* nData, klContext* nContext)
{
	GET_COMMAND(CMD_BindTexture, nData, cmd, size);

	nContext->glBindTexture(0, cmd->texture);			// nTarget is ignored...
	return size;
}

inline void
klDisplayList::store_BindTexture(GLuint nTexture)
{
	ALLOC_COMMAND(CMD_BindTexture, cmd);

	cmd->func = exec_BindTexture;
	cmd->texture = nTexture;
 
}


///////////////////////////////////////////////////////////
//
//                      Begin
//
struct CMD_Begin {
	DPL_EXECCOMMAND	func;
	GLenum			type;
};

int
exec_Begin(const void* nData, klContext* nContext)
{
	GET_COMMAND(CMD_Begin, nData, cmd, size);

	nContext->glBegin(cmd->type);
	return size;
}

inline void
klDisplayList::store_Begin(GLenum nType)
{
	ALLOC_COMMAND(CMD_Begin, cmd);

	cmd->func = exec_Begin;
	cmd->type = nType;
}


///////////////////////////////////////////////////////////
//
//                      End
//
struct CMD_End {
	DPL_EXECCOMMAND	func;
};

int
exec_End(const void* /*nData*/, klContext* nContext)
{
	GET_SIZE(CMD_End, size);

	nContext->glEnd();
	return size;
}

inline void
klDisplayList::store_End()
{
	ALLOC_COMMAND(CMD_End, cmd);

	cmd->func = exec_End;
}


///////////////////////////////////////////////////////////
//
//                      Color
//
struct CMD_Color {
	DPL_EXECCOMMAND	func;
	klColor4		color;
};

int
exec_Color(const void* nData, klContext* nContext)
{
	GET_COMMAND(CMD_Color, nData, cmd, size);

	nContext->glColor(cmd->color);
	return size;
}

inline void
klDisplayList::store_Color(const klColor4& nColor)
{
	ALLOC_COMMAND(CMD_Color, cmd);

	cmd->func = exec_Color;
	cmd->color = nColor;
}


///////////////////////////////////////////////////////////
//
//                      Normal
//
struct CMD_Normal {
	DPL_EXECCOMMAND	func;
	klVec3			normal;
};

int
exec_Normal(const void* nData, klContext* nContext)
{
	GET_COMMAND(CMD_Normal, nData, cmd, size);

	nContext->glNormal(cmd->normal);
	return size;
}

inline void
klDisplayList::store_Normal(const klVec3& nNormal)
{
	ALLOC_COMMAND(CMD_Normal, cmd);

	cmd->func = exec_Normal;
	cmd->normal = nNormal;
}


///////////////////////////////////////////////////////////
//
//                      TexCoord
//
struct CMD_TexCoord {
	DPL_EXECCOMMAND	func;
	klVec4			texCoord;
};

int
exec_TexCoord(const void* nData, klContext* nContext)
{
	GET_COMMAND(CMD_TexCoord, nData, cmd, size);

	nContext->glTexCoord(cmd->texCoord);
	return size;
}

inline void
klDisplayList::store_TexCoord(const klVec4& nTexCoord)
{
	ALLOC_COMMAND(CMD_TexCoord, cmd);

	cmd->func = exec_TexCoord;
	cmd->texCoord = nTexCoord;
}


///////////////////////////////////////////////////////////
//
//                      Vertex
//
struct CMD_Vertex {
	DPL_EXECCOMMAND	func;
	klVec4			vertex;
};

int
exec_Vertex(const void* nData, klContext* nContext)
{
	GET_COMMAND(CMD_Vertex, nData, cmd, size);

	nContext->glVertex(cmd->vertex);
	return size;
}

inline void
klDisplayList::store_Vertex(const klVec4& nVertex)
{
	ALLOC_COMMAND(CMD_Vertex, cmd);

	cmd->func = exec_Vertex;
	cmd->vertex = nVertex;
}


///////////////////////////////////////////////////////////
//
//                      TexImage2D
//
struct CMD_TexImage2D {
	DPL_EXECCOMMAND	func;
	GLint			level;
	GLint			internalformat;
	GLsizei			width;
	GLsizei			height;
	GLint			border;
	GLenum			format;
	GLenum			type;
	void			*rgbaPixels;
};

int
exec_TexImage2D(const void* nData, klContext* nContext)
{
	GET_COMMAND(CMD_TexImage2D, nData, cmd, size);

	nContext->getRasterizer()->updateCurrentTexture(cmd->level, cmd->internalformat, cmd->width, cmd->height,
												   cmd->border, cmd->format, cmd->type, cmd->rgbaPixels);
	return size;
}

inline void
klDisplayList::store_TexImage2D(GLint level, GLint internalformat, GLsizei width, GLsizei height,
								GLint border, GLenum format, GLenum type, const GLvoid *pixels)
{
	unsigned char* data = new unsigned char[width*height*4];		// only rgba32 allowed to be passed into this function

	memcpy(data, pixels, width*height*4);

	ALLOC_COMMAND(CMD_TexImage2D, cmd);

	cmd->func = exec_TexImage2D;
	cmd->level = level;
	cmd->internalformat = internalformat;
	cmd->width = width;
	cmd->height = height;
	cmd->border = border;
	cmd->format = format;
	cmd->type = type;
	cmd->rgbaPixels = data;

	storedExecMem.addTail(data);		// add this pointer so it will be deleted when calling cleanup()
}


///////////////////////////////////////////////////////////
//
//                    DrawPixels
//
struct CMD_DrawPixels {
	DPL_EXECCOMMAND	func;
	GLsizei			width;
	GLsizei			height;
	GLenum			format;
	GLenum			type;
	void*			rgbxPixels;
};

int
exec_DrawPixels(const void* nData, klContext* nContext)
{
	GET_COMMAND(CMD_DrawPixels, nData, cmd, size);

	nContext->glDrawPixels(cmd->width, cmd->height, cmd->format, cmd->type, cmd->rgbxPixels);
	return size;
}

inline void
klDisplayList::store_DrawPixels(GLsizei nWidth, GLsizei nHeight, GLenum nFormat, GLenum nType, const void* nPixels)
{
	void* pixels = new unsigned char[nWidth*nHeight*4];
	memcpy(pixels, nPixels, nWidth*nHeight*4);

	ALLOC_COMMAND(CMD_DrawPixels, cmd);

	cmd->func = exec_DrawPixels;
	cmd->width = nWidth;
	cmd->height = nHeight;
	cmd->format = nFormat;
	cmd->type = nType;
	cmd->rgbxPixels = pixels;

	storedExecMem.addTail(pixels);		// add this pointer so it will be deleted when calling cleanup()
}
