/**************************************************************************\
 *
 *  This file is part of the Klimt library.
 *  Copyright (C) 2003 by IMS, Vienna University of Technology.
 *  All rights reserved.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  ("GPL") version 2 as published by the Free Software Foundation.
 *  See the file LICENSE.GPL at the root directory of this source
 *  distribution for additional information about the GNU GPL.
 *  For the full GPL license see
 *  <URL:http://www.gnu.org/copyleft/gpl.html>
 *
 *  For using Klimt with software that can not be combined with the
 *  GNU GPL, and for taking advantage of the additional benefits of
 *  our support services, please contact IMS about acquiring a
 *  Klimt Professional Edition License.
 *
 *  Contact: <mailto:klimt@studierstube.org>
 *  See <URL:http://www.studierstube.org/klimt>
 *  for more information.
 *
 *  Vienna University of Technology
 *  Institute for Software Technology and Interactive Systems
 *  Interactive Media Systems Group
 *  Favoritenstrasse 9-11/188/2
 *  A-1040 Vienna, Austria
 *  <URL:http://www.ims.tuwien.ac.at>.
 *
 **************************************************************************
 *
 * $Header: /cvsroot/klimt/klimt/klimt/src/klNative_egl.h,v 1.2 2004/01/13 11:20:09 drgoldie Exp $
 *
\**************************************************************************/


#include "klEGLDisplay.h"


#ifdef __cplusplus
extern "C" {
#endif


/// eglGetDisplay returns an EGLDisplay object
/**
 *   Works as specified.
 */
KLIMT_API EGLDisplay eglGetDisplay(NativeDisplayType nDisplay)
{
	// currently we simply support only one display
	// therefore we don't care about the NativeDisplayType
	//
	return new klEGLDisplay(nDisplay);
}


/// eglInitialize initializes OpenGL|ES
/**
 *   Works as specified.
 */
KLIMT_API EGLBoolean eglInitialize(EGLDisplay dpy, EGLint *major, EGLint *minor)
{
	*major = 1;
	*minor = 0;
	return EGL_TRUE;
}


/// eglCreateWindowSurface creates an on-screen render target
/**
 *   config and attrib_list are ignored. Under WindowsCE always
 *   fullscreen windows are created.
 */
KLIMT_API EGLSurface eglCreateWindowSurface(EGLDisplay dpy, EGLConfig config, NativeWindowType window, const EGLint *attrib_list)
{
	// we ignore every parameter here except for window,
	// which is used to create the render surface
	//
	klEGLSurface* surface = new klEGLSurfaceWindow();

	if(!surface->setNativeWindow(window))
	{
		surface->destroy();
		delete surface;
		return NULL;
	}

//
// TODO: Check if we really need this preprocessor switches
//
#ifdef _IS_QTEMBEDDED_
	// For QT Embedded the QWidget is the NativeDisplay. the NativeWindow is the double buffered ZSurface.
	if(!surface->setNativeDisplay(((klEGLDisplay*)dpy)->getNativeDisplay()))
	{
		surface->destroy();
		delete surface;
		return NULL;
	}
#endif //_IS_QTEMBEDDED_

	return surface;
}


/*KLIMT_API EGLSurface eglCreatePbufferSurface(EGLDisplay dpy, EGLConfig config, const EGLint *attrib_list)
{
	//
	// TODO: eglCreatePbufferSurface
	//
}*/


/// eglDestroySurface destroys a render target
/**
 *   Works as specified.
 */
KLIMT_API EGLBoolean eglDestroySurface(EGLDisplay dpy, EGLSurface surface)
{
	if(surface==NULL)
		return EGL_FALSE;

	klEGLSurface* s = reinterpret_cast<klEGLSurface*>(surface);
	EGLBoolean ret;

	ret = s->destroy() ? EGL_TRUE : EGL_FALSE;
	delete s;

	return ret;
}


/// eglCreateContext creates a render context
/**
 *   config, share_list and attrib_list are ignored.
 *   Klimt does not provide thread-specific contexts.
 */
KLIMT_API EGLContext eglCreateContext(EGLDisplay dpy, EGLConfig config, EGLContext share_list, const EGLint *attrib_list)
{
	// we ignore all parameters and just return a new context...
	//
	return new klContext(new klRasterizer);
}


/// eglMakeCurrent sets the active render context
/**
 *   draw is used both for reading and drawing.
 *   Klimt does not provide thread-specific contexts.
 */
KLIMT_API EGLBoolean eglMakeCurrent(EGLDisplay dpy, EGLSurface draw, EGLSurface read, EGLContext ctx)
{
	// we ignore dpy and the fact the draw and read might be different...
	//
	currentContext = (klContext*)ctx;
	klEGLDisplay* display = reinterpret_cast<klEGLDisplay*>(dpy);
	klEGLSurface* surface = reinterpret_cast<klEGLSurface*>(draw);

	if(display== NULL || surface==NULL || currentContext==NULL || currentContext->getRasterizer()==NULL)
		return EGL_FALSE;

	currentContext->getRasterizer()->setDisplay(display);
	currentContext->getRasterizer()->setSurface(surface);
	return EGL_TRUE;
}


/// eglGetCurrentContext returns the active render context
/**
 *   Klimt does not provide thread-specific contexts.
 */
KLIMT_API EGLContext eglGetCurrentContext(void)
{
	return currentContext;
}


/// eglGetCurrentContext returns the active render target
/**
 *   readdraw is ignored. Klimt currently does reading and drawing
 *   from and to the same render target.
 *   Klimt does not provide thread-specific contexts.
 */
KLIMT_API EGLSurface eglGetCurrentSurface(EGLint readdraw)
{
	if(currentContext==NULL || currentContext->getRasterizer()==NULL)
		return NULL;

	return currentContext->getRasterizer()->getSurface();
}


/// eglGetCurrentDisplay returns the current active display
/**
 *   Klimt does not provide thread-specific contexts.
 */
KLIMT_API EGLDisplay eglGetCurrentDisplay(void)
{
	if(currentContext==NULL || currentContext->getRasterizer()==NULL)
		return NULL;

	return currentContext->getRasterizer()->getDisplay();
}


/// eglWaitGL waits until all OpenGL calls are exectuted
/**
 *   Works as specified.
 */
KLIMT_API EGLBoolean eglWaitGL(void)
{
	// nothing to do...
	return EGL_TRUE;
}


/// eglWaitGL waits until all native OpenGL calls are exectuted
/**
 *   Works as specified.
 */
KLIMT_API EGLBoolean eglWaitNative(EGLint engine)
{
	// nothing to do...
	return EGL_TRUE;
}


/// eglSwapBuffers displays the current surface on the display
/**
 *   Works as specified.
 */
KLIMT_API EGLBoolean eglSwapBuffers(EGLDisplay dpy, EGLSurface draw)
{
	klEGLSurface* surface = reinterpret_cast<klEGLSurface*>(draw);

	if(surface==NULL ||
	   currentContext==NULL ||
	   currentContext->getRasterizer()==NULL)
		return EGL_FALSE;

	if(!surface->flip())
		return EGL_FALSE;

	// need to set the surface again since it might have changed
	// some attributes (e.g. color buffer)
	return currentContext->getRasterizer()->setSurface(surface) ? GL_TRUE : GL_FALSE;
}


#ifdef __cplusplus
}
#endif
