/**************************************************************************\
 *
 *  This file is part of the Klimt library.
 *  Copyright (C) 2003 by IMS, Vienna University of Technology.
 *  All rights reserved.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  ("GPL") version 2 as published by the Free Software Foundation.
 *  See the file LICENSE.GPL at the root directory of this source
 *  distribution for additional information about the GNU GPL.
 *  For the full GPL license see
 *  <URL:http://www.gnu.org/copyleft/gpl.html>
 *
 *  For using Klimt with software that can not be combined with the
 *  GNU GPL, and for taking advantage of the additional benefits of
 *  our support services, please contact IMS about acquiring a
 *  Klimt Professional Edition License.
 *
 *  Contact: <mailto:klimt@studierstube.org>
 *  See <URL:http://www.studierstube.org/klimt>
 *  for more information.
 *
 *  Vienna University of Technology
 *  Institute for Software Technology and Interactive Systems
 *  Interactive Media Systems Group
 *  Favoritenstrasse 9-11/188/2
 *  A-1040 Vienna, Austria
 *  <URL:http://www.ims.tuwien.ac.at>.
 *
 **************************************************************************
 *
 * $Header: /cvsroot/klimt/klimt/klimt/src/klWrapper.h,v 1.5 2004/02/06 14:15:10 drgoldie Exp $
 *
\**************************************************************************/


#ifdef __cplusplus
extern "C" {
#endif


//
// C-Wrapper functions for the OpenGL(|ES) interface
//


/// glGetError currently always returns GL_NO_ERROR
KLIMT_API GLenum glGetError()
{
	return GL_NO_ERROR;
}

// wrapper methods implemented in klContext_oglMatrix.h
//


/// glMatrixMode specifies which matrix is the current matrix
/**
 *  Works with GL_MODELVIEW, GL_PROJECTION, GL_TEXTURE
 */
KLIMT_API void glMatrixMode(GLenum nMode)
{
	currentContext->glMatrixMode(nMode);
}


/// glLoadMatrixf replaces the current matrix with an arbitrary matrix
/**
 *   Works as specified.
 */
KLIMT_API void glLoadMatrixf(const GLfloat *nMatrix)
{
	currentContext->glLoadMatrixf(nMatrix);
}


/// glLoadMatrixd replaces the current matrix with an arbitrary matrix
/**
 *   Works as specified
 */
KLIMT_API void glLoadMatrixd(const GLdouble *nMatrix)
{
	currentContext->glLoadMatrixd(nMatrix);
}


/// glLoadMatrixx replaces the current matrix with an arbitrary matrix
/**
 *   Works as specified
 */
KLIMT_API void glLoadMatrixx(const GLfixed *nMatrix)
{
	currentContext->glLoadMatrixx(nMatrix);
}


/// glMultMatrixf multiplies the current matrix by an arbitrary matrix
/**
 *   Works as specified
 */
KLIMT_API void glMultMatrixf(const GLfloat *nMatrix)
{
	currentContext->glMultMatrixf(nMatrix);
}


/// glMultMatrixd multiplies the current matrix by an arbitrary matrix
/**
 *   Works as specified
 */
KLIMT_API void glMultMatrixd(const GLdouble *nMatrix)
{
	currentContext->glMultMatrixd(nMatrix);
}


/// glMultMatrixx multiplies the current matrix by an arbitrary matrix
/**
 *   Works as specified
 */
KLIMT_API void glMultMatrixx(const GLfixed *nMatrix)
{
	currentContext->glMultMatrixx(nMatrix);
}


/// glPushMatrix pushes the current matrix onto the stack
/**
 *   Works as specified
 */
KLIMT_API void glPushMatrix()
{
	currentContext->glPushMatrix();
}


/// glPushMatrix pop the current matrix from the stack
/**
 *   Works as specified
 */
KLIMT_API void glPopMatrix()
{
	currentContext->glPopMatrix();
}


/// glLoadIdentity replaces the current matrix with the identity matrix
/**
 *   Works as specified
 */
KLIMT_API void glLoadIdentity()
{
	currentContext->glLoadIdentity();
}


/// glTranslatef multiplies the current matrix by a translation matrix
/**
 *   Works as specified
 */
KLIMT_API void glTranslatef(GLfloat nX, GLfloat nY, GLfloat nZ)
{
	currentContext->glTranslatef(nX, nY, nZ);
}


/// glTranslated multiplies the current matrix by a translation matrix
/**
 *   Works as specified
 */
KLIMT_API void glTranslated(GLdouble nX, GLdouble nY, GLdouble nZ)
{
	currentContext->glTranslated(nX, nY, nZ);
}


/// glTranslatefv multiplies the current matrix by a translation matrix
/**
 *   Works as specified
 */
KLIMT_API void glTranslatefv(const GLfloat* nTrans)
{
	currentContext->glTranslatefv(nTrans);
}


/// glTranslatex multiplies the current matrix by a translation matrix
/**
 *   Works as specified
 */
KLIMT_API void glTranslatex(GLfixed nX, GLfixed nY, GLfixed nZ)
{
	currentContext->glTranslatex(nX, nY, nZ);
}


/// glScalef multiplies the current matrix by a general scaling matrix
/**
 *   Works as specified
 */
KLIMT_API void glScalef(GLfloat nX, GLfloat nY, GLfloat nZ)
{
	currentContext->glScalef(nX, nY, nZ);
}


/// glScalex multiplies the current matrix by a general scaling matrix
/**
 *   Works as specified
 */
KLIMT_API void glScalex(GLfixed nX, GLfixed nY, GLfixed nZ)
{
	currentContext->glScalex(nX, nY, nZ);
}


/// glRotatef multiplies the current matrix by a rotation matrix
/**
 *   Works as specified
 */
KLIMT_API void glRotatef(GLfloat nAngle, GLfloat nX, GLfloat nY, GLfloat nZ)
{
	currentContext->glRotatef(nAngle, nX, nY, nZ);
}


/// glRotatex multiplies the current matrix by a rotation matrix
/**
 *   Works as specified
 */
KLIMT_API void glRotatex(GLfixed nAngle, GLfixed nX, GLfixed nY, GLfixed nZ)
{
	currentContext->glRotatex(nAngle, nX, nY, nZ);
}


/// glOrtho multiplies the current matrix by an orthographic matrix
/**
 *   Works as specified
 */
KLIMT_API void glOrtho(GLdouble left, GLdouble right, GLdouble bottom, GLdouble top, GLdouble nearval, GLdouble farval)
{
	currentContext->glOrtho(left, right, bottom, top, nearval, farval);
}


/// glOrthox multiplies the current matrix by an orthographic matrix
/**
 *   Works as specified
 */
KLIMT_API void glOrthox(GLfixed left, GLfixed right, GLfixed bottom, GLfixed top, GLfixed nearval, GLfixed farval)
{
	currentContext->glOrthox(left, right, bottom, top, nearval, farval);
}


/// glFrustum multiplies the current matrix by a perspective matrix
/**
 *   Works as specified
 */
KLIMT_API void glFrustum(GLdouble left, GLdouble right, GLdouble bottom, GLdouble top, GLdouble znear, GLdouble zfar)
{
	currentContext->glFrustum(left, right, bottom, top, znear, zfar);
}


/// glFrustumx multiplies the current matrix by a perspective matrix
/**
 *   Works as specified
 */
KLIMT_API void glFrustumx(GLfixed left, GLfixed right, GLfixed bottom, GLfixed top, GLfixed znear, GLfixed zfar)
{
	currentContext->glFrustumx(left, right, bottom, top, znear, zfar);
}


// wrapper methods implemented in klContext_oglAttributeStack.h
//


/// glPushAttrib pushes the attribute stack
/**
 *   Works as specified - but only for those features
 *   that supported by Klimt. E.g. GL_ACCUM_BUFFER_BIT
 *   won't effect anything since the accumulation buffer
 *   is not supported anyhow...
 */
KLIMT_API void glPushAttrib(GLbitfield nMask)
{
	currentContext->glPushAttrib(nMask);
}


/// glPopAttrib pops the attribute stack
/**
 *   Works as specified - but only for those features
 *   that supported by Klimt. E.g. GL_ACCUM_BUFFER_BIT
 *   won't effect anything since the accumulation buffer
 *   is not supported anyhow...
 */
KLIMT_API void glPopAttrib()
{
	currentContext->glPopAttrib();
}


/// glPushClientAttrib saves groups of client-state variables on the client-attribute stack
/**
 *   Works as specified - but only for those features
 *   that supported by Klimt. E.g. GL_CLIENT_PIXEL_STORE_BIT
 *   won't effect anything...
 */
KLIMT_API void glPushClientAttrib(GLbitfield mask)
{
	currentContext->glPushClientAttrib(mask);
}


/// glPopClientAttrib restores groups of client-state variables from the client-attribute stack
/**
 *   Works as specified - but only for those features
 *   that supported by Klimt. E.g. GL_CLIENT_PIXEL_STORE_BIT
 *   won't effect anything...
 */
KLIMT_API void glPopClientAttrib()
{
	currentContext->glPopClientAttrib();
}



// wrapper methods implemented in klContext_oglPrimitve.h
//


/// glDisableClientState enables arrays respectively
/**
 *   GL_INDEX_ARRAY and GL_EDGE_FLAG_ARRAY are not supported
 */
KLIMT_API void glEnableClientState(GLenum nState)
{
	currentContext->glEnableClientState(nState);
}


/// glDisableClientState disables arrays respectively
/**
 *   GL_INDEX_ARRAY and GL_EDGE_FLAG_ARRAY are not supported
 */
KLIMT_API void glDisableClientState(GLenum nState)
{
	currentContext->glDisableClientState(nState);
}


/// glColorPointer specifies a vertex
/**
 *   Works as specified
 */
KLIMT_API void glColorPointer(GLint size, GLenum type, GLsizei stride, const GLvoid *pointer)
{
	currentContext->glColorPointer(size, type, stride, pointer);
}


/// glNormalPointer specifies a vertex
/**
 *   Works as specified
 */
KLIMT_API void glNormalPointer(GLenum type, GLsizei stride, const GLvoid *pointer)
{
	currentContext->glNormalPointer(type, stride, pointer);
}


/// glTexCoordPointer specifies a vertex
/**
 *   Works as specified
 */
KLIMT_API void glTexCoordPointer(GLint size, GLenum type, GLsizei stride, const GLvoid *pointer)
{
	currentContext->glTexCoordPointer(size, type, stride, pointer);
}


/// glVertexPointer specifies a vertex
/**
 *   Works as specified
 */
KLIMT_API void glVertexPointer(GLint size, GLenum type, GLsizei stride, const GLvoid *pointer)
{
	currentContext->glVertexPointer(size, type, stride, pointer);
}


/// glDrawElements renders primitives from array data
/**
 *   Works as specified
 */
KLIMT_API void glDrawElements(GLenum polymode, GLsizei count, GLenum datatype, const GLvoid *indices)
{
	currentContext->glDrawElements(polymode, count, datatype, indices);
}


// wrapper methods implemented in klContext_oglSetup.h
//


/// glClearColor specifies clear values for the color buffers
/**
 *   The alpha value is ignored
 */
KLIMT_API void glClearColor(GLclampf nR, GLclampf nG, GLclampf nB, GLclampf nA)
{
	currentContext->glClearColor(nR, nG, nB, nA);
}


/// glClearColorx specifies clear values for the color buffer
/**
 *   The alpha value is ignored
 */
KLIMT_API void glClearColorx(GLfixed nR, GLfixed nG, GLfixed nB, GLfixed nA)
{
	currentContext->glClearColorx(nR, nG, nB, nA);
}


/// glClearDepth specifies clear values for the depth buffer
/**
 *   Works as specified
 */
KLIMT_API void glClearDepth(GLclampd nDepth)
{
	currentContext->glClearDepth(nDepth);
}


/// glClearDepthx specifies clear values for the color buffers
/**
 *   Works as specified
 */
KLIMT_API void glClearDepthx(GLfixed nDepth)
{
	currentContext->glClearDepthx(nDepth);
}


/// glClear clears buffers to preset values
/**
 *   Only GL_COLOR_BUFFER_BIT and GL_DEPTH_BUFFER_BIT are used.
 */
KLIMT_API void glClear(GLbitfield nFlags)
{
	currentContext->glClear(nFlags);
}


/// glPolygonMode selects a polygon rasterization mode
/**
 *   Works as specified
 */
KLIMT_API void glPolygonMode(GLenum nSide, GLenum nPoly)
{
	currentContext->glPolygonMode(nSide, nPoly);
}


/// glEnable enables OpenGL capabilities
/**
 *   The following list of parameters is currently supported:
 *   - GL_BLEND
 *   - GL_COLOR_MATERIAL
 *   - GL_CULL_FACE
 *   - GL_DEPTH_TEST
 *   - GL_FOG
 *   - GL_LIGHTING
 *   - GL_LIGHT0-7
 *   - GL_NORMALIZE
 *   - GL_SCISSOR_TEST
 *   - GL_TEXTURE_2D
 */
KLIMT_API void glEnable(GLenum nSelect)
{
	currentContext->glEnable(nSelect);
}


/// glDisable disables OpenGL capabilities
/**
 *  @see glEnable for a list of supported parameters
 */
KLIMT_API void glDisable(GLenum nSelect)
{
	currentContext->glDisable(nSelect);
}


/// glIsEnabled tests whether a capability is enabled
/**
 *  @see glEnable for a list of supported parameters
 */
KLIMT_API GLboolean glIsEnabled(GLenum nSelect)
{
	return currentContext->glIsEnabled(nSelect);
}


/// glDepthMask enables or disables writing into the depth buffer
/**
 *   Works as specified
 */
KLIMT_API void glDepthMask(GLboolean nSet)
{
	currentContext->glDepthMask(nSet);
}


/// glCullFace specifies whether front- or back-facing facets can be culled
/**
 *   Works as specified
 */
KLIMT_API void glCullFace(GLenum nCull)
{
	currentContext->glCullFace(nCull);
}


/// glFrontFace defines front- and back-facing polygons
/**
 *   Works as specified
 */
KLIMT_API void glFrontFace(GLenum nFace)
{
	currentContext->glFrontFace(nFace);
}


/// glGetBooleanv returns the value or values of a selected parameter
/**
 *   Only GL_RGBA_MODE is supported
 */
KLIMT_API void glGetBooleanv(GLenum nType, GLboolean *nParams)
{
	currentContext->glGetBooleanv(nType, nParams);
}


/// glGetFloatv returns the value or values of a selected parameter
/**
 *   Supported parameters for nType:
 *   - GL_PROJECTION_MATRIX
 *   - GL_MODELVIEW_MATRIX
 *   - GL_LIGHT_MODEL_AMBIENT
 *   - GL_CURRENT_COLOR
 *   - GL_LINE_WIDTH_RANGE
 *   - GL_POINT_SIZE_RANGE
 */
KLIMT_API void glGetFloatv(GLenum nType, GLfloat* nParams)
{
	currentContext->glGetFloatv(nType, nParams);
}


/// glGetIntegerv returns the value or values of a selected parameter
/**
 *   Supported parameters for nType:
 *   - GL_FRONT_FACE
 *   - GL_CULL_FACE_MODE
 *   - GL_LIGHT_MODEL_AMBIENT
 *   - GL_MAX_LIGHTS
 *   - GL_RED_BITS
 *   - GL_GREEN_BITS
 *   - GL_BLUE_BITS
 *   - GL_ALPHA_BITS
 *   - GL_MAX_VIEWPORT_DIMS
 *   - GL_MAX_CLIP_PLANES
 *   - GL_MAX_TEXTURE_SIZE
 */
KLIMT_API void glGetIntegerv(GLenum nType, GLint* nParams)
{
	currentContext->glGetIntegerv(nType, nParams);
}


/// glGetString returns the value or values of a selected parameter
/**
 *   Supported parameters:
 *   - GL_VENDOR
 *   - GL_RENDERER
 *   - GL_VERSION
 *   - GL_EXTENSIONS
 */
KLIMT_API const GLubyte* glGetString(GLenum nName)
{
	return currentContext->glGetString(nName);
}


/// glColorMask enables and disables writing of frame-buffer color components
/**
 *   The alpha value is ignored.
 */
KLIMT_API void glColorMask(GLboolean nRed, GLboolean nGreen, GLboolean nBlue, GLboolean nAlpha)
{
	currentContext->glColorMask(nRed, nGreen, nBlue, nAlpha);
}


/// glScissor defines the scissor box
/**
 *   Works as specified
 */
KLIMT_API void glScissor(GLint x, GLint y, GLsizei width, GLsizei height)
{
	currentContext->glScissor(x, y, width, height);
}


/// glFogi specifies fog parameters
/**
*   Only linear fog is supported.
*/
KLIMT_API void glFogi(GLenum pname, GLint param)
{
	currentContext->glFogi(pname, param);
}


/// glFogfv specifies fog parameters
/**
*    Only linear fog is supported.
*/
KLIMT_API void glFogf(GLenum pname, GLfloat param)
{
	currentContext->glFogf(pname, param);
}


/// glFogfv specifies fog parameters
/**
*    Only linear fog is supported.
*/
KLIMT_API void glFogfv(GLenum pname, const GLfloat *params)
{
	currentContext->glFogfv(pname, params);
}


/// glFogiv specifies fog parameters
/**
*    Only linear fog is supported.
*/
KLIMT_API void glFogiv(GLenum pname, const GLint *params)
{
	currentContext->glFogiv(pname, params);
}


/// glBlendFunc specifies pixel arithmetic
/**
 *   glBlendFunc works with all parameters except destination alpha
 *   (there is no alpha buffer support at the moment in Klimt)
 *
 */
KLIMT_API void glBlendFunc(GLenum sfactor, GLenum dfactor)
{
	currentContext->glBlendFunc(sfactor, dfactor);
}


/// glDepthFunc specifies the value used for depth-buffer comparisons
/**
 *   Works as specified
 */
KLIMT_API void glDepthFunc(GLenum nFunc)
{
	currentContext->glDepthFunc(nFunc);
}


// wrapper methods implemented in klContext_oglLight.h
//


/// glLightfv sets light-source parameters
/**
 *   Spot lights and attenuation are not supported
 */
KLIMT_API void glLightfv(GLenum nLight, GLenum nSelect, const GLfloat* nValues)
{
	currentContext->glLightfv(nLight, nSelect, nValues);
}


/// glLightxv sets light-source parameters
/**
 *   Spot lights and attenuation are not supported
 */
KLIMT_API void glLightxv(GLenum nLight, GLenum nSelect, const GLfixed* nValues)
{
	currentContext->glLightxv(nLight, nSelect, nValues);
}


/// glLightModeli sets the lighting model parameters
/**
 *   GL_LIGHT_MODEL_TWO_SIDE is not supported
 */
KLIMT_API void glLightModeli(GLenum nParam, GLint nValue)
{
	currentContext->glLightModeli(nParam, nValue);
}


/// glLightModelf sets the lighting model parameters
/**
 *   GL_LIGHT_MODEL_TWO_SIDE is not supported
 */
KLIMT_API void glLightModelf(GLenum nParam, GLfloat nValue)
{
	currentContext->glLightModelf(nParam, nValue);
}


/// glLightModelx sets the lighting model parameters
/**
*   GL_LIGHT_MODEL_TWO_SIDE is not supported
*/
KLIMT_API void glLightModelx(GLenum nParam, GLfixed nValue)
{
	currentContext->glLightModelx(nParam, nValue);
}


/// glLightModelfv sets the lighting model parameters
/**
 *   Only GL_LIGHT_MODEL_AMBIENT is supported
 */
KLIMT_API void glLightModelfv(GLenum nParam, const GLfloat* nValues)
{
	currentContext->glLightModelfv(nParam, nValues);
}


/// glLightModelxv sets the lighting model parameters
/**
*   Only GL_LIGHT_MODEL_AMBIENT is supported
*/
KLIMT_API void glLightModelxv(GLenum nParam, const GLfixed* nValues)
{
	currentContext->glLightModelxv(nParam, nValues);
}


/// glShadeModel selects flat or smooth shading
/**
 *   Works as specified
 */
KLIMT_API void glShadeModel(GLenum nModel)
{
	currentContext->glShadeModel(nModel);
}


// wrapper methods implemented in klContext_oglMaterial.h
//


/// glMaterialf specifies material parameters for the lighting model
/**
 *   Works as specified
 */
KLIMT_API void glMaterialf(GLenum nSide, GLenum nSelect, GLfloat nValue)
{
	currentContext->glMaterialf(nSide, nSelect, nValue);
}


/// glMaterialx specifies material parameters for the lighting model
/**
 *   Works as specified
 */
KLIMT_API void glMaterialx(GLenum nSide, GLenum nSelect, GLfixed nValue)
{
	currentContext->glMaterialx(nSide, nSelect, nValue);
}


/// glMaterialfv specifies material parameters for the lighting model
/**
 *   GL_COLOR_INDEXES is not supported
 */
KLIMT_API void glMaterialfv(GLenum nSide, GLenum nSelect, const GLfloat* nValues)
{
	currentContext->glMaterialfv(nSide, nSelect, nValues);
}


/// glMaterialiv specifies material parameters for the lighting model
/**
 *   GL_COLOR_INDEXES is not supported
 */
KLIMT_API void glMaterialiv(GLenum nSide, GLenum nSelect, const GLint* nValues)
{
	currentContext->glMaterialiv(nSide, nSelect, nValues);
}


/// glMaterialxv specifies material parameters for the lighting model
/**
*   GL_COLOR_INDEXES is not supported
*/
KLIMT_API void glMaterialxv(GLenum nSide, GLenum nSelect, const GLfixed* nValues)
{
	currentContext->glMaterialxv(nSide, nSelect, nValues);
}


/// glColorMaterial causes a material color to track the current color
/**
 *   Works as specified
 */
KLIMT_API void glColorMaterial(GLenum nSide, GLenum nMode)
{
	currentContext->glColorMaterial(nSide, nMode);
}


/// glGetMaterialfv returns material parameters
/**
 *   GL_COLOR_INDEXES is not supported
 */
KLIMT_API void glGetMaterialfv(GLenum nSide, GLenum nSelect, GLfloat* nValues)
{
	currentContext->glGetMaterialfv(nSide, nSelect, nValues);
}


/// glGetMaterialiv returns material parameters
/**
 *   GL_COLOR_INDEXES is not supported
 */
KLIMT_API void glGetMaterialiv(GLenum nSide, GLenum nSelect, GLint* nValues)
{
	currentContext->glGetMaterialiv(nSide, nSelect, nValues);
}


/// glGetMaterialxv returns material parameters
/**
 *   GL_COLOR_INDEXES is not supported
 */
KLIMT_API void glGetMaterialxv(GLenum nSide, GLenum nSelect, GLfixed* nValues)
{
	currentContext->glGetMaterialxv(nSide, nSelect, nValues);
}


// wrapper methods implemented in klContext_oglPixels.h
//


/// glDrawPixels writes a block of pixels to the framebuffer
/**
 *   - Only GL_RGBA for nFormat (alpha value ignored) and
 *     GL_UNSIGNED_BYTE for nType are supported.
 *   - nWidth currently has to be full screen width.
 *   - glDrawPixels is currently only draws
 *     full screen images.
 */
KLIMT_API void glDrawPixels(GLsizei nWidth, GLsizei nHeight, GLenum nFormat, GLenum nType, const GLvoid *nPixels)
{
	currentContext->glDrawPixels(nWidth, nHeight, nFormat, nType, nPixels);
}


/// glReadPixels is silently ignored
KLIMT_API void glReadPixels(GLint x, GLint y, GLsizei width, GLsizei height, GLenum format, GLenum type, GLvoid *pixels)
{
	currentContext->glReadPixels(x, y, width, height, format, type, pixels);
}


// wrapper methods implemented in klContext_oglBeginEnd.h
//

/// glBegin delimits the vertices of a primitive or a group of like primitives
/**
 *   Works with all primitve types as specified
 */
KLIMT_API void glBegin(GLenum nType)
{
	currentContext->glBegin(nType);
}


/// glEnd delimits the vertices of a primitive or a group of like primitives
/**
 *   Works with all primitve types as specified
 */
KLIMT_API void glEnd()
{
	currentContext->glEnd();
}


/// glColor3f sets the current color
/**
 *   Works as specified
 */
KLIMT_API void glColor3f(GLfloat nR, GLfloat nG, GLfloat nB)
{
	currentContext->glColor3f(nR, nG, nB);
}


/// glColor3fv sets the current color
/**
 *   Works as specified
 */
KLIMT_API void glColor3fv(const GLfloat *nV)
{
	currentContext->glColor3fv(nV);
}


/// glColor3i sets the current color
/**
 *   Works as specified
 */
KLIMT_API void glColor3i(GLint nR, GLint nG, GLint nB)
{
	currentContext->glColor3i(nR, nG, nB);
}


/// glColor3iv sets the current color
/**
 *   Works as specified
 */
KLIMT_API void glColor3iv(const GLint* nV)
{
	currentContext->glColor3iv(nV);
}


/// glColor3ub sets the current color
/**
 *   Works as specified
 */
KLIMT_API void glColor3ub(GLubyte nR, GLubyte nG, GLubyte nB)
{
	currentContext->glColor3ub(nR, nG, nB);
}


/// glColor3ubv sets the current color
/**
 *   Works as specified
 */
KLIMT_API void glColor3ubv(const GLubyte *nV)
{
	currentContext->glColor3ubv(nV);
}


/// glColor3x sets the current color
/**
 *   Works as specified
 */
KLIMT_API void glColor3x(GLfixed nR, GLfixed nG, GLfixed nB)
{
	currentContext->glColor3x(nR, nG, nB);
}


/// glColor3xv sets the current color
/**
 *   Works as specified
 */
KLIMT_API void glColor3xv(const GLfixed *nV)
{
	currentContext->glColor3xv(nV);
}


/// glColor4f sets the current color
/**
 *   Works as specified
 */
KLIMT_API void glColor4f(GLfloat nR, GLfloat nG, GLfloat nB, GLfloat nA)
{
	currentContext->glColor4f(nR, nG, nB, nA);
}


/// glColor4fv sets the current color
/**
 *   Works as specified
 */
KLIMT_API void glColor4fv(const GLfloat *nV)
{
	currentContext->glColor4fv(nV);
}


/// glColor4i sets the current color
/**
 *   Works as specified
 */
KLIMT_API void glColor4i(GLint nR, GLint nG, GLint nB, GLint nA)
{
	currentContext->glColor4i(nR, nG, nB, nA);
}


/// glColor4iv sets the current color
/**
 *   Works as specified
 */
KLIMT_API void glColor4iv(const GLint* nV)
{
	currentContext->glColor4iv(nV);
}


/// glColor4ub sets the current color
/**
 *   Works as specified
 */
KLIMT_API void glColor4ub(GLubyte nR, GLubyte nG, GLubyte nB, GLubyte nA)
{
	currentContext->glColor4ub(nR, nG, nB, nA);
}


/// glColor4ubv sets the current color
/**
 *   Works as specified
 */
KLIMT_API void glColor4ubv(const GLubyte *nV)
{
	currentContext->glColor4ubv(nV);
}


/// glColor4x sets the current color
/**
 *   Works as specified
 */
KLIMT_API void glColor4x(GLfixed nR, GLfixed nG, GLfixed nB, GLfixed nA)
{
	currentContext->glColor4x(nR, nG, nB, nA);
}


/// glColor4xv sets the current color
/**
 *   Works as specified
 */
KLIMT_API void glColor4xv(const GLfixed *nV)
{
	currentContext->glColor4xv(nV);
}


/// glNormal3f sets the current normal vector
/**
 *   Works as specified
 */
KLIMT_API void glNormal3f(GLfloat nX, GLfloat nY, GLfloat nZ)
{
	currentContext->glNormal3f(nX, nY, nZ);
}


/// glNormal3fv sets the current normal vector
/**
 *   Works as specified
 */
KLIMT_API void glNormal3fv(const GLfloat *nV)
{
	currentContext->glNormal3fv(nV);
}


/// glNormal3d sets the current normal vector
/**
 *   Works as specified
 */
KLIMT_API void glNormal3d(GLdouble nX, GLdouble nY, GLdouble nZ)
{
	currentContext->glNormal3d(nX, nY, nZ);
}


/// glNormal3dv sets the current normal vector
/**
 *   Works as specified
 */
KLIMT_API void glNormal3dv(const GLdouble *nV)
{
	currentContext->glNormal3dv(nV);
}


/// glNormal3x sets the current normal vector
/**
 *   Works as specified
 */
KLIMT_API void glNormal3x(GLfixed nX, GLfixed nY, GLfixed nZ)
{
	currentContext->glNormal3x(nX, nY, nZ);
}


/// glNormal3xv sets the current normal vector
/**
 *   Works as specified
 */
KLIMT_API void glNormal3xv(const GLfixed *nV)
{
	currentContext->glNormal3xv(nV);
}


/// glTexCoord2f sets the current texture coordinates
/**
 *   Works as specified
 */
KLIMT_API void glTexCoord2f(GLfloat nX, GLfloat nY)
{
	currentContext->glTexCoord2f(nX, nY);
}


/// glTexCoord2fv sets the current texture coordinates
/**
 *   Works as specified
 */
KLIMT_API void glTexCoord2fv(const GLfloat *nV)
{
	currentContext->glTexCoord2fv(nV);
}


/// glTexCoord3f sets the current texture coordinates
/**
 *   Works as specified
 */
KLIMT_API void glTexCoord3f(GLfloat nX, GLfloat nY, GLfloat nZ)
{
	currentContext->glTexCoord3f(nX, nY, nZ);
}


/// glTexCoord3fv sets the current texture coordinates
/**
 *   Works as specified
 */
KLIMT_API void glTexCoord3fv(const GLfloat *nV)
{
	currentContext->glTexCoord3fv(nV);
}


/// glTexCoord4f sets the current texture coordinates
/**
 *   Works as specified
 */
KLIMT_API void glTexCoord4f(GLfloat nX, GLfloat nY, GLfloat nZ, GLfloat nW)
{
	currentContext->glTexCoord4f(nX, nY, nZ, nW);
}


/// glTexCoord4fv sets the current texture coordinates
/**
 *   Works as specified
 */
KLIMT_API void glTexCoord4fv(const GLfloat *nV)
{
	currentContext->glTexCoord4fv(nV);
}


/// glTexCoord2d sets the current texture coordinates
/**
 *   Works as specified
 */
KLIMT_API void glTexCoord2d(GLdouble nX, GLdouble nY)
{
	currentContext->glTexCoord2d(nX, nY);
}


/// glTexCoord2dv sets the current texture coordinates
/**
 *   Works as specified
 */
KLIMT_API void glTexCoord2dv(const GLdouble *nV)
{
	currentContext->glTexCoord2dv(nV);
}


/// glTexCoord3d sets the current texture coordinates
/**
 *   Works as specified
 */
KLIMT_API void glTexCoord3d(GLdouble nX, GLdouble nY, GLdouble nZ)
{
	currentContext->glTexCoord3d(nX, nY, nZ);
}


/// glTexCoord3dv sets the current texture coordinates
/**
 *   Works as specified
 */
KLIMT_API void glTexCoord3dv(const GLdouble *nV)
{
	currentContext->glTexCoord3dv(nV);
}


/// glTexCoord4d sets the current texture coordinates
/**
 *   Works as specified
 */
KLIMT_API void glTexCoord4d(GLdouble nX, GLdouble nY, GLdouble nZ, GLdouble nW)
{
	currentContext->glTexCoord4d(nX, nY, nZ, nW);
}


/// glTexCoord4dv sets the current texture coordinates
/**
 *   Works as specified
 */
KLIMT_API void glTexCoord4dv(const GLdouble *nV)
{
	currentContext->glTexCoord4dv(nV);
}


/// glTexCoord2x sets the current texture coordinates
/**
 *   Works as specified
 */
KLIMT_API void glTexCoord2x(GLfixed nX, GLfixed nY)
{
	currentContext->glTexCoord2x(nX, nY);
}


/// glTexCoord2xv sets the current texture coordinates
/**
 *   Works as specified
 */
KLIMT_API void glTexCoord2xv(const GLfixed *nV)
{
	currentContext->glTexCoord2xv(nV);
}


/// glTexCoord3x sets the current texture coordinates
/**
 *   Works as specified
 */
KLIMT_API void glTexCoord3x(GLfixed nX, GLfixed nY, GLfixed nZ)
{
	currentContext->glTexCoord3x(nX, nY, nZ);
}


/// glTexCoord3xv sets the current texture coordinates
/**
 *   Works as specified
 */
KLIMT_API void glTexCoord3xv(const GLfixed *nV)
{
	currentContext->glTexCoord3xv(nV);
}


/// glTexCoord4x sets the current texture coordinates
/**
 *   Works as specified
 */
KLIMT_API void glTexCoord4x(GLfixed nX, GLfixed nY, GLfixed nZ, GLfixed nW)
{
	currentContext->glTexCoord4x(nX, nY, nZ, nW);
}


/// glTexCoord4xv sets the current texture coordinates
/**
 *   Works as specified
 */
KLIMT_API void glTexCoord4xv(const GLfixed *nV)
{
	currentContext->glTexCoord4xv(nV);
}


/// glVertex2s specifies a vertex
/**
 *   Works as specified
 */
KLIMT_API void glVertex2s(GLshort x, GLshort y)
{
	currentContext->glVertex2s(x, y);
}


/// glVertex2f specifies a vertex
/**
 *   Works as specified
 */
KLIMT_API void glVertex2f(GLfloat x, GLfloat y)
{
	currentContext->glVertex2f(x, y);
}


/// glVertex3x specifies a vertex
/**
 *   Works as specified
 */
KLIMT_API void glVertex3x(GLfixed nX, GLfixed nY, GLfixed nZ)
{
	currentContext->glVertex3x(nX, nY, nZ);
}


/// glVertex3f specifies a vertex
/**
 *   Works as specified
 */
KLIMT_API void glVertex3f(GLfloat nX, GLfloat nY, GLfloat nZ)
{
	currentContext->glVertex3f(nX, nY, nZ);
}


/// glVertex3fv specifies a vertex
/**
 *   Works as specified
 */
KLIMT_API void glVertex3fv(const GLfloat *nVertex)
{
	currentContext->glVertex3fv(nVertex);
}


/// glVertex3d specifies a vertex
/**
 *   Works as specified
 */
KLIMT_API void glVertex3d(GLdouble nX, GLdouble nY, GLdouble nZ)
{
	currentContext->glVertex3d(nX, nY, nZ);
}


/// glVertex4fv specifies a vertex
/**
 *   Works as specified
 */
KLIMT_API void glVertex4fv(const GLfloat *nVertex)
{
	currentContext->glVertex4fv(nVertex);
}


// wrapper methods implemented in klContext_oglTexture.h
//


/// glGenTextures generates texture names
/**
 *   Works as specified
 */
KLIMT_API void glGenTextures(GLsizei n, GLuint *textures)
{
	currentContext->glGenTextures(n, textures);
}


/// glDeleteTextures deletes named textures
/**
 *   Works as specified
 */
KLIMT_API void glDeleteTextures(GLsizei n, const GLuint *textures)
{
	currentContext->glDeleteTextures(n, textures);
}


/// glBindTexture enables the creation of a named texture that is bound to a texture target
/**
 *   nTarget must be GL_TEXTURE_2D.
 */
KLIMT_API void glBindTexture(GLenum nTarget, GLuint nTexture)
{
	currentContext->glBindTexture(nTarget, nTexture);
}


/// glGetTexLevelParameteriv returns texture parameter values for a specific level of detail
/**
 *   target must be GL_TEXTURE_2D.\n
 *   level is ignored.\n
 *   pname is allowed to be
 *   - GL_TEXTURE_WIDTH
 *   - GL_TEXTURE_HEIGHT
 *   - GL_TEXTURE_INTERNAL_FORMAT
 */
KLIMT_API void glGetTexLevelParameteriv(GLenum target, GLint level, GLenum pname, GLint *params)
{
	currentContext->glGetTexLevelParameteriv(target, level, pname, params);
}


/// glTexImage2D specifies a two-dimensional texture image
/**
 *   glTexImage2D works with the following restrictions:
 *   - glTexImage2D accepts only RGB24 or RGBX32 images.
 *   - nTarget must be GL_TEXTURE_2D.
 *   - nInternalFormat must 3 or 4.
 *   - nBorder must is ignored.
 *   - nFormat must be GL_RGB or GL_RGBA.
 *   - nPixelType must be GL_UNSIGNED_BYTE
 */
KLIMT_API void glTexImage2D(GLenum target, GLint level, GLint internalformat, GLsizei width, GLsizei height,
							 GLint border, GLenum format, GLenum type, const GLvoid *pixels)
{
	currentContext->glTexImage2D(target, level, internalformat, width, height, border, format, type, pixels);
}


/// glTexEnvi sets texture environment parameters
/**
*   Only GL_MODULATE and GL_DECAL are supported
*/
KLIMT_API void glTexEnvi(GLenum target, GLenum pname, GLint param)
{
	currentContext->glTexEnvi(target, pname, param);
}


/// glTexEnvf sets texture environment parameters
/**
*   Only GL_MODULATE and GL_DECAL are supported
*/
KLIMT_API void glTexEnvf(GLenum target, GLenum pname, GLfloat param)
{
	currentContext->glTexEnvf(target, pname, param);
}


/// glTexEnvx sets texture environment parameters
/**
*   Only GL_MODULATE and GL_DECAL are supported
*/
KLIMT_API void glTexEnvx(GLenum target, GLenum pname, GLfixed param)
{
	currentContext->glTexEnvx(target, pname, param);
}


// wrapper methods implemented in klContext_oglDisplayList.h
//


/// glCallLists executes a list of display lists
/**
 *   Works as specified
 */
KLIMT_API void glCallLists(GLsizei n, GLenum type, const GLvoid *lists)
{
	currentContext->glCallLists(n, type, lists);
}


/// glCallList executes a display list
/**
 *   Works as specified
 */
KLIMT_API void glCallList(GLuint list)
{
	currentContext->glCallList(list);
}


/// glGenLists generates a contiguous set of empty display lists
/**
 *   Works as specified
 */
KLIMT_API GLuint glGenLists(GLsizei range)
{
	return currentContext->glGenLists(range);
}


/// glDeleteLists deletes a contiguous group of display lists
/**
 *   Works as specified
 */
KLIMT_API void glDeleteLists(GLuint list, GLsizei range)
{
	currentContext->glDeleteLists(list, range);
}


/// glNewList creates or replaces a display list
/**
 *   Works as specified
 */
KLIMT_API void glNewList(GLuint list, GLenum mode)
{
	currentContext->glNewList(list, mode);
}


/// glEndList closes display list compiling
/**
 *   Works as specified
 */
KLIMT_API void glEndList()
{
	currentContext->glEndList();
}


/// glListBase sets the display list base for glCallLists
KLIMT_API void glListBase(GLuint base)
{
	currentContext->glListBase(base);
}


/// gllsList tests for display list existence
KLIMT_API GLboolean glIsList(GLuint list)
{
	return currentContext->glIsList(list);
}



// Klimt "extensions"
//


/// klCreateDefaultContext creates a render context
/**
 *   Use this method in case you don't want to use
 *   wgl or egl methods.
 */
KLIMT_API void klCreateDefaultContext()
{
	if(currentContext)
		delete currentContext;
	currentContext = new klContext(new klRasterizer);
}




/// klSetBufferSize sets the size of the color and depth buffer
KLIMT_API void klSetBufferSize(int nWidth, int nHeight)
{
	currentContext->klSetBufferSize(nWidth, nHeight);
}


/// klSetColorBuffer sets the color buffer Klimt renders to
/**
 *   The color buffer must be large enough to hold all
 *   pixels as specified in klSetBufferSize.\n
 *   This works only with the software rasterizer.
 */
KLIMT_API void klSetColorBuffer(void* nMem)
{
	currentContext->klSetColorBuffer(nMem);
}


/// klSetColorBuffer sets the color buffer Klimt renders to
/**
 *   This works only with the software rasterizer.
 */
KLIMT_API void* klGetColorBuffer()
{
	return currentContext->klGetColorBuffer();
}


KLIMT_API void klTweak(GLenum nSelect, GLboolean nEnable)
{
	currentContext->klTweak(nSelect, nEnable);
}

KLIMT_API void klPrepareVertexArrays(int nNumVertices, int nNumNormals, int nNumColors, int nNumTexCoords,
									 void* nVertices, void* nColors, void* nTexCoords)
{
	currentContext->klPrepareVertexArrays(nNumVertices, nNumNormals, nNumColors, nNumTexCoords,
										  nVertices, nColors, nTexCoords);
}


KLIMT_API void klRenderPreparedIndexedArray(GLenum nPrimType, GLsizei nCount,
										    const unsigned int* nVertexIndices, const unsigned int* nColorIndices, const unsigned int* nTexCoordsIndices,
											const void* nVertices, const void* nColors, const void* nTexCoords)
{
	currentContext->klRenderPreparedIndexedArray(nPrimType, nCount, nVertexIndices, nColorIndices, nTexCoordsIndices,
												 nVertices, nColors, nTexCoords);
}



/////////////////////////////////////////////////////////////////////////////
//
//       the following API functions are currently not implemented
//
//    The empty wrapper functions allow to link against them, although they
//    won't do anything...
//



/// glTexParameteri is silently ignored
KLIMT_API void glTexParameteri(GLenum /*nTarget*/, GLenum /*nPName*/, GLint /*nParam*/)
{
	//report(reportMode, reportSelector, KL_REPORT_IGNORED, "glTexParameteri(): this method call is ignored");
}


/// glTexParameterf is silently ignored
KLIMT_API void glTexParameterf(GLenum /*nTarget*/, GLenum /*nPName*/, GLfloat /*nParam*/)
{
	//report(reportMode, reportSelector, KL_REPORT_IGNORED, "glTexParameterf(): this method call is ignored");
}


/// glTexEnvfv is silently ignored
KLIMT_API void glTexEnvfv(GLenum target, GLenum pname, const GLfloat *params)
{
	//report(reportMode, reportSelector, KL_REPORT_IGNORED, "glTexEnvfv(): this method call is ignored");
}


/// glTexGenf is silently ignored
KLIMT_API void glTexGenf(GLenum coord, GLenum pname, GLfloat param)
{
	//report(reportMode, reportSelector, KL_REPORT_IMPORTANT, "glTexGenf(): this method call is ignored");
}


/// glTexGeni is silently ignored
KLIMT_API void glTexGeni(GLenum coord, GLenum pname, GLint param)
{
	//report(reportMode, reportSelector, KL_REPORT_IMPORTANT, "glTexGeni(): this method call is ignored");
}


/// glTexGenfv is silently ignored
KLIMT_API void glTexGenfv(GLenum coord, GLenum pname, const GLfloat *params)
{
	//report(reportMode, reportSelector, KL_REPORT_IMPORTANT, "glTexGenfv(): this method call is ignored");
}


/// glAccum is silently ignored
KLIMT_API void glAccum(GLenum nOp, GLfloat nValue)
{
	//report(reportMode, reportSelector, KL_REPORT_IGNORED, "glAccum(): this method call is ignored");
}


/// glLightf is silently ignored
KLIMT_API void glLightf(GLenum nLight, GLenum nParamName, GLfloat nParam)
{
	//report(reportMode, reportSelector, KL_REPORT_IGNORED2, "glLightf(): this method call is ignored");
}


/// glLightx is silently ignored
KLIMT_API void glLightx(GLenum nLight, GLenum nParamName, GLfixed nParam)
{
	//report(reportMode, reportSelector, KL_REPORT_IGNORED2, "glLightx(): this method call is ignored");
}


/// glPointSize is silently ignored
KLIMT_API void glPointSize(GLfloat nSize)
{
	//report(reportMode, reportSelector, KL_REPORT_IGNORED, "glPointSize(): this method call is ignored");
}


/// glLineWidth is silently ignored
KLIMT_API void glLineWidth(GLfloat nWidth)
{
	//report(reportMode, reportSelector, KL_REPORT_IGNORED, "glLineWidth(): this method call is ignored");
}


/// glLineStipple is silently ignored
KLIMT_API void glLineStipple(GLint factor, GLushort pattern)
{
	//report(reportMode, reportSelector, KL_REPORT_IGNORED, "glLineStipple(): this method call is ignored");
}


/// glPolygonStipple is silently ignored
KLIMT_API void glPolygonStipple(const GLubyte *mask)
{
	//report(reportMode, reportSelector, KL_REPORT_IGNORED, "glPolygonStipple(): this method call is ignored");
}


/// glRasterPos3f is silently ignored
KLIMT_API void glRasterPos3f(GLfloat x, GLfloat y, GLfloat z)
{
	//report(reportMode, reportSelector, KL_REPORT_IMPORTANT, "glRasterPos3f(): this method call is not implemented");
}


/// glPixelStorei is silently ignored
KLIMT_API void glPixelStorei(GLenum pname, GLint param)
{
	//report(reportMode, reportSelector, KL_REPORT_IMPORTANT, "glPixelStorei(): this method call is not implemented");
}


/// glBitmap is silently ignored
KLIMT_API void glBitmap(GLsizei width, GLsizei height, GLfloat xorig, GLfloat yorig, GLfloat xmove, GLfloat ymove, const GLubyte *bitmap)
{
	//report(reportMode, reportSelector, KL_REPORT_IMPORTANT, "glBitmap(): this method call is not implemented");
}


/// glClearIndex is silently ignored
KLIMT_API void glClearIndex(GLfloat c)
{
	//report(reportMode, reportSelector, KL_REPORT_IMPORTANT, "glClearIndex(): this method call is not implemented");
}


/// glIndexi is silently ignored
KLIMT_API void glIndexi(GLint c)
{
	//report(reportMode, reportSelector, KL_REPORT_IMPORTANT, "glIndexi(): this method call is not implemented");
}


/// glViewport is silently ignored
KLIMT_API void glViewport(GLint x, GLint y, GLsizei width, GLsizei height)
{
	//report(reportMode, reportSelector, KL_REPORT_IGNORED2, "glViewport(): this method call is not implemented");
}


/// glClipPlane is silently ignored
KLIMT_API void glClipPlane(GLenum plane, const GLdouble *equation)
{
	//report(reportMode, reportSelector, KL_REPORT_UNIMPORTANT, "glClipPlane(): this method call is not implemented");
}


/// glAlphaFunc is silently ignored
KLIMT_API void glAlphaFunc(GLenum func, GLclampf ref)
{
	//report(reportMode, reportSelector, KL_REPORT_UNIMPORTANT, "glAlphaFunc(): this method call is not implemented");
}


#ifdef __cplusplus
}
#endif
