// parse_frame.hpp
// ---------------
//
//  (C) Copyright Gerald Thaler 2008.
//
// Distributed under the Boost Software License, Version 1.0.
//    (See accompanying file LICENSE_1_0.txt or copy at
//          http://www.boost.org/LICENSE_1_0.txt)

#ifndef INTREPID_PARSE_FRAME_HPP
#define INTREPID_PARSE_FRAME_HPP

#if defined(_MSC_VER) && (_MSC_VER >= 1200)
# pragma once
#endif // defined(_MSC_VER) && (_MSC_VER >= 1200)

#include "../comm_model.hpp"

#include "geo.hpp"

namespace intrepid
{
    namespace vram_object_type
    {
        int const asteroid1  =  1;
        int const asteroid2  =  2;
        int const asteroid3  =  3;
        int const asteroid4  =  4;
        int const ship       =  5;
        int const saucer     =  6;
        int const shot       =  7;
        int const explosion1 =  8;
        int const explosion2 =  9;
        int const explosion3 = 10;
        int const explosion4 = 11;
    }

    struct vectorram_object
    {
        vectorram_object(point pos, int obj_type, int s);
        point pos;
        int obj_type, s;
    };

    class parsed_frame
    {
    public:
        typedef vector<vectorram_object>::const_iterator iterator;

        parsed_frame();

        iterator begin() const;
        iterator end() const;
        bool even() const;
        uint8_t get_frameno() const;
        uint8_t get_next_ping() const;
        uint8_t get_ping() const;
        point get_ship_orientation() const;
        comm_model::ping_to_keys_map const &get_ping_to_keys() const;
        bool ship_present() const;
        size_t size() const;

        void add(point pos, int obj_type, int s);
        void clear();
        void set_current_keys(uint8_t ping, uint8_t keys);
        void set_even(bool even);
        void set_frameno(uint8_t frameno);
        void set_ping(uint8_t ping);
        void set_ping_to_keys(
                            comm_model::ping_to_keys_map const &ping_to_keys);
        void set_ship_present(point orientation);

#ifdef INTREPID_WITH_ASTEROIDS_RAM
        array<uint8_t, comm_model::asteroids_ram_size> asteroids_ram;
#endif

    private:
        // copying is undesirable
        parsed_frame(parsed_frame const &); // = delete;
        parsed_frame &operator=(parsed_frame const &); // = delete;

        vector<vectorram_object>     objects_;
        bool                         even_;
        uint8_t                      frameno_, ping_, next_ping_;
        comm_model::ping_to_keys_map ping_to_keys_;
        bool                         ship_present_;
        point                        ship_orientation_;
    };

    void parse_frame(comm_model::frame_packet const &frame_packet,
                     comm_model::ping_to_keys_map const &ping_to_keys,
                     parsed_frame &frame);

// struct vectorram_object

    inline vectorram_object::vectorram_object(point pos,
                                              int obj_type, int s)
    {
        this->pos = pos;
        this->obj_type = obj_type;
        this->s = s;
    }

// class parsed_frame
// public:

    inline parsed_frame::parsed_frame()
    {
        objects_.reserve(35);
        clear();
    }

    inline parsed_frame::iterator parsed_frame::begin() const
    {
        return objects_.begin();
    }

    inline parsed_frame::iterator parsed_frame::end() const
    {
        return objects_.end();
    }

    inline bool parsed_frame::even() const
    {
        return even_;
    }

    inline uint8_t parsed_frame::get_frameno() const
    {
        return frameno_;
    }

    inline uint8_t parsed_frame::get_next_ping() const
    {
        return next_ping_;
    }

    inline uint8_t parsed_frame::get_ping() const
    {
        return ping_;
    }

    inline
    comm_model::ping_to_keys_map const &parsed_frame::get_ping_to_keys() const
    {
        return ping_to_keys_;
    }

    inline point parsed_frame::get_ship_orientation() const
    {
        return ship_orientation_;
    }

    inline bool parsed_frame::ship_present() const
    {
        return ship_present_;
    }

    inline size_t parsed_frame::size() const
    {
        return objects_.size();
    }

    inline void parsed_frame::add(point pos, int obj_type, int s)
    {
        objects_.push_back(vectorram_object(pos, obj_type, s));
    }

    inline void parsed_frame::clear()
    {
        objects_.clear();
        ship_present_ = false;
    }

    inline void parsed_frame::set_even(bool even)
    {
        even_ = even;
    }

    inline void parsed_frame::set_frameno(uint8_t frameno)
    {
        frameno_ = frameno;
    }

    inline void parsed_frame::set_ping(uint8_t ping)
    {
        ping_ = ping;
    }

    inline void parsed_frame::set_ping_to_keys(
                            comm_model::ping_to_keys_map const &ping_to_keys)
    {
        ping_to_keys_ = ping_to_keys;
    }

    inline void parsed_frame::set_ship_present(point orientation)
    {
        ship_present_ = true;
        ship_orientation_ = orientation;
    }

    inline void parsed_frame::set_current_keys(uint8_t ping, uint8_t keys)
    {
        ping_to_keys_[ping] = keys;
        next_ping_ = ping + 1;
    }
} // end of namespace intrepid

#endif // include guard
