// player_impl.cpp
// --------------
//
//  (C) Copyright Gerald Thaler 2008.
//
// Distributed under the Boost Software License, Version 1.0.
//    (See accompanying file LICENSE_1_0.txt or copy at
//          http://www.boost.org/LICENSE_1_0.txt)

#include "stdafx.hpp"

#include "player_impl.hpp"

#define foreach BOOST_FOREACH

namespace intrepid
{
// class player_impl
// public:

    player_impl::player_impl()
        :   strategy_(screen_state_)
    {
        threads_ = thread_range(thread_container_.begin(),
                                thread_container_.begin());
    }

    uint8_t player_impl::next_frame(uint8_t ping,
                            comm_model::frame_packet const &frame_packet,
                            comm_model::ping_to_keys_map const &ping_to_keys)
    {
        // Request workers to finish.
        thread_comm_.set_finish();

        // Parse the vectorram in this thread.
        parse_frame(frame_packet, ping_to_keys, parsed_frame_);
#ifdef INTREPID_WITH_ASTEROIDS_RAM
        copy(boost::begin(frame_packet.asteroids_ram),
             boost::end(frame_packet.asteroids_ram),
             parsed_frame_.asteroids_ram.begin());
#endif

        // Retrieve the result.
        uint8_t keys = thread_comm_.get_result();
        parsed_frame_.set_current_keys(ping, keys);

        // Push the new frame.
        thread_comm_.push_frame();

        return keys;
    }

    void player_impl::start()
    {
        try
        {
            nof_threads_ = get_nof_threads();
            screen_state_.reset();
            thread_comm_.reset(nof_threads_);
            start_threads();
        }
        catch (...)
        {
            stop();
            throw;
        }
    }

    void player_impl::stop()
    {
        thread_comm_.interrupt();
        stop_threads();
    }

// private:

    int player_impl::get_nof_threads() const
    {
        int num_threads = thread::hardware_concurrency();
        if (num_threads == 0)
        {
            return default_num_threads;
        }
        return min(num_threads, max_threads);
    }

    void player_impl::start_threads()
    {
        threads_ = thread_range(thread_container_.begin(),
                                thread_container_.begin()
                                 + nof_threads_);
        // Start the threads.
        for (ptrdiff_t n = 0; n < threads_.size(); ++n)
        {
            threads_[n] = thread(bind(&player_impl::thread_proc,
                                      this, n)).move();
        }
    }

    void player_impl::stop_threads()
    {
        foreach (thread &th, threads_)
        {
            th.interrupt();
            th.join();
        }
        threads_ = thread_range(thread_container_.begin(),
                                thread_container_.begin());
    }

    void player_impl::thread_proc(int thread_nr)
    {
        bool const worker0 = thread_nr == 0;
        try
        {
            for (;;)
            {
                uint8_t keys;
                int     rating;
                strategy_.compute_next_move(thread_nr, keys, rating);
                thread_comm_.push_result(keys, rating);
                if (worker0)
                {
                    thread_comm_.worker0_wait();
                    screen_state_.next_frame(parsed_frame_);
                    thread_comm_.start_computing();
                }
                else
                {
                    thread_comm_.worker_wait();
                }
            }
        }
        catch (thread_interrupted const &)
        {
        }
        catch (exception const &ex)
        {
            clog << "Error: " << ex.what() << endl;
        }
        catch (...)
        {
            clog << "Error: Unknown exception" << endl;
        }
        thread_comm_.interrupt();
    }

} // end of namespace intrepid
