// screen_state.hpp
// ----------------
//
//  (C) Copyright Gerald Thaler 2008.
//
// Distributed under the Boost Software License, Version 1.0.
//    (See accompanying file LICENSE_1_0.txt or copy at
//          http://www.boost.org/LICENSE_1_0.txt)

#ifndef INTREPID_SCREEN_STATE_HPP
#define INTREPID_SCREEN_STATE_HPP

#if defined(_MSC_VER) && (_MSC_VER >= 1200)
# pragma once
#endif // defined(_MSC_VER) && (_MSC_VER >= 1200)

#include "parse_frame.hpp"

namespace intrepid
{
    struct screen_asteroid
    {
        point         position;
        point         velocity;
        int8_t        lifetime; // 0 = Empty, 100 = Asteroid, < 99 Explosion
        int8_t        precision_x, precision_y;
        int8_t        shape;
        int8_t        size;
        bool          tracked;
        vector<point> prev_screen_positions;

        screen_asteroid();

        bool frame_pesent() const;

        void reset();

    private:
        screen_asteroid(screen_asteroid const &); // = delete;
        screen_asteroid &operator=(screen_asteroid const &); // = delete;
    };

    struct screen_saucer
    {
        point   position;
        point   velocity;
        int8_t  size;
        bool    present;

        screen_saucer();
    };

    struct screen_ship
    {
        point   position;
        point   velocity;
        uint8_t min_angle, max_angle;
        bool    present;

        screen_ship();

        void accelerate(uint8_t keys);
        void move();
        void reset();

    private:
        screen_ship(screen_ship const &); // = delete;
        screen_ship &operator=(screen_ship const &); // = delete;
    };

    struct screen_shot
    {
        point position;
        bool  present;
    };

    class screen_state
    {
    public:
        screen_state();

        array<screen_asteroid, 27> const &get_asteroids() const;
        screen_saucer const &get_saucer() const;
        screen_ship const &get_ship() const;
        array<screen_shot, 6> const &get_shots() const;
        bool frame_present() const;
        int get_frame_diff() const;
        uint8_t get_ping() const;
        comm_model::ping_to_keys_map const &get_ping_to_keys() const;

        void next_frame(parsed_frame const &frame);
        void reset();

        parsed_frame const *frame_;

    private:
        screen_state(screen_state const &); // = delete;
        screen_state &operator=(screen_state const &); // = delete;

        void adjust_existing_asteroid(screen_asteroid &ast,
                                      vectorram_object const &vram_obj);
        void adjust_new_asteroid(screen_asteroid &ast,
                                 vectorram_object const &vram_obj);
        void adjust_ship(vectorram_object const &vram_obj);
        void adjust_ship_angle(uint8_t keys);
        void adjust_ship_velocity(uint8_t keys,
                                  vectorram_object const &vram_obj);
        void track_objects();

        array<screen_shot, 6>      shots_;
        screen_saucer              saucer_;
        screen_ship                ship_;
        array<screen_asteroid, 27> asteroids_;

        bool                first_frame_;
        int8_t              frame_diff_;
        int8_t              frame_skip_;
        uint8_t             frameno_byte_;
        int                 frameno_;
        int                 latency_;
    };

// class screen_asteroid
// public:

    inline screen_asteroid::screen_asteroid()
    {
        prev_screen_positions.reserve(9);
        reset();
    }

// struct screen_saucer

    inline screen_saucer::screen_saucer()
        :   position(0, 0),
            velocity(0, 0),
            size(0),
            present(false)
    {
    }

// class screen_ship
// public:

    inline screen_ship::screen_ship()
        :   min_angle(256 - 3), max_angle(3), velocity(0, 0)
    {
        reset();
    }

// class screen_state
// public:

    inline screen_state::screen_state()
        :   frame_(0)
    {
    }

    inline bool screen_state::frame_present() const
    {
        return frame_ != 0;
    }

    inline
        array<screen_asteroid, 27> const &screen_state::get_asteroids() const
    {
        return asteroids_;
    }

    inline screen_ship const &screen_state::get_ship() const
    {
        return ship_;
    }

    inline screen_saucer const &screen_state::get_saucer() const
    {
        return saucer_;
    }

    inline array<screen_shot, 6> const &screen_state::get_shots() const
    {
        return shots_;
    }

    inline int screen_state::get_frame_diff() const
    {
        return frame_diff_;
    }

    inline
    uint8_t screen_state::get_ping() const
    {
        return frame_->get_ping();
    }

    inline
    comm_model::ping_to_keys_map const &screen_state::get_ping_to_keys() const
    {
        return frame_->get_ping_to_keys();
    }
} // end of namespace intrepid

#endif // include guard
