
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>

#include "connection.h"
#include "control.h"

const int LOC_SIZE = 30;
const int MAX_X = 1023;
const int MAX_Y = 895; // ab 128

int playfield[MAX_X + 1][MAX_X + 1];
int timefield[MAX_X + 1][MAX_X + 1];

typedef struct ship_vectors {
  int x1, y1, x2, y2;
} ship_vectors;

typedef struct heading {
  int dx, dy;
} heading;

typedef struct ship {
  // current direction
  ship_vectors vabsolut;

  // desired heading
  heading moveto;
  
  // current heading
  heading current;
  
  // absolute position
  int x;
  int y;
} ship;

typedef struct point {
  int x;
  int y;
  bool isUfo;
  bool isBig;
} point;

typedef struct asteroid {
  int x;
  int y;
  int size;
  bool isBig;
  heading vector;
  int locX[2];
  int locY[2];
  int frame; // 0..1 frames recorded. 2 = data calculated
  bool isUfo;
} asteroid;



const int ASTEROIDS_SIZE = 25;
int nasteroids = 0;
asteroid a[ASTEROIDS_SIZE];

ship s;
KeysPacket keys;

int globalTarget = -1;


inline bool normalizeXY(int& x, int& y)
{
	bool normalized = false;
	
	if (x < 0)
	{
		x = MAX_X;
		normalized = true;
 	} else if (x > MAX_X)
 	{
 		x = 0;
 		normalized = true;
  	}
		  	
	if (y < 128)
	{
		y = MAX_Y;
		normalized = true;
	} else if (y > MAX_Y)
	{
		y = 128;
		normalized = true;
	}
	
	return normalized;
}

inline double arc(int dx1, int dy1, int dx2, int dy2)
{
	int dotproduct = dx1 * dx2 + dy1 * dy2;
	double len_current = sqrt(dx1 * dx1 + dy1 * dy1);
	double len_moveto  = sqrt(dx2 * dx2 + dy2 * dy2);
	double c = acos(dotproduct / (len_current * len_moveto));
  
	return c;
}

inline bool isHeadingEqual(const heading& h1, const heading& h2)
{
	return h1.dx == h2.dx && h1.dy == h2.dy;
}

bool ship_Rotate()
{
	if (isHeadingEqual(s.current, s.moveto))
	{
		return true;
	}

	int dotproduct = s.current.dx * s.moveto.dx + s.current.dy * s.moveto.dy;
	double len_current = sqrt(s.current.dx * s.current.dx + s.current.dy * s.current.dy);
	double len_moveto  = sqrt(s.moveto.dx * s.moveto.dx + s.moveto.dy * s.moveto.dy);
	double c = acos(dotproduct / (len_current * len_moveto));
  
	if (c < 0.04)
	{
		return true;
	}

	// Continue rotate
	if (s.current.dx * s.moveto.dy - s.current.dy * s.moveto.dx > 0)
	{
		keys.left(true);
	} else
	{
		keys.right(true);
	}
	
	return false;
}

void doShip()
{
	if (ship_Rotate())
	{
		keys.fire(true);
	}
}


int getAsteroid(int radiusX, int radiusY, bool onlyClosing)
{
	int sx = s.x - radiusX / 2;
	int sy = s.y + radiusY / 2;
	normalizeXY(sx, sy);
	
	int targetNr = -1;
	int minTime = 10000;
	
	int targets[ASTEROIDS_SIZE];
	
	for (int i = 0; i < ASTEROIDS_SIZE; ++i)
		targets[i] = -1;
	
//	printf("Beginne Prüfiumg bei %d, %d\n", sx,sy);	

	for (int x = 0; x < radiusX; x++)
	{
		for (int y = 0; y < radiusY; y++)
		{
			int xx = sx + x;
			int yy = sy - y;

			if (yy < 128)
				yy = 128;
			if (xx > 1023)
				xx = 1023;

			if (playfield[xx][yy] != 0)
			{
				targets[playfield[xx][yy]] = playfield[xx][yy];
			}			
			if (timefield[xx][yy] != 0 && timefield[xx][yy] < minTime)
			{
//				printf("(%d) Neue mintime (%d, %d  %d, %d): %d %d\n", radius, xx,yy, s.x, s.y,  timefield[xx][yy], playfield[xx][yy]);
				minTime = timefield[xx][yy];
				targetNr = playfield[xx][yy];
			}
		}
	}

	int shipWatchingCorner = 0; // top lefT;
	if (s.current.dx >= 0 && s.current.dy >= 0)
		shipWatchingCorner = 1; // top right;
	if (s.current.dx >= 0 && s.current.dy <= 0)
		shipWatchingCorner = 2; // bottom right;
	if (s.current.dx <= 0 && s.current.dy <= 0)
		shipWatchingCorner = 3; // bottom left;
	

	if (onlyClosing)
		targetNr = -1;
		
//	targetNr = -1;
	for (int i = 0; i < ASTEROIDS_SIZE; ++i)
	{
		int nr = targets[i];
		if (nr >= 0)
		{
		
			int isClosingIn = 0;
			int dx  = s.x - a[nr].x;
			int dy  = s.y - a[nr].y;
			int dx2 = s.x - (a[nr].x + a[nr].vector.dx);
			int dy2 = s.y - (a[nr].y + a[nr].vector.dx);
			
			int len1 = abs(dx * dx + dy * dy);
			int len2 = abs(dx2 * dx2 + dy2 * dy2);			
		
			if (len2 < len1)
				isClosingIn = 1;
		
			int asteroidCorner = 0; // top left;
			if (a[nr].x >= 524 && a[nr].y >= 524)
				asteroidCorner = 1; // top right
			if (a[nr].x >= 524 && a[nr].y <= 524)
				asteroidCorner = 2; // bottom right
			if (a[nr].x <= 524 && a[nr].y <= 524)
				asteroidCorner = 1; // bottom left;
				
			if (asteroidCorner == shipWatchingCorner && isClosingIn)
			{
//				printf("override: old: %d  new: %d\n", targetNr, nr);
				targetNr = nr;
				break;
			}
			
		}
	}

	return targetNr;	
}

int decodeFrame(FramePacket *p)
{
	// letzte strahlenposition
	int labsX = 0;
	int labsY = 0;
	int gsf = 0;
	int object = 0;
	int x = 0;
	int y = 0;
	int z = 0;
	
	static int resetCounter = 0;
	
	// ship
	int ship_alive = 0;
	int second = 0;
	
	// Asteroid tracking
	point hazards[ASTEROIDS_SIZE];
	int asteroidCount = 0;
	
	// Vector Ram
	unsigned short offset = 0;
	unsigned short cmd1 = 0; 
	unsigned short cmd2 = 0;
	unsigned short *ram = (unsigned short*)p->vectorram;
	
	while (cmd1 != 0xB000 && cmd2 != 0xB000)
	{
		cmd1 = *(ram + offset);
		int opcode = (cmd1 & 0xF000) >> 12;
		
		switch (opcode)
		{
			case 0:
			case 1:
			case 2:
			case 3:
			case 4:
			case 5:
			case 6:
			case 7:
			case 8:
			case 9:
				// VCTR - langer Vektor
				cmd2 = *(ram + offset + 1);
				y = (cmd1 & 0x03FF) * (cmd1 & 0x0400 ? -1 : 1);
				x = (cmd2 & 0x03FF) * (cmd2 & 0x0400 ? -1 : 1);
				z = (cmd2 & 0xF000) >> 12;
				
				if (opcode == 6 && z == 12)
				{
					ship_alive = 1;
					
					s.x = labsX;
					s.y = labsY;
					
					if (second == 0)
					{
					  second = 1;
					  s.vabsolut.x1 = x;
					  s.vabsolut.y1 = y;
					} else
					{
					  s.vabsolut.x2 = x;
					  s.vabsolut.y2 = y;
					  
					  s.current.dx = s.vabsolut.x1 - s.vabsolut.x2;
					  s.current.dy = s.vabsolut.y1 - s.vabsolut.y2;
					}
//					printf("%04X: %04X %04X  VCTR (x: %d, y: %d), s%d, z%d\n", offset, cmd1, cmd2, x, y, opcode, z);
				}				
				offset += 2;
				break;
			case 0x0A:
				// LABS - Strahl positionieren
				cmd2 = *(ram + offset + 1);
				labsY = cmd1 & 0x03FF;
				labsX = cmd2 & 0x03FF;
				gsf = (cmd2 & 0xF000) >> 12;				
//				printf ADD_WINKEL("%04X: %04X %04X  LABS (x: %d, y: %d), s%d\n", offset, cmd1, cmd2, labsX, labsY, gsf);
				offset += 2;
				break;
			case 0x0B:
				// HALT - Programmende
				cmd1 = 0xB000;
//				printf("%04X: %04X       HALT\n", offset, cmd1);
				break;
			case 0x0C:
				// JSRL - Subroutine aufrufen
				object = cmd1 & 0x0FFF;
//				printf("%04X: %04X       JSRL %X (%s)\n", offset, cmd1, object, getObjectName(object));				
				switch (object)
				{
					case 0x8F3:
					case 0x8FF:
					case 0x90D:
					case 0x91A:
					case 0x929:
						if (asteroidCount < ASTEROIDS_SIZE)
						{
							hazards[asteroidCount].x = labsX;
							hazards[asteroidCount].y = labsY;						
							hazards[asteroidCount].isUfo = (object == 0x929);
							hazards[asteroidCount].isBig = (gsf == 0 || gsf == 15);
							asteroidCount += 1;
						}
						break;						
				}
				offset += 1;
				break;
			default:
				offset += 1;
				break;
		}
	}
	
	if (ship_alive == 0)
	{
	  s.x = -1;
	  s.y = -1;
	}


    if (nasteroids != asteroidCount  ||  asteroidCount == 0)
    {
		memset(a, 0, sizeof(asteroid) * ASTEROIDS_SIZE);
//		printf("Wipe asteroids\n");
	}
	
	resetCounter++;
	
	nasteroids = asteroidCount;
	
	for (int i = 0; i < nasteroids; ++i)
	{
	  a[i].x = hazards[i].x;
	  a[i].y = hazards[i].y;
	  a[i].isUfo = hazards[i].isUfo;
	  a[i].size = 1;
	  a[i].frame += 1;
	  a[i].isBig = hazards[i].isBig;
	  
	  if (a[i].isUfo && a[i].frame > 4)
	  {
	  	if (   (a[i].locX[0] == a[i].locX[1] && a[i].x != a[i].locX[0])
	  	    || (a[i].locY[0] == a[i].locY[1] && a[i].y != a[i].locY[0]))
	  	{
	  		a[i].frame = 0; // Ufo fliegt nicht immer konstant
	  		printf("Ufo reset! Changed course\n");
	  	}
	  }
	  
	  if (a[i].frame == 2)
	  {
		a[i].locX[0] = a[i].x;
	  	a[i].locY[0] = a[i].y;
	  }
	  if (a[i].frame == 4)
	  {
		a[i].locX[1] = a[i].x;
	  	a[i].locY[1] = a[i].y;
	  }
	}

	for (int i = 0; i < nasteroids; ++i)
	{
		// dx, dy rausbekommen (Flugvektor)
		if (a[i].frame == 4)
		{
			a[i].vector.dx = a[i].locX[1] - a[i].locX[0];
			a[i].vector.dy = a[i].locY[1] - a[i].locY[0];
		}
	}
	
	
//	memset(playfield, 0, sizeof(int) * (MAX_X + 1));
//	memset(timefield, 0, sizeof(int) * (MAX_X + 1));
	
	for (int i = 0; i < (MAX_X + 1); ++i)
		for (int j = 0; j < (MAX_X + 1); ++j)
		{
			playfield[i][j] = 0;
			timefield[i][j] = 0;
		}
	
	for (int i = 0; i < nasteroids; ++i)
	{
	  if (a[i].frame >= 4)
	  {
	  	int x = a[i].x; 
	  	int y = a[i].y;
	  	
  		// Projektion von den nächsten 50 frames
	  	for (int j = 0; j < 50; ++j)
	  	{
	  		normalizeXY(x, y);	  		
  		
//  			printf("%d (%d):  vec: %d, %d  pos %d %d\n", i, j, a[i].vector.dx, a[i].vector.dy, x, y);
	  		playfield[x][y] = i;
	  		timefield[x][y] = j;
	  		
	  		x += a[i].vector.dx;
	  		y += a[i].vector.dy;
	  	}
	  }
	}
	
	
	// Um das Schiff herum den Asteroid anvisieren, der am schnellsten vorbei fliegt (größte gefahr);
	// 300x300 um das schiff herum prüfen
	if (s.x != -1)
	{
		int rangeX = 50;
		int rangeY = 50;
		int targetNr = -1;
		
		int ufoNr = -1;
		
		for (int i = 0; i < nasteroids && ufoNr == -1; ++i)
		{
			ufoNr =  (a[i].isUfo ? i : -1);
		}
		
		if (ufoNr != -1)
		{
		  targetNr = getAsteroid(50, 50, true);
		  if (targetNr == -1)
		  {
		  	targetNr = ufoNr;
		  }		  
		} else
		{
			while (targetNr == -1 && rangeX <= 850)
			{
				targetNr = getAsteroid(rangeX, rangeY, false);
				rangeX += (rangeX < 200 ? 25 : 200);
				rangeY += (rangeY < 200 ? 25 : 200);
				if (rangeY > 700)
					rangeY = 700;
			}
			
		}

#define BIG_RANGE 16
#define CLOSE_RANGE 18
#define BORDER 300
		globalTarget = -1;
		
		if (targetNr >= 0 && targetNr < nasteroids && nasteroids > 0 && a[0].frame >= 4)
		{
		
			globalTarget = targetNr;
			
			// Entfernung 
			int x = a[targetNr].x;
			int y = a[targetNr].y;
			
//			int x_trans = x - 524;
//			int y_trans = y - 392;

			int steps = 0;

			double distance = (double)sqrt( abs(s.x - x) * abs(s.x - x) + abs(s.y - y) * abs(s.y - y));
			
			int stepFactor = distance < BORDER ? CLOSE_RANGE : BIG_RANGE;


			for (int i = 0; i < 100; ++i)
			{
				bool hadToNorm = normalizeXY(x, y);

				if (stepFactor == CLOSE_RANGE && distance > BORDER)
				{
					steps = 0;
					i = 0;
					stepFactor = BIG_RANGE;
				}
				
				if (stepFactor == BIG_RANGE && distance < BORDER)
				{
					steps = 0;
					i = 0;
					stepFactor = CLOSE_RANGE;
				}
				
				// Reset falls wir über die Ränder gingen				
				if (hadToNorm)
				{
					steps = 0;
					i = 0;
				}
				
				distance = (double)sqrt( abs(s.x - x) * abs(s.x - x) + abs(s.y - y) * abs(s.y - y));
				steps++;
				
				if ((steps * stepFactor) >= distance)
				{
					if ((steps * stepFactor) >= (distance + stepFactor/2))
					{
				  		x += a[targetNr].vector.dx;
				  		y += a[targetNr].vector.dy;
				  		normalizeXY(x, y);
					}
					break;
				}
			
		  		x += a[targetNr].vector.dx;
		  		y += a[targetNr].vector.dy;
			}
			
			int dirX = x - s.x;
			int dirY = y - s.y;
			s.moveto.dx = dirX;
			s.moveto.dy = dirY;
		} 
	}
	return 0;
}

int main(int argc, char* argv[])
{
	char* ip = "127.0.0.1";
	FramePacket frame;
	
	printf("Mini Controller 1 - ct Programmierwettbewerb 08\n");
	printf("von Bastian Pflieger <wb@illogical.de>\n\n");
	
	if (argc > 1)
	{
		ip = argv[1];
	}
	
	printf("Baue Verbindung zu Server auf: %s\n", ip);

	Connection connection(ip);
	
	keys.clear();
	memset(a, 0, sizeof(asteroid) * ASTEROIDS_SIZE);
    	
    s.x = -1;
    s.y = -1;
	
	while(true)
	{
		keys.clear();
		++keys.ping;
		
		doShip();
	
    	connection.SendPacket(keys);
    	connection.ReceivePacket(frame);
    	
		decodeFrame(&frame);
	}
	return 0;
}
