// ============================================================================
// File:               $File$
//
// Project:            
//
// Purpose:            
//
// Author:             Rammi
//
// Copyright Notice:   (c) 2008  Rammi (rammi@caff.de)
//                     The usage of this source code in commercial products
//                     is not allowed without explicite permission.
//
// Latest change:      $Date$
//
// History:	       $Log$
//=============================================================================
package de.caff.asteroid.analysis.statistics;

import de.caff.asteroid.GameData;
import de.caff.asteroid.Property;
import de.caff.asteroid.analysis.FrameKeyInfo;
import de.caff.i18n.I18n;

import java.util.*;

/**
 *  How often keys were pressed.
 */
public class KeyStatistics
        extends AbstractBasicDumpFileStatistics
        implements GameData
{
  private static class KeyCounter
  {
    private final String tag;
    private final int keyMask;
    private int count = 0;

    KeyCounter(String tag, int keyMask)
    {
      this.tag = tag;
      this.keyMask = keyMask;
    }

    public void reset()
    {
      count = 0;
    }

    public void check(int keys)
    {
      if ((keyMask & keys) != 0) {
        ++count;
      }
    }

    public Property<String> getStatistics(int frameCount)
    {
      return new Property<String>(I18n.getString(tag),
                                  String.format("%5d (%.2f%%)",
                                                count,
                                                100.0*count/frameCount));
    }
  }
  private KeyCounter[] counter = {
          new KeyCounter("propKeyLeft",   BUTTON_LEFT),
          new KeyCounter("propKeyRight",  BUTTON_RIGHT),
          new KeyCounter("propKeyFire",   BUTTON_FIRE),
          new KeyCounter("propKeyHyper",  BUTTON_HYPERSPACE),
          new KeyCounter("propKeyThrust", BUTTON_THRUST)
  };
  private int frameCount;

  /**
   * Analyse the frames.
   *
   * @param infos frame key infos to analyse
   */
  public void analyse(Collection<FrameKeyInfo> infos)
  {
    frameCount = infos.size();
    for (KeyCounter c: counter) {
      c.reset();
    }
    for (FrameKeyInfo info: infos) {
      List<FrameKeyInfo.ButtonInfo> buttonList = info.getButtons();
      if (!buttonList.isEmpty()) {
        int keys = buttonList.get(0).getButtons().getKeys();
        for (KeyCounter c: counter) {
          c.check(keys);
        }
      }
    }
  }

  /**
   * Get a title for this statistics.
   *
   * @return statistics title
   */
  public String getTitle()
  {
    return I18n.getString("titleKeyStat");
  }

  /**
   * Does this statistic need prepared frames?
   *
   * @return the answer
   */
  public boolean needPreparation()
  {
    return false;
  }

  /**
   *  Get a ratio for a count.
   *  @param count the count
   *  @return the ratio as a preformatted percent value
   */
  private String getRatio(int count)
  {
    return String.format("%3.2f%%", count/(double)frameCount);
  }

  /**
   * Get the properties of this object.
   *
   * @return collection of properties
   */
  public Collection<Property> getProperties()
  {
    Collection<Property> props = new ArrayList<Property>(counter.length);
    for (KeyCounter c: counter) {
      props.add(c.getStatistics(frameCount));
    }
    return props;
  }
}
