;;; -*- Mode: LISP; Syntax: COMMON-LISP; Package: WELTRAUMPUTZE; Base: 10 -*-

;;; Data and functions related to gun positioning

(in-package :weltraumputze)

(defparameter *angle-byte-list*
  #((0 (1536 0) (8.0 0.0))
    (1 (1536 0) (7.984214 0.50232416))
    (2 (1528 152) (7.91821 1.1410314))
    (3 (1504 296) (7.8042765 1.7587695))
    (4 (1472 440) (7.670558 2.2721229))
    (5 (1472 440) (7.4585905 2.8929965))
    (6 (1416 584) (7.2386165 3.4062343))
    (7 (1360 720) (6.962846 3.9393885))
    (8 (1280 856) (6.671087 4.415496))
    (9 (1280 856) (6.3212404 4.9032564))
    (10 (1192 976) (5.9731054 5.3218427))
    (11 (1088 1088) (5.5371456 5.7740817))
    (12 (976 1192) (5.056235 6.199556))
    (13 (976 1192) (4.5542364 6.5771523))
    (14 (856 1280) (4.072331 6.8859363))
    (15 (720 1360) (3.5820725 7.153234))
    (16 (584 1416) (3.0614674 7.391036))
    (17 (584 1416) (2.5251925 7.591008))
    (18 (440 1472) (1.9218245 7.765732))
    (19 (296 1504) (1.4029335 7.8760257))
    (20 (152 1528) (0.7667658 7.9631696))
    (21 (152 1528) (0.12565842 7.999013))
    (22 (-152 1528) (-0.39079827 7.990449))
    (23 (-296 1504) (-1.0026649 7.936918))
    (24 (-296 1504) (-1.6359688 7.830939))
    (25 (-440 1472) (-2.1379075 7.7090435))
    (26 (-584 1416) (-2.762385 7.5079446))
    (27 (-720 1360) (-3.266644 7.3026733))
    (28 (-720 1360) (-3.7680953 7.057015))
    (29 (-856 1280) (-4.3336663 6.7245326))
    (30 (-976 1192) (-4.781001 6.414205))
    (31 (-1088 1088) (-5.259002 6.0285068))
    (32 (-1088 1088) (-5.656854 5.656854))
    (33 (-1192 976) (-6.0741696 5.2061944))
    (34 (-1280 856) (-6.414205 4.781001))
    (35 (-1360 720) (-6.776975 4.2511888))
    (36 (-1360 720) (-7.057015 3.7680953))
    (37 (-1416 584) (-7.342037 3.177183))
    (38 (-1472 440) (-7.5504074 2.644116))
    (39 (-1504 296) (-7.7090435 2.1379075))
    (40 (-1504 296) (-7.858298 1.4990506))
    (41 (-1528 152) (-7.936918 1.0026649))
    (42 (-1536 0) (-7.991119 0.37685126))
    (43 (-1536 0) (-7.9956017 -0.26524016))
    (44 (-1528 -152) (-7.9631696 -0.76676726))
    (45 (-1528 -152) (-7.8808746 -1.3754315))
    (46 (-1504 -296) (-7.745181 -2.00304))
    (47 (-1472 -440) (-7.5997767 -2.498678))
    (48 (-1416 -584) (-7.3640385 -3.12585))
    (49 (-1416 -584) (-7.13438 -3.619478))
    (50 (-1360 -720) (-6.857338 -4.1203055))
    (51 (-1280 -856) (-6.5210233 -4.634248))
    (52 (-1192 -976) (-6.128356 -5.1423006))
    (53 (-1192 -976) (-5.7740817 -5.5371456))
    (54 (-1088 -1088) (-5.4047227 -5.8982177))
    (55 (-976 -1192) (-4.9032574 -6.3212395))
    (56 (-856 -1280) (-4.4851117 -6.624483))
    (57 (-856 -1280) (-3.9636686 -6.949053))
    (58 (-720 -1360) (-3.3682861 -7.2563524))
    (59 (-584 -1416) (-2.8929949 -7.458591))
    (60 (-440 -1472) (-2.3790007 -7.638086))
    (61 (-440 -1472) (-1.75877 -7.8042765))
    (62 (-296 -1504) (-1.2376829 -7.903679))
    (63 (-152 -1528) (-0.61375266 -7.976422))
    (64 (0 -1536) (0.0 -8.0))
    (65 (152 -1528) (0.5023237 -7.984214))
    (66 (296 -1504) (1.1410323 -7.9182096))
    (67 (440 -1472) (1.75877 -7.8042765))
    (68 (440 -1472) (2.2721236 -7.6705575))
    (69 (584 -1416) (2.8929977 -7.45859))
    (70 (720 -1360) (3.380945 -7.250463))
    (71 (856 -1280) (3.878477 -6.996958))
    (72 (856 -1280) (4.4154954 -6.671087))
    (73 (976 -1192) (4.903255 -6.3212414))
    (74 (1088 -1088) (5.321843 -5.973105))
    (75 (1192 -976) (5.7740836 -5.537143))
    (76 (1192 -976) (6.128356 -5.1423006))
    (77 (1280 -856) (6.512924 -4.6456237))
    (78 (1360 -720) (6.8284063 -4.168077))
    (79 (1416 -584) (7.108938 -3.669196))
    (80 (1416 -584) (7.342037 -3.177183))
    (81 (1472 -440) (7.591008 -2.5251918))
    (82 (1504 -296) (7.7381415 -2.0300646))
    (83 (1528 -152) (7.8760257 -1.4029335))
    (84 (1528 -152) (7.9631696 -0.76676726))
    (85 (1536 0) (7.9956017 -0.26524016))
    (86 (1536 0) (7.990449 0.39079815))
    (87 (1528 152) (7.936918 1.0026659))
    (88 (1504 296) (7.8530173 1.526472))
    (89 (1504 296) (7.7090435 2.137907))
    (90 (1472 440) (7.550408 2.644115))
    (91 (1416 584) (7.342037 3.177183))
    (92 (1360 720) (7.057015 3.768095))
    (93 (1360 720) (6.724533 4.3336654))
    (94 (1280 856) (6.414205 4.781001))
    (95 (1192 976) (6.0741706 5.206194))
    (96 (1088 1088) (5.656854 5.656854))
    (97 (1088 1088) (5.2061934 6.0741706))
    (98 (976 1192) (4.781001 6.414205))
    (99 (856 1280) (4.3336654 6.724533))
    (100 (720 1360) (3.7680945 7.057015))
    (101 (720 1360) (3.177183 7.342037))
    (102 (584 1416) (2.6441152 7.5504074))
    (103 (440 1472) (2.1379068 7.7090435))
    (104 (296 1504) (1.5264722 7.8530173))
    (105 (296 1504) (1.0026656 7.936918))
    (106 (152 1528) (0.39079827 7.990449))
    (107 (-152 1528) (-0.26524168 7.9956017))
    (108 (-152 1528) (-0.7667658 7.9631696))
    (109 (-296 1504) (-1.4029335 7.8760257))
    (110 (-440 1472) (-2.0300632 7.738142))
    (111 (-584 1416) (-2.5251918 7.591008))
    (112 (-584 1416) (-3.177183 7.342037))
    (113 (-720 1360) (-3.669197 7.1089373))
    (114 (-856 1280) (-4.168077 6.8284063))
    (115 (-976 1192) (-4.6569834 6.5048065))
    (116 (-976 1192) (-5.1423006 6.128356))
    (117 (-1088 1088) (-5.5371456 5.7740817))
    (118 (-1192 976) (-5.9731064 5.321842))
    (119 (-1280 856) (-6.3212395 4.9032574))
    (120 (-1280 856) (-6.671087 4.4154954))
    (121 (-1360 720) (-6.996958 3.878477))
    (122 (-1416 584) (-7.250462 3.3809462))
    (123 (-1472 440) (-7.4585905 2.8929963))
    (124 (-1472 440) (-7.674512 2.2587304))
    (125 (-1504 296) (-7.8042765 1.75877))
    (126 (-1528 152) (-7.91821 1.1410309))
    (127 (-1536 0) (-7.9859185 0.47444955))
    (128 (-1536 0) (-8.0 0.0))
    (129 (-1536 0) (-7.976422 -0.61375266))
    (130 (-1528 -152) (-7.903679 -1.2376829))
    (131 (-1504 -296) (-7.8042765 -1.75877))
    (132 (-1472 -440) (-7.638086 -2.3790007))
    (133 (-1472 -440) (-7.458591 -2.8929949))
    (134 (-1416 -584) (-7.250463 -3.380945))
    (135 (-1360 -720) (-6.949053 -3.9636686))
    (136 (-1280 -856) (-6.6166453 -4.496666))
    (137 (-1280 -856) (-6.3212395 -4.9032574))
    (138 (-1192 -976) (-5.8982177 -5.4047227))
    (139 (-1088 -1088) (-5.5371456 -5.7740817))
    (140 (-976 -1192) (-5.1423006 -6.128356))
    (141 (-976 -1192) (-4.634248 -6.5210233))
    (142 (-856 -1280) (-4.1203055 -6.857338))
    (143 (-720 -1360) (-3.619478 -7.13438))
    (144 (-584 -1416) (-3.12585 -7.3640385))
    (145 (-584 -1416) (-2.498678 -7.5997767))
    (146 (-440 -1472) (-2.00304 -7.745181))
    (147 (-296 -1504) (-1.3754315 -7.8808746))
    (148 (-152 -1528) (-0.76676726 -7.9631696))
    (149 (-152 -1528) (-0.26524016 -7.9956017))
    (150 (152 -1528) (0.37685424 -7.991119))
    (151 (296 -1504) (1.0026679 -7.9369173))
    (152 (296 -1504) (1.4990491 -7.8582983))
    (153 (440 -1472) (2.137906 -7.709044))
    (154 (584 -1416) (2.6441145 -7.550408))
    (155 (720 -1360) (3.177183 -7.342037))
    (156 (720 -1360) (3.7680953 -7.057015))
    (157 (856 -1280) (4.263011 -6.769545))
    (158 (976 -1192) (4.7810035 -6.414203))
    (159 (1088 -1088) (5.2061934 -6.0741706))
    (160 (1088 -1088) (5.656854 -5.656854))
    (161 (1192 -976) (6.0285068 -5.259002))
    (162 (1280 -856) (6.414203 -4.7810035))
    (163 (1360 -720) (6.724534 -4.3336635))
    (164 (1360 -720) (7.057015 -3.7680953))
    (165 (1416 -584) (7.302673 -3.2666452))
    (166 (1472 -440) (7.507943 -2.7623878))
    (167 (1504 -296) (7.709044 -2.137906))
    (168 (1504 -296) (7.83094 -1.635966))
    (169 (1528 -152) (7.9369173 -1.0026679))
    (170 (1536 0) (7.990449 -0.39079678))
    (171 (1536 0) (7.999013 0.12565854))
    (172 (1528 152) (7.9631696 0.766766))
    (173 (1528 152) (7.876025 1.4029338))
    (174 (1504 296) (7.765732 1.9218243))
    (175 (1472 440) (7.591008 2.5251923))
    (176 (1416 584) (7.391036 3.0614674))
    (177 (1416 584) (7.153234 3.5820727))
    (178 (1360 720) (6.8859363 4.0723314))
    (179 (1280 856) (6.5771523 4.5542364))
    (180 (1192 976) (6.199556 5.056235))
    (181 (1192 976) (5.774082 5.537145))
    (182 (1088 1088) (5.321843 5.973105))
    (183 (976 1192) (4.9032564 6.32124))
    (184 (856 1280) (4.4154963 6.6710863))
    (185 (856 1280) (3.9393888 6.9628453))
    (186 (720 1360) (3.418863 7.2326603))
    (187 (584 1416) (2.8929963 7.4585905))
    (188 (440 1472) (2.2721229 7.670558))
    (189 (440 1472) (1.7587692 7.8042765))
    (190 (296 1504) (1.1410309 7.91821))
    (191 (152 1528) (0.5023237 7.984214))
    (192 (0 1536) (0.0 8.0))
    (193 (-152 1528) (-0.6276722 7.975339))
    (194 (-296 1504) (-1.2376829 7.903679))
    (195 (-440 1472) (-1.75877 7.8042765))
    (196 (-440 1472) (-2.3789992 7.6380863))
    (197 (-584 1416) (-2.8929963 7.4585905))
    (198 (-720 1360) (-3.418863 7.2326603))
    (199 (-856 1280) (-4.0241594 6.9141984))
    (200 (-856 1280) (-4.496667 6.616645))
    (201 (-976 1192) (-4.9032574 6.3212395))
    (202 (-1088 1088) (-5.4047217 5.8982186))
    (203 (-1192 976) (-5.7644095 5.547214))
    (204 (-1192 976) (-6.199556 5.056235))
    (205 (-1280 856) (-6.5771527 4.5542355))
    (206 (-1360 720) (-6.9141984 4.0241594))
    (207 (-1416 584) (-7.1842203 3.5195143))
    (208 (-1416 584) (-7.4122286 3.0097952))
    (209 (-1472 440) (-7.5997763 2.4986794))
    (210 (-1504 296) (-7.772393 1.8947043))
    (211 (-1528 152) (-7.8808746 1.3754315))
    (212 (-1528 152) (-7.9604445 0.7945572))
    (213 (-1536 0) (-7.999013 0.12565842))
    (214 (-1536 0) (-7.9897547 -0.40474412))
    (215 (-1528 -152) (-7.936918 -1.0026649))
    (216 (-1504 -296) (-7.8393974 -1.5949439))
    (217 (-1504 -296) (-7.7053003 -2.1513596))
    (218 (-1472 -440) (-7.503111 -2.7754865))
    (219 (-1416 -584) (-7.302673 -3.2666452))
    (220 (-1360 -720) (-7.057015 -3.7680953))
    (221 (-1360 -720) (-6.7769747 -4.25119))
    (222 (-1280 -856) (-6.422539 -4.7697997))
    (223 (-1192 -976) (-6.037678 -5.2484713))
    (224 (-1088 -1088) (-5.656854 -5.656854))
    (225 (-1088 -1088) (-5.259002 -6.0285068))
    (226 (-976 -1192) (-4.781001 -6.414205))
    (227 (-856 -1280) (-4.263011 -6.769545))
    (228 (-720 -1360) (-3.7680953 -7.057015))
    (229 (-720 -1360) (-3.2666452 -7.302673))
    (230 (-584 -1416) (-2.762385 -7.5079446))
    (231 (-440 -1472) (-2.137906 -7.709044))
    (232 (-296 -1504) (-1.6086228 -7.836602))
    (233 (-296 -1504) (-1.0026679 -7.9369173))
    (234 (-152 -1528) (-0.37685424 -7.991119))
    (235 (152 -1528) (0.12565842 -7.999013))
    (236 (152 -1528) (0.76676726 -7.9631696))
    (237 (296 -1504) (1.3754315 -7.8808746))
    (238 (440 -1472) (1.8947073 -7.7723923))
    (239 (584 -1416) (2.4986808 -7.599776))
    (240 (584 -1416) (3.0097952 -7.4122286))
    (241 (720 -1360) (3.519513 -7.184221))
    (242 (856 -1280) (4.0241623 -6.914197))
    (243 (976 -1192) (4.5542383 -6.577151))
    (244 (976 -1192) (5.056236 -6.199555))
    (245 (1088 -1088) (5.537143 -5.7740836))
    (246 (1192 -976) (5.8982177 -5.4047227))
    (247 (1280 -856) (6.3212414 -4.903255))
    (248 (1280 -856) (6.616644 -4.4966683))
    (249 (1360 -720) (6.914197 -4.0241623))
    (250 (1416 -584) (7.232661 -3.4188616))
    (251 (1472 -440) (7.45859 -2.8929977))
    (252 (1472 -440) (7.638087 -2.3789978))
    (253 (1504 -296) (7.801195 -1.7723873))
    (254 (1528 -152) (7.903679 -1.2376829))
    (255 (1536 0) (7.975339 -0.6276722))))

(defparameter *visible-angle-list*
  '(((1536 0) 0 1 85 86 170 171 255)
    ((1528 152) 2 87 172 173)
    ((1504 296) 3 88 89 174)
    ((1472 440) 4 5 90 175)
    ((1416 584) 6 91 176 177)
    ((1360 720) 7 92 93 178)
    ((1280 856) 8 9 94 179)
    ((1192 976) 10 95 180 181)
    ((1088 1088) 11 96 97 182)
    ((976 1192) 12 13 98 183)
    ((856 1280) 14 99 184 185)
    ((720 1360) 15 100 101 186)
    ((584 1416) 16 17 102 187)
    ((440 1472) 18 103 188 189)
    ((296 1504) 19 104 105 190)
    ((152 1528) 20 21 106 191)
    ((-152 1528) 22 107 108 193)
    ((-296 1504) 23 24 109 194)
    ((-440 1472) 25 110 195 196)
    ((-584 1416) 26 111 112 197)
    ((-720 1360) 27 28 113 198)
    ((-856 1280) 29 114 199 200)
    ((-976 1192) 30 115 116 201)
    ((-1088 1088) 31 32 117 202)
    ((-1192 976) 33 118 203 204)
    ((-1280 856) 34 119 120 205)
    ((-1360 720) 35 36 121 206)
    ((-1416 584) 37 122 207 208)
    ((-1472 440) 38 123 124 209)
    ((-1504 296) 39 40 125 210)
    ((-1528 152) 41 126 211 212)
    ((-1536 0) 42 43 127 128 129 213 214)
    ((-1528 -152) 44 45 130 215)
    ((-1504 -296) 46 131 216 217)
    ((-1472 -440) 47 132 133 218)
    ((-1416 -584) 48 49 134 219)
    ((-1360 -720) 50 135 220 221)
    ((-1280 -856) 51 136 137 222)
    ((-1192 -976) 52 53 138 223)
    ((-1088 -1088) 54 139 224 225)
    ((-976 -1192) 55 140 141 226)
    ((-856 -1280) 56 57 142 227)
    ((-720 -1360) 58 143 228 229)
    ((-584 -1416) 59 144 145 230)
    ((-440 -1472) 60 61 146 231)
    ((-296 -1504) 62 147 232 233)
    ((-152 -1528) 63 148 149 234)
    ((0 -1536) 64)
    ((152 -1528) 65 150 235 236)
    ((296 -1504) 66 151 152 237)
    ((440 -1472) 67 68 153 238)
    ((584 -1416) 69 154 239 240)
    ((720 -1360) 70 155 156 241)
    ((856 -1280) 71 72 157 242)
    ((976 -1192) 73 158 243 244)
    ((1088 -1088) 74 159 160 245)
    ((1192 -976) 75 76 161 246)
    ((1280 -856) 77 162 247 248)
    ((1360 -720) 78 163 164 249)
    ((1416 -584) 79 80 165 250)
    ((1472 -440) 81 166 251 252)
    ((1504 -296) 82 167 168 253)
    ((1528 -152) 83 84 169 254)
    ((0 1536) 192)))

(defgeneric angle-byte-to-gun (angle-byte))

;; Returns the exact gun direction for the supplied angle byte
(defmethod angle-byte-to-gun ((angle-byte number))
  (values-list (caddr (svref *angle-byte-list* angle-byte))))

;; Returns the mean gun direction for a list of possible angles
(defmethod angle-byte-to-gun ((angle-bytes sequence))
  (let* ((gun-positions (reduce #'append angle-bytes
                               :key #'(lambda (b) (caddr (svref *angle-byte-list* b)))))
         (position-sum
          (do ((positions gun-positions (cddr positions))
               (sum-dx 0)
               (sum-dy 0))
              ((endp positions) (list sum-dx sum-dy))
            (incf sum-dx (car positions))
            (incf sum-dy (cadr positions))
            ))
         (position-sum-magnitude (magnitude (car position-sum) (cadr position-sum))))
    (if (zerop position-sum-magnitude)
        (values 0 0)
        (values (* (/ (car position-sum) position-sum-magnitude) 8)
                (* (/ (cadr position-sum) position-sum-magnitude) 8)))))

(defun visible-angle-to-angle-bytes (visible-angle)
  (let ((entry (assoc visible-angle *visible-angle-list* :test #'equal)))
    (if entry
        (cdr entry)
        (format t "visible-angle-to-angle-bytes: Unknown visible angle ~A.~%" visible-angle))))

;; Given the supplied command byte (or nil), returns a list of possible new angle bytes
(defgeneric trace-angle-byte (old command-byte))

(defmethod trace-angle-byte ((old-angle-byte number) command-byte)
  (cond
    ((or (null command-byte) (and (left command-byte) (right command-byte))) ; we have no idea how the ship behaved
     (list (mod (1- old-angle-byte) 256) old-angle-byte (mod (1+ old-angle-byte) 256)))
    ((left command-byte) ; ship turned left, angle byte increases
     (mod (1+ old-angle-byte) 256))
    ((right command-byte) ; ship turned right, angel byte decreases
     (mod (1- old-angle-byte) 256))
    (t ; ship did not change directions
     old-angle-byte)))

(defmethod trace-angle-byte ((old-angle-bytes sequence) command-byte)
  (reduce #'union old-angle-bytes :initial-value nil
          :key #'(lambda (b) (let ((bs (trace-angle-byte b command-byte))) (if (listp bs) bs (list bs))))))

(defun weed-out-impossible-angle-bytes (angle-bytes visible-angle)
  (remove-if #'(lambda (b)
                 (not (equal (cadr (svref *angle-byte-list* b)) visible-angle)))
             (if (listp angle-bytes) angle-bytes (list angle-bytes))))


(defun test-angle-tracing (angle)
  (do ((possible-angles (visible-angle-to-angle-bytes (cadr (svref *angle-byte-list* angle))))
       (n 1 (1+ n)))
      ((= 1 (length possible-angles)) n)
    (setf possible-angles (test-angle-tracing1 possible-angles (mod (- angle n) 256)))))

(defun test-angle-tracing1 (possible-angles new-angle)
  (weed-out-impossible-angle-bytes
   (trace-angle-byte possible-angles 8)
   (cadr (svref *angle-byte-list* new-angle))))

  