/******************************************************************************
* vector.inl
******************************************************************************/


/*****************************************************************************/
/** Constructs the vector instance.
*
* This function is the default constructor of the vector instance.
*****************************************************************************/

inline Vector2::Vector2()
: x(0), y(0)
{
}

/*****************************************************************************/
/** Constructs the vector instance.
*
* This function is a constructor of the vector instance.
*
* @param v The vector to construct this vector instance from.
*****************************************************************************/

inline Vector2::Vector2(const Vector2& v)
: x(v.x), y(v.y)
{
}

/*****************************************************************************/
/** Constructs the vector instance.
*
* This function is a constructor of the vector instance.
*
* @param vx The x component.
* @param vy The y component.
* @param vz The z component.
*****************************************************************************/

inline Vector2::Vector2(VecFloat vx, VecFloat vy)
: x(vx), y(vy)
{
}

/*****************************************************************************/
/** Destructs the vector instance.
*
* This function is the destructor of the vector instance.
*****************************************************************************/

inline Vector2::~Vector2()
{
}

/*****************************************************************************/
/** Interpolates the vector instance to another vector.
*
* This function interpolates the vector instance to another vector by a given
* factor.
*
* @param d The blending factor in the range [0.0, 1.0].
* @param v The vector to be interpolated to.
*****************************************************************************/

inline void Vector2::blend(VecFloat d, const Vector2& v)
{
  x += d * (v.x - x);
  y += d * (v.y - y);
}

/*****************************************************************************/
/** Clears the vector instance.
*
* This function clears the vector instance.
*****************************************************************************/

inline void Vector2::clear()
{
  x = 0;
  y = 0;
}

/*****************************************************************************/
/** Skalares Kreuzprodukt (richtig gerechnet???)
*
* 
*****************************************************************************/

inline VecFloat Vector2::cross(Vector2& v)
{
	return x * v.y - y * v.x; 
}

/*****************************************************************************/
/** Inverts the vector instance.
*
* This function inverts the vector instance.
*****************************************************************************/

inline void Vector2::invert()
{
  x = 1.0 / x;
  y = 1.0 / y;
}

/*****************************************************************************/
/** Returns the length of the vector instance.
*
* This function returns the length of the vector instance.
*
* @return The length of the vector instance.
*****************************************************************************/

inline VecFloat Vector2::length() const
{
  return sqrt(x * x + y * y);
}

/*****************************************************************************/
/** Returns the squared length of the vector instance.
*
* This function returns the squared length of the vector instance.
*
* @return The squared length of the vector instance.
*****************************************************************************/

inline VecFloat Vector2::length2() const
{
  return x * x + y * y;
}

/*****************************************************************************/
/** Multiplies a matrix to the vector instance.
*
* This operator multiplies a matrix to the vector instance.
*
* @param a The matrix to be multiplied.
*****************************************************************************/

inline void Vector2::multiply(const Matrix22& a)
{
  Vector2 temp(x, y);

  x = a.m[0][0] * temp.x + a.m[0][1] * temp.y;
  y = a.m[1][0] * temp.x + a.m[1][1] * temp.y;
}

/*****************************************************************************/
/** Negates the vector instance.
*
* This function negates the vector instance.
*****************************************************************************/

inline void Vector2::negate()
{
  x = -x;
  y = -y;
}

/*****************************************************************************/
/** Normalizes the vector instance.
*
* This function normalizes the vector instance and returns its former length.
*
* @return The length of the vector instance before normalizing.
*****************************************************************************/

inline VecFloat Vector2::normalize()
{
  // calculate the length of the vector
  VecFloat length;
  length = sqrt(x * x + y * y);
  if(length == 0) return 0;

  // normalize the vector
  x /= length;
  y /= length;

  return length;
}

/*****************************************************************************/
/** Calculates phase (angle) of vector
*
*****************************************************************************/

inline VecFloat Vector2::phase()
{
	VecFloat f;
	if (x * x + y * y > 0) 
		f = acos(x / sqrt(x * x + y * y));
	else
		f = 0;
		
	if (y < 0) f = -f;
	return f;
}

/*****************************************************************************/
/** Project vector on another
*
*****************************************************************************/

inline Vector2 Vector2::projectOn(const Vector2& direction)
{
  VecFloat cosine = (*this) * direction;
  return Vector2(direction / direction.length2() * cosine);
}

/*****************************************************************************/
/** Rotates the vector around alpha
*
*****************************************************************************/

inline void Vector2::rotate(VecFloat alpha)
{
	Vector2 temp(x, y);
 	x =  temp.x * cos(alpha) + temp.y * sin(alpha);
	y = -temp.x * sin(alpha) + temp.y * cos(alpha);
}

/*****************************************************************************/
/** Scales the vector instance.
*
* This function scales the vector instance.
*
* @param f The scaling factors.
*****************************************************************************/

inline void Vector2::scale(const Vector2& f)
{
  x *= f.x;
  y *= f.y;
}

/*****************************************************************************/
/** Sets new values.
*
* This function sets new values in the vector instance.
*
* @param x The x component.
* @param y The y component.
* @param z The z component.
*****************************************************************************/

inline void Vector2::set(VecFloat vx, VecFloat vy)
{
  x = vx;
  y = vy;
}

/*****************************************************************************/
/** Squares the vector instance.
*
* This function squares the vector instance.
*****************************************************************************/

inline void Vector2::square()
{
  x *= x;
  y *= y;
}

/*****************************************************************************/
inline void Vector2::subtract(const Matrix22& a, const Matrix22& b, const Vector2& f)
{
  VecFloat temp0 = f.x * (a.m[0][0]*b.m[0][0] + a.m[0][1]*b.m[1][0]) 
    + f.y * (a.m[0][0]*b.m[0][1] + a.m[0][1]*b.m[1][1]);
  VecFloat temp1 = f.x * (a.m[1][0]*b.m[0][0] + a.m[1][1]*b.m[1][0]) 
    + f.y * (a.m[1][0]*b.m[0][1] + a.m[1][1]*b.m[1][1]);
    
  x -= temp0; 
  y -= temp1; 

}

/*****************************************************************************/
/** Angle between vectors
*
* 
*****************************************************************************/

inline VecFloat Vector2::vectorAngle(Vector2* a)
{
	VecFloat f, fa;
	
	f = this->phase(); fa = a->phase();
	while (f < fa) f += 2 * Pi_double; 
	f -= fa;
	while (f > Pi_double) f -= 2 * Pi_double;
	
	return f; 
}

/*****************************************************************************/
/** Subtracts another vector from the vector instance.
*
* This operator subtracts another vector from the vector instance.
*
* @param v The vector to be subtracted.
*****************************************************************************/

inline void Vector2::operator-=(const Vector2& v)
{
  x -= v.x;
  y -= v.y;
}

  inline Vector2 Vector2::operator-() const
 {
		Vector2 result(*this);
		result.negate();
		return result;
 }

/*****************************************************************************/
/** Calculates the vector product of two vectors.
*
* This operator calculates the vector product of two vectors.
*
* @param v The second vector.
*****************************************************************************/

inline void Vector2::operator%=(const Vector2& v)
{
  Vector2 temp(y * v.x - x * v.y, x * v.y - y * v.x); 
  *this = temp;
}

/*****************************************************************************/
/** Calculates the vector product of two vectors.
*
* This operator calculates the vector product of two vectors.
*
* @param v The second vector.
* @return The vector product of the two vectors.
*****************************************************************************/

inline Vector2 Vector2::operator%(const Vector2& v)
{
	Vector2 temp(y * v.x - x * v.y, x * v.y - y * v.x); 
  	return temp;	
}

/*****************************************************************************/
/** Computes the dot product of the vector instance with another vector.
*
* This operator computes the dot product of the vector instance with another
* vector.
*
* @param v The vector to compute the dot product with.
*****************************************************************************/

inline VecFloat Vector2::operator*(const Vector2& v)
{
  return x * v.x + y * v.y;
}

/*****************************************************************************/
/** Scales the vector instance.
*
* This operator scales the vector instance by multiplying its components by
* a specific factor.
*
* @param d The factor to multiply the vector components by.
*****************************************************************************/

inline void Vector2::operator*=(const VecFloat d)
{
  x *= d;
  y *= d;
}


/*****************************************************************************/
/** Scales the vector instance.
*
* This operator scales the vector instance by dividing its components by a
* specific factor.
*
* @param d The factor to divide the vector components by.
*****************************************************************************/

inline void Vector2::operator/=(const VecFloat d)
{
  x /= d;
  y /= d;
}

/*****************************************************************************/
/** Provides access to the components of the vector instance.
*
* This function provides read and write access to the three components of the
* vector instance.
*
* @param i The index to the specific component.
*
* @return A reference to the specific component.
*****************************************************************************/

inline VecFloat& Vector2::operator[](unsigned int i)
{
  return (&x)[i];
}

/*****************************************************************************/
/** Provides access to the components of the vector instance.
*
* This function provides read access to the three components of the vector
* instance.
*
* @param i The index to the specific component.
*
* @return A constant reference to the specific component.
*****************************************************************************/

inline const VecFloat& Vector2::operator[](unsigned int i) const
{
  return (&x)[i];
}

/*****************************************************************************/
/** Adds another vector to the vector instance.
*
* This operator adds another vector to the vector instance.
*
* @param v The vector to be added.
*****************************************************************************/

inline void Vector2::operator+=(const Vector2& v)
{
  x += v.x;
  y += v.y;
}

/*****************************************************************************/
/** Equates the vector instance with another vector.
*
* This operator equates the vector instance with another vector.
*
* @param v The vector to equate the vector instance with.
*****************************************************************************/

inline void Vector2::operator=(const Vector2& v)
{
  x = v.x;
  y = v.y;
}

/*****************************************************************************/
/** Tests the equality of 2 vectors
*
* This operator checks to see if 2 vectors are equal
*
* @param v The vector to be tested against.
*****************************************************************************/

inline bool Vector2::operator==(const Vector2& v)
{
  return ((x == v.x) && (y == v.y));
}

/*****************************************************************************/
/** Subtracts a vector from another vector.
*
* This operator subtracts a vector from another vector.
*
* @param v1 The first vector.
* @param v2 The second vector.
*
* @return The result of the vector subtraction.
*****************************************************************************/

 inline Vector2 operator-(const Vector2 &v1, const Vector2 &v2)
{
  return Vector2(v1.x - v2.x, v1.y - v2.y);
}

/*****************************************************************************/
/** Calculates the vector product of two vectors.
*
* This operator calculates the vector product of two vectors.
*
* @param v1 The first vector.
* @param v2 The second vector.
*
* @return The vector product of the two vectors.
*****************************************************************************/

 inline Vector2 operator%(const Vector2 &v1, const Vector2 &v2)
{
  return Vector2(v1.y * v2.x - v1.x * v2.y, v1.x * v2.y - v1.y * v2.x);
}

/*****************************************************************************/
/** Scales a vector.
*
* This operator scales a vector instance.
*
* @param v The vector to scale.
* @param d The scale factor.
*
* @return The scaled vector.
*****************************************************************************/

 inline Vector2 operator*(const Vector2 &v, const VecFloat d)
{
  return Vector2(v.x * d, v.y * d);
}

/*****************************************************************************/
/** Calculates the dot product of two vectors.
*
* This operator calculates the dot product of two vectors.
*
* @param v1 The first vector.
* @param v2 The second vector.
*
* @return The dot product of the two vectors.
*****************************************************************************/

 inline VecFloat operator*(const Vector2 &v1, const Vector2 &v2)
{
  return v1.x * v2.x + v1.y * v2.y;
}

/*****************************************************************************/
/** Scales a vector.
*
* This operator scales a vector instance.
*
* @param v The vector to scale.
* @param d The reciprocal scale factor.
*
* @return The scaled vector.
*****************************************************************************/

 inline Vector2 operator/(const Vector2 &v, const VecFloat d)
{
  return Vector2(v.x / d, v.y / d);
}

/*****************************************************************************/
/** Adds two vectors.
*
* This operator adds two vectors.
*
* @param v1 The first vector.
* @param v2 The second vector.
*
* @return The sum of the two vectors.
*****************************************************************************/

 inline Vector2 operator+(const Vector2 &v1, const Vector2 &v2)
{
  return Vector2(v1.x + v2.x, v1.y + v2.y);
}

/*****************************************************************************/

 inline Matrix22 tensor(const Vector2 &v1, const Vector2 &v2)
{
  //return the tensor product of the vectors v1 and v2. 
  return Matrix22(v1*v2.x,v1*v2.y);
}


