#ifndef FRAMESTATUS_H
#define FRAMESTATUS_H

#include <string>
#include <vector>
#include <list>

#include "../FramePacket.h"
#include "../MovingObjects/Asteroid.h"
#include "../MovingObjects/Ufo.h"
#include "../MovingObjects/Ship.h"
#include "../KeysPacket.h"
#include "ObjectLink.h"
#include "LinkedClass.h"


const int SHOT_WAIT_ROUNDS = 40;


int const LINKED_FRAMES_SIZE = 10000;

/// \brief Dataset for an individual frame, possibly linked with previous and next frame.

/**
 *  This class analyzes individual frames, and
 *  puts them in relation to previous frames.
 *  Some history is recorded.
 */

class FrameStatus : public LinkedClass<FrameStatus>
{
public:

    Ufo*                   ufo;       ///< Any Ufo, if present.
    Ship*                  ship;      ///< The player's ship, if present.
    std::vector<Shot*>     shots;     ///< Any shots.
    std::vector<Asteroid*> asteroids; ///< Any asteroids.

    static int shipShots;

    KeysPacket keys;

    bool even; ///< Flag for even ($e0) or odd ($e2) frame.

    double latency; ///< Transmission latency.

    int score; ///< Current score of the player.

    double freeRadius; ///< Free space around ship.

    int frameDelta; ///< Dropped frames with respect to the previous frame stored.



    /**
     *  This constructor evaluates a vector ram packet (FramePacket).
     *  Fields are set accordingly.
     */
    FrameStatus(FramePacket &packet, KeysPacket &keys, int latency)
    : ufo(0), ship(0), keys(keys), latency((double)latency), LinkedClass<FrameStatus>()
    {
        /// First, the vector ram is read.
        processRam(packet);

        /// Then, the frame is connected to previous
        /// frames, linking objects to their previous
        /// counterparts.
        linkFrame();
    }


    /**
     *  This little helper routine normalizes screen
     *  coordinate offsets, given as floating point values.
     */
    static void normalize(double &dx, double &dy) {
        while (dx<-512.) dx+=1024.;
        while (dx>=512.) dx-=1024.;
        while (dy<-384.) dy+=768.;
        while (dy>=384.) dy-=768.;
    }


    /**
     *  This little helper routine normalizes screen
     *  coordinate offsets, given as integer values.
     */
    static void normalize(int &dx, int &dy) {
        while (dx<-512) dx+=1024;
        while (dx>=512) dx-=1024;
        while (dy<-384) dy+=768;
        while (dy>=384) dy-=768;
    }



    /**
     *  This little helper routine normalizes an
     *  angle (in radian) to fall between -pi and pi.
     */
    static void normalize(double &da) {
        while (da<-M_PI) da+=M_PI;
        while (da>=M_PI) da-=M_PI;
    }


    void linkFrame();


    virtual ~FrameStatus()
    {
        if (ufo) delete ufo;
        if (ship) delete ship;
        while(asteroids.size()>0) {
            delete (*(asteroids.rbegin()));
            asteroids.pop_back();
        }
        while(shots.size()>0) {
            delete (*(shots.rbegin()));
            shots.pop_back();
        }
    }



    /// \brief An exception of FrameStatus, with a text message.

    class FrameException
    {
    public:
        std::string msg;
        FrameException() : msg("") {};
        FrameException(std::string msg) : msg(msg) {};
    };



    void processRam(FramePacket &packet)
    {
        unsigned short vector_ram[512];
        for (int i=0; i<512; ++i)
            vector_ram[i] = (unsigned char)packet.vectorram[2*i]
                          | (unsigned char)packet.vectorram[2*i+1] << 8;

        if (vector_ram[0] == 0xe001)
            even = true;
        else if (vector_ram[0] == 0xe201)
            even = false;
        else
            throw new FrameException("Illegal Entry Point");

        int vx, vy; // momentane Position
        int vs;     // globaler Skalierungsfaktor
        int pc = 1;
        int shipdetect = 0;
        int v1x, v1y;
        int dx,dy;
        int sf, vz;
        while (pc < 512)
        {
            int op = vector_ram[pc] >> 12;
            switch (op)
            {
                case 0xa: // LABS
                    vy = vector_ram[pc] & 0x3ff;
                    vx = vector_ram[pc+1] & 0x3ff;
                    vs = vector_ram[pc+1] >> 12;

                if (  (vector_ram[pc  ] == 0xa36c)
                    &&(vector_ram[pc+1] == 0x1064)) {
                    score = 0;
                    for (int j=0;j<5;j++) {
                        score *=10;
                        switch (vector_ram[pc+4+j]) {
                            case 0xcb2c: break;
                            case 0xcB2E: score+=1;break;
                            case 0xcB32: score+=2;break;
                            case 0xcB3A: score+=3;break;
                            case 0xcB41: score+=4;break;
                            case 0xcB48: score+=5;break;
                            case 0xcB4F: score+=6;break;
                            case 0xcB56: score+=7;break;
                            case 0xcB5B: score+=8;break;
                            case 0xcB63: score+=9;break;
                            case 0xcadd: break;
                            default:
                                throw new FrameException("Illegal character in score");
                        }
                    }
                }
                break;
            case 0xb: // HALT
                pc = 512;
                break;
            case 0xc: // JSRL
                switch (vector_ram[pc] & 0xfff)
                {
                case 0x8f3:
                asteroids.push_back(new Asteroid(this, vx, vy, 1, vs));
                break;
            case 0x8ff:
                asteroids.push_back(new Asteroid(this, vx, vy, 2, vs));
                break;
            case 0x90d:
                asteroids.push_back(new Asteroid(this, vx, vy, 3, vs));
                break;
            case 0x91a:
                asteroids.push_back(new Asteroid(this, vx, vy, 4, vs));
                break;
            case 0x929:
                ufo = new Ufo(this, vx,vy,vs);
                break;
            }  
            break;
        case 0xd: // RTSL
            throw new FrameException("Instruction RTSL encountered.");
        case 0xe: // JMPL
            throw new FrameException("Instruction JMPL encountered.");
        case 0xf: // SVEC
            break;
        default:

            // y delta
            dy = vector_ram[pc] & 0x3ff;
            if ((vector_ram[pc] & 0x400) != 0)
                dy = -dy;

            // x delta
            dx = vector_ram[pc+1] & 0x3ff;
            if ((vector_ram[pc+1] & 0x400) != 0)
                dx = -dx;

            // remember op
            sf = op;

            vz = vector_ram[pc+1] >> 12;

            // ganz hell, ohne Bewegung -> Schuss
            if (dx == 0 && dy == 0 && vz == 15)
                shots.push_back(new Shot(this, vx, vy));

            // hell(12), per vector -> Schiff
            if (op == 6 && vz == 12 && dx != 0 && dy != 0)
            {
                switch (shipdetect)
                {
                case 0: // lange Kante 1
                    v1x = dx;
                    v1y = dy;
                    ++shipdetect;
                    break;
                case 1: // lange Kante 2
                    // beide Kanten -> ungefaehre Ausrichtung
                    ship = new Ship(this,vx,vy,v1x-dx,v1y-dy);
                    ++shipdetect;
                    break;
                }
            }
            else if (shipdetect == 1)
                shipdetect = 0;

            break;
        }
        if (op <= 0xa) ++pc;
        if (op != 0xe) ++pc;
    }
    }


    /// \brief Description of something to aim for or shot at.
    class Aim {
    public:
        int dx;
        int dy;
        double dist;
        double weight;
        MovingObject* target;

        Aim() : dx(0), dy(0), dist(1000.), weight(0.), target(0) {}
    };

    /// Analyse this frame for a direction to aim at.
    Aim aimForTargets();

    /// Analyse this frame for evasion measures.
    void evasionMeasures(KeysPacket& keys);
};

#include "FrameStatus_linkFrame.h"
#endif
