/*
  c't creativ-Wettbewerb 2008 / Asteroids.

  Dirk Farin / dirk.farin@gmail.com
 */

#include "ai.h"
#include "player.h"

#include <math.h>

#include <iostream>

using namespace std;

const int xMin = 0;
const int xMax = 1023;
const int yMin = 128;  // unten
const int yMax = 895;  // oben

const bool WINDOW_OUTPUT=false;

const int MAX_IMPACT_TIME=200;
const int SHIP_SIZE=12;
const double ANGLE_PER_STEP = 4.21875;

const int SAUCER_RADIUS_SMALL = 8;
const int SAUCER_RADIUS_LARGE = 12;

const int SHOT_START_DISTANCE_THRESHOLD = 40;

const struct { short x,y; } shottab[256][72] = {
#include "tab.cpp"
} ;

const int logg=0;


AIAsteroid::AIAsteroid()
  : newcnt(0)
{
}


AIGameStatus::AIGameStatus()
  : ship_present(false),
    saucer_present(false),
    nAsteroids(0),
    nShots(0),
    timestep(0)
{
}


void checkParticleOrigin(const Position& particle_pos,
			 const Vector&   particle_v8,
			 const Position& potential_source,
			 Position& closest_point,
			 double&   travel_time)
{
  const Position& p = particle_pos;
  const Vector&   v = particle_v8;
  const Position& s = potential_source;
  double& t = travel_time;

  // t = (s-p)^T v / ( v^T v )

  t = ((s.x-p.x)*v.x + (s.y-p.y)*v.y)*8.0/(v.x*v.x + v.y*v.y);

  closest_point.x = p.x + t*v.x/8;
  closest_point.y = p.y + t*v.y/8;
}


/* note: do not try to estimate a shot-age that is already far away.
   This may give inaccurate (to small) age.
*/
double checkShotAge(const Position& particle_pos,
		    const Vector&   particle_v8___________UNUSED,
		    const Position& potential_source)
{
  Position p = particle_pos;
  p.x *= 8;
  p.y *= 8;

  Position s = potential_source;
  s.x *= 8;
  s.y *= 8;

  double dist = hypot(s.x-p.x, s.y-p.y);
  dist -= 95; // initial offset
  dist -= 64; // first (implicit step)

  return dist/64 + 0.75;
  /* +0.75 since it seems we are always a bit too small (maybe because
     of the coordinate rounding in the ROM ? */
}


void PositionHist::pushPosition(int x,int y, int nFrameDrops, int maxHistLen)
{
  Position p;
  p.x=x; p.y=y;
  pushPosition(p, nFrameDrops, maxHistLen);
}

void PositionHist::pushPosition(const Position& p, int nFrameDrops, int maxHistLen)
{
  // if we had framedrops, duplicate previous positions ...

  if (histlen>0)
    for (int i=0;i<nFrameDrops;i++)
      pushPosition(predictPosition(),0,maxHistLen);


  // shorten history if full

  if (histlen == maxHistLen)
    histlen--;

  // shift history queue

  for (int i=histlen;i>=1;i--)
    pos[i] = pos[i-1];

  // add new position

  histlen++;
  pos[0] = p;

  if (histlen>1)
    {
      int xwrap=0;
      int ywrap=0;

      if (abs(pos[0].x - pos[1].x) > 512)
	{
	  if (pos[0].x < pos[1].x) xwrap=-1024; // wrap to left side
	  else                     xwrap=+1024;
	}

      if (abs(pos[0].y - pos[1].y) > 256)
	{
	  if (pos[0].y < pos[1].y) ywrap=-768;
	  else                     ywrap=+768;
	}

      if (xwrap || ywrap)
	{
	  for (int i=1;i<histlen;i++)
	    {
	      pos[i].x += xwrap;
	      pos[i].y += ywrap;
	    }
	}

      estimSpeed();
    }
}


void PositionHist::estimSpeed()
{
  if (histlen==0)
    { v8.x = v8.y = 0; }
  else
    {
      v8.x = pos[0].x - pos[histlen-1].x;
      v8.y = pos[0].y - pos[histlen-1].y;
      v8.x *= 8;
      v8.y *= 8;
      v8.x /= histlen;
      v8.y /= histlen;
    }
}

Position PositionHist::predictPosition(int nFramesInAdvance) const
{
  Position p = Pos(0);
  p.x += nFramesInAdvance*v8.x/8;
  p.y += nFramesInAdvance*v8.y/8;
  return p;
}

static int mod9Dist(int dx,int dy)
{
  if (dx<-512) dx+=1024;
  if (dx> 512) dx-=1024;
  if (dy<-384) dy+= 768;
  if (dy> 384) dy-= 768;

  return dx*dx+dy*dy;
}

int findClosestAsteroid(const AIAsteroid* oldast, int nAsteroids,
			Asteroid a, int framedrops)
{
  int bestidx=-1;
  int mindist=1000*1000;

  for (int i=0;i<nAsteroids;i++)
    if (a.sf   == oldast[i].sf &&
	a.type == oldast[i].type)
      {
	Position predPos = oldast[i].p.predictPosition( 1 + framedrops );
	int diffX = predPos.x - a.x;
	int diffY = predPos.y - a.y;

	int dist = mod9Dist(diffX,diffY);

	if (dist < mindist && dist < 50*50)
	  {
	    mindist=dist;
	    bestidx=i;
	  }
      }

  return bestidx;
}


int findClosestShot(const AIShot* oldshots, int nShots,
		    int x,int y, int framedrops)
{
  int bestidx=-1;
  int mindist=1000*1000;

  for (int i=0;i<nShots;i++)
    {
      Position predPos = oldshots[i].predictPosition( 1+framedrops );
      int diffX = predPos.x - x;
      int diffY = predPos.y - y;

      int dist = mod9Dist(diffX,diffY);

      int accuracy = 50;
      if (oldshots[i].historyLen()>=10) accuracy=10;

      if (dist < mindist && dist < accuracy*accuracy)
	{
	  mindist=dist;
	  bestidx=i;
	}
    }

  return bestidx;
}


void AIGameStatus::updateGameStatus(const class GameStatus& newStatus, int framedrops)
{
  timestep++;
  timestep += framedrops;

  if (logg) cout << "TIME: " << timestep << endl;

  // update player's ship

  ship_present = newStatus.ship_present;

  if (ship_present == false)
    shipPos.clear();
  else
    shipPos.pushPosition(newStatus.ship_x, newStatus.ship_y, framedrops);


  // update saucer

  saucer_present = newStatus.saucer_present;

  if (saucer_present == false)
    saucer.p.clear();
  else
    {
      saucer.p.pushPosition(newStatus.saucer_x, newStatus.saucer_y, framedrops, 10);
      saucer.radius = ((newStatus.saucer_size==15) ? SAUCER_RADIUS_LARGE : SAUCER_RADIUS_SMALL);
    }


  // update asteroids

  AIAsteroid oldast[MAX_ASTEROIDS];
  for (int i=0;i<nAsteroids;i++)
    oldast[i] = asteroid[i];

  if (logg) cout << "nAsteroids: " << newStatus.nasteroids << endl;

  for (int i=0;i<newStatus.nasteroids;i++)
    {
      const Asteroid& a = newStatus.asteroids[i];

      int idx = findClosestAsteroid(oldast, nAsteroids, a, framedrops);

      if (idx>=0)
	asteroid[i] = oldast[idx];
      else
	{
	  asteroid[i].p.clear();
	  switch (a.sf)
	    {
	    case 0:  asteroid[i].radius = 31; break;
	    case 15: asteroid[i].radius = 15; break;
	    case 14: asteroid[i].radius =  7; break;
	    }

	  asteroid[i].sf   = newStatus.asteroids[i].sf;
	  asteroid[i].type = newStatus.asteroids[i].type;
	  asteroid[i].newcnt = 25;
	}

      asteroid[i].p.pushPosition(a.x, a.y, framedrops);
    }

  nAsteroids = newStatus.nasteroids;


  // update shots

  AIShot oldshots[MAX_SHOTS];
  for (int i=0;i<nShots;i++)
    oldshots[i] = shotPos[i];

  for (int i=0;i<newStatus.nshots;i++)
    {
      const Shot& s = newStatus.shots[i];

      int idx = findClosestShot(oldshots, nShots, s.x, s.y, framedrops);

      if (idx>=0)
	shotPos[i] = oldshots[idx];
      else
	{
	  shotPos[i].clear();
	  shotPos[i].startframe = -1;
	  shotPos[i].origin = AIShot::unknown;
	}

      if (shotPos[i].historyLen() >= 10 && shotPos[i].origin==AIShot::unknown)
	{
	  if (!ship_present)
	    shotPos[i].origin = AIShot::saucer;
	  else
	    {
	      Position closest_point;
	      double travel_time;

	      checkParticleOrigin(shotPos[i].Pos(), shotPos[i].estimSpeed8(),
				  shipPos.Pos(),
				  closest_point, travel_time);

	      if (travel_time < 0 && travel_time > -80 &&
		  sqrDistance(closest_point, shipPos.Pos()) < SHOT_START_DISTANCE_THRESHOLD*SHOT_START_DISTANCE_THRESHOLD)
		{ shotPos[i].origin = AIShot::ship; }
	      else
		{ shotPos[i].origin = AIShot::saucer; }

	      //cout << "travel_time = " << travel_time << endl;
	      //cout << "  dist = " << sqrt(sqrDistance(closest_point, shipPos.Pos())) << endl;
	    }
	}

      //cout << "hist " << shotPos[i].historyLen() << " type=" << shotPos[i].origin << endl;

      shotPos[i].pushPosition(s.x, s.y, framedrops);
    }

  nShots = newStatus.nshots;


  if (ship_present)
    for (int i=0;i<nShots;i++)
      {
	Position closest_point;
	double travel_time=0.0;
	double age=0.0;

	if (shotPos[i].origin == AIShot::ship &&
	    shotPos[i].historyLen()> 5 &&
	    shotPos[i].historyLen()<25)
	  {
	    checkParticleOrigin(shotPos[i].Pos(), shotPos[i].estimSpeed8(),
				shipPos.Pos(),
				closest_point, travel_time);
	    age = checkShotAge(shotPos[i].Pos(), shotPos[i].estimSpeed8(),
			       shipPos.Pos());

	    if (travel_time < 0)
	      {
		if (logg)
		  {
		    cout << "SHOT " << i;
		    cout << " this was shot in frame " << timestep-int(age); //-2;

		    cout << " displayed shot (" << i << "): "
			 << shotPos[i].Pos().x << "; "
			 << shotPos[i].Pos().y << " ... " << travel_time
			 << " age=" << age;
		  }

		shotPos[i].startframe = timestep-int(age);
	      }

	    if (logg) cout << endl;
	  }
      }
}



AIControl::AIControl()
  : state(state_dead),
    nMarks(0),
    nSalve(0),
    last_calibration_shot(-1)
    //planActive(false)
{
#if 0
  if (WINDOW_OUTPUT)
    win.Create(1024,768,"debug");

  img.Create(1024,768,Colorspace_RGB);
#endif
}


bool AIControl::aimCollision(const Position& ship, double angle,
			     const Target& target,
			     double& time_shot, double& time_target) const
{
  const double SHOOTSPEED_CORRECTION_FACTOR = 1.00;

  const double vx = cos(angle) * SHOOTSPEED_CORRECTION_FACTOR;
  const double vy = sin(angle) * SHOOTSPEED_CORRECTION_FACTOR;
  
  Position p0 = ship;
  p0.x *= 8;
  p0.y *= 8;
  p0.x += vx*95;
  p0.y += vy*95;
  p0.x += vx*64;
  p0.y += vy*64;

  const Vector vast = target.p.estimSpeed8();

  {
    double rhs_x = target.p.Pos().x*8 - p0.x;
    double rhs_y = target.p.Pos().y*8 - p0.y;

    double a = vx*64.0;
    double c = vy*64.0;

    double b = -vast.x;
    double d = -vast.y;

    /*
    cout << endl;
    cout << a << " " << b << "  ---  " << rhs_x << endl;
    cout << c << " " << d << "  ---  " << rhs_y << endl;
    */
    double det = 1.0/(a*d-b*c);
    double A =  d*det;
    double B = -b*det;
    double C = -c*det;
    double D =  a*det;

    double alpha = A*rhs_x + B*rhs_y;
    double t     = C*rhs_x + D*rhs_y;

    time_shot = alpha;
    time_target = t;
    //cout << alpha << " | " << t << endl;
  }

  time_target -= 2;

  return (time_shot>0 && time_target>0);
}



Plan AIControl::planShotTab(const Position& ship, unsigned char current_angle,
			    const Target& target, int besttime)
{
  bool solution_found=false;
  int  best_total_time;
  int  timeShot;

  Plan plan;

  if (logg)
    {
      cout << "----------- PLAN -------------\n";
      cout << "ship = " << ship.x*8 << ";" << ship.y*8 << endl;
      cout << "target = " << target.p.Pos().x*8 << ";" << target.p.Pos().y*8 << endl;
      cout << "t.speed = " << target.p.estimSpeed8().x << ";" << target.p.estimSpeed8().y << endl;
    }

  assert(ship.x == 524);
  assert(ship.y == 524);

  for (int t=0;t<70;t++)
    {
      Position tpos = target.p.Pos()*8;
      tpos = tpos + target.p.estimSpeed8() * (t+2);
      tpos = tpos/8;

      doOverflow(tpos);

      int approxdist = checkShotAge(tpos, Vector(), ship);
      int mindist = max(approxdist-10,  0);
      int maxdist = min(approxdist+10, 72);

      int range=50;
      if (besttime<range) range=besttime;

      for (int a=current_angle-range;a<=current_angle+range;a++)
	for (int dist=mindist;dist<maxdist;dist++)
	  {
	    int angle=a;
	    if (angle<0) angle+=256;
	    if (angle>255) angle-=256;

	    Position spos;
	    spos.x = shottab[angle][dist].x;
	    spos.y = shottab[angle][dist].y;

	    if (abs(tpos.x-spos.x) < target.radius &&
		abs(tpos.y-spos.y) < target.radius &&
		sqrDistance(tpos, spos) < target.radius*target.radius*0.75)
	      {
		int adiff1 = angle - current_angle;
		int adiff2 = adiff1+256;
		int adiff3 = adiff1-256;
		int adiff;
		/**/ if (abs(adiff1) <= abs(adiff2) && abs(adiff1) <= abs(adiff3)) adiff=adiff1;
		else if (abs(adiff2) <= abs(adiff1) && abs(adiff2) <= abs(adiff3)) adiff=adiff2;
		else adiff=adiff3;

		int rotSteps = adiff;
		int shotTime = dist;
		int rotTime  = abs(rotSteps);
		int waitTime = t-shotTime-rotTime-2;

		if (waitTime>=0)
		  {
		    int total_time = rotTime + waitTime + shotTime/2;

		    if (!solution_found || total_time < best_total_time)
		      {
			best_total_time = total_time;
			solution_found = true;

			plan.nRotate = rotSteps;
			plan.nWaitFrames = waitTime;
			plan.nShotFrames = shotTime;
			plan.nShotSeq = 1;
			plan.valid = true;
			timeShot = shotTime;
			//planActive = true;
		      }
		  }
	      }
	  }

    }

#if 0
  if (plan.valid)
    {
      cout << "SIMULATION ----------------------------------\n";

      cout << "Plan: rotate=" << plan.nRotate << "  wait=" << plan.nWaitFrames << " shottime=" << timeShot << endl;

      Position sastr = target.p.Pos()*8;
      sastr = sastr + target.p.estimSpeed8()*2;  // position after turnaround time
      sastr = sastr + target.p.estimSpeed8()*abs(plan.nRotate);
      sastr = sastr + target.p.estimSpeed8()*abs(plan.nWaitFrames);
      sastr = sastr + target.p.estimSpeed8()*abs(plan.nShotFrames);

      double angle = current_angle + plan.nRotate*ANGLE_PER_STEP*M_PI/180;
      Vector shotv;
      shotv.x = cos(angle)*64;
      shotv.y = sin(angle)*64;

      Position sship = ship*8;
      Position sshot = sship + shotv*(95+64)/64;
      sshot = sshot + shotv*plan.nShotFrames;

      cout << "position shot:     " << sshot << endl;
      cout << "position asteroid: " << sastr << endl;
    }
#endif

  return plan;
}


bool AIControl::checkCollisionTab(const Position& ship, unsigned char angle_byte,
				  const Target& target) const
{
  const Vector vast = target.p.estimSpeed8();

  for (int t=0;t<72;t++)
    {
      Position pshot;
      pshot.x = shottab[angle_byte][t].x;
      pshot.y = shottab[angle_byte][t].y;

      Position past = target.p.Pos();
      past.x += (t+2)*vast.x/8;
      past.y += (t+2)*vast.y/8;

      double dist = sqrDistance(pshot, past);

      if (dist<target.radius * target.radius * 0.95)
	{
	  /*
	  AIControl* th = const_cast<AIControl*>(this);
	  th->mark[0] = p0;
	  th->mark[0].x += vx*8;
	  th->mark[0].y += vy*8;
	  th->mark[1] = pshot;
	  th->mark[2] = p0;
	  th->mark[2].x += vx*8 * 30;
	  th->mark[2].y += vy*8 * 30;
	  for (int i=0;i<3;i++) { th->mark[0].x /= 8; th->mark[0].y /= 8; }
	  th->nMarks=3;
	  th->markcnt=500;
	  */

	  //cout << "t=" << t << endl;

	  return true;
	}
    }

  return false;
}



int AIControl::collisionTime(const Target& target, const Position& ship) const
{
  const Vector vast = target.p.estimSpeed8();

  double dist = sqrDistance(ship, target.p.Pos());
  if (dist< (4*SHIP_SIZE) * (4*SHIP_SIZE))  // TODO: proper calculation with ship size
    return 0;

  for (int t=0;t<MAX_IMPACT_TIME;t++)
    {
      // TODO: assumes a ship-velocity of zero

      Position past = target.p.Pos();
      past.x += t*vast.x/8.0;
      past.y += t*vast.y/8.0;

      dist = sqrDistance(ship, past);

      if (dist< (target.radius+SHIP_SIZE) * (target.radius+SHIP_SIZE)*2)  // TODO: proper calculation with ship size
	{
	  return t;
	}
    }

  return MAX_IMPACT_TIME;
}



      const double angletab[256]=
    {
      0.0,    3.6,    8.2,   12.7,   16.5,   21.2,   25.2,   29.5,   33.5,   37.8,   41.7,   46.2,   50.8,   55.3,   59.4,   63.4,
	67.5,   71.6,   76.1,   79.9,   84.5,   89.1,   92.8,   97.2,  101.8,  105.5,  110.2,  114.1,  118.1,  122.8,  126.7,  131.1,
	135.0,  139.4,  143.3,  147.9,  151.9,  156.6,  160.7,  164.5,  169.2,  172.8,  177.3,  181.9,  185.5,  189.9,  194.5,  198.2,
	203.0,  206.9,  211.0,  215.4,  220.0,  223.8,  227.5,  232.2,  235.9,  240.3,  245.1,  248.8,  252.7,  257.3,  261.1,  265.6,
	270.0,  273.6,  278.2,  282.7,  286.5,  291.2,  295.0,  299.0,  303.5,  307.8,  311.7,  316.2,  320.0,  324.5,  328.6,  332.7,
	336.6,  341.6,  345.3,  349.9,  354.5,  358.1,
	2.8,    7.2,   11.0,   15.5,   19.3,   23.4,   28.1,   32.8,   36.7,   40.6,   45.0,   49.4,   53.3,   57.2,   61.9,   66.6,
	70.7,   74.5,   79.0,   82.8,   87.2,   91.9,   95.5,  100.1,  104.7,  108.4,  113.4,  117.3,  121.4,  125.6,  130.0,  133.8,
	138.3,  142.2,  146.5,  151.0,  155.0,  158.8,  163.6,  167.3,  171.8,  176.6,  180.0,  184.4,  188.9,  192.7,  197.3,  201.2,
	205.0,  209.7,  214.2,  217.8,  222.5,  226.2,  230.0,  234.6,  239.0,  243.1,  247.0,  251.8,  255.5,  260.1,  264.5,  268.1,
	272.7,  277.2,  280.8,  285.5,  289.3,  293.4,  298.1,  302.2,  306.7,  310.6,  315.0,  318.9,  323.3,  327.2,  331.9,  335.9,
	339.8,  344.5,  348.2,  352.8,  357.2,
	0.9,    5.5,   10.1,   13.9,   18.4,   22.5,   26.6,   30.6,   34.7,   39.2,   43.8,   48.3,   52.2,   56.5,   60.5,   64.7,
	68.8,   73.5,   77.3,   81.8,   86.4,   90.0,   94.5,   98.9,  102.7,  107.3,  111.2,  115.3,  120.2,  124.2,  127.8,  132.5,
	136.1,  140.8,  145.3,  149.8,  153.9,  157.9,  161.8,  166.3,  170.1,  174.3,  179.1,  182.9,  187.2,  191.5,  195.6,  200.3,
	204.1,  208.1,  212.1,  216.6,  221.0,  225.0,  228.9,  233.3,  237.8,  241.9,  245.9,  249.8,  254.5,  258.4,  262.8,  267.3,
	270.9,  275.5,  279.9,  283.7,  288.2,  292.1,  296.1,  300.2,  304.7,  309.2,  313.8,  317.5,  322.2,  325.8,  329.8,  334.7,
	338.8,  342.7,  347.2,  351.1,  355.5
	};


      static int asteroidSF2ShotSeq(int sf)
    {
      if (sf== 0) return 2*4-1;
      if (sf==15) return 2*3-1;
      return 1;
    }

KeysPacket AIControl::process(const class GameStatus& newStatus, int framedrops)
{
  status.updateGameStatus(newStatus, framedrops);
  showDebugImg();

  const int t=status.timestep+2;

  KeysPacket keys;
  keys.clear();

  bool fired_this_frame=false;

  // state machine

  switch (state)
    {
    case state_dead:
      if (status.ship_present)
	state=state_normalplay;
      break;

    case state_normalplay:
      if (!status.ship_present)
	{
	  state=state_dead;
	  if (logg) cout << "--------------- CRASH ----------\n";
	}
      break;
    }


  // calibrate angle ship angle

  for (int i=0;i<status.nShots;i++)
    if (status.shotPos[i].startframe > last_calibration_shot &&
	status.shotPos[i].historyLen() >= 10)
      {
	for (int a=0;a<256;a++)
	  {
	    bool match=true;
	    int h = status.shotPos[i].historyLen();
	      for (int k=0;k<h && match;k++)
		if (shottab[a][k].x != status.shotPos[i].Pos(h-1-k).x ||
		    shottab[a][k].y != status.shotPos[i].Pos(h-1-k).y)
		  match=false;

	    if (match)
	      {
		int my_angle_byte = a;

		if (logg)
		  {
		    cout << "CALIBRATION FOUND\n";
		    cout << "startframe(" << i << ")=" << status.shotPos[i].startframe << " last_shot=" << last_calibration_shot << endl;
		    cout << "shot angle-byte=" << ((int)my_angle_byte) << " expected: " << int(anglebuf[status.shotPos[i].startframe % ANGLE_BUF_SIZE]) << endl;
		  }

		last_calibration_shot = status.shotPos[i].startframe;

		unsigned char diff = my_angle_byte - anglebuf[status.shotPos[i].startframe % ANGLE_BUF_SIZE];
		if (logg) cout << "correction = " << int(diff) << " -/- " << int(diff)-256 << endl;

		if (diff)
		  {
		    for (int k=0;k<ANGLE_BUF_SIZE;k++)
		      anglebuf[k] += diff;

		    angle_byte += diff;
		    angle=angletab[angle_byte]/180*M_PI;
		  }

		break;
	      }
	  }
#if 0
	double my_angle = atan2(status.shotPos[i].estimSpeed8().y,
				status.shotPos[i].estimSpeed8().x);

	if (my_angle<0) my_angle+=2*M_PI;

	unsigned char my_angle_byte;

	for (int k=0;k<256;k++)
	  if (abs(angletab[k]-my_angle*180/M_PI) < abs(angletab[my_angle_byte]-my_angle*180/M_PI))
	    my_angle_byte = k;

#endif
      }


  // remember angle used in this shot
  anglebuf[t%ANGLE_BUF_SIZE] = angle_byte;


#if 1
  // check if we have to hyperspace ...

  if (0)
    {
      const int SAFETY_RADIUS = 25;

      if (status.ship_present)
	{
	  for (int i=0;i<status.nAsteroids;i++)
	    {
	      int radiussum = status.asteroid[i].radius + SAFETY_RADIUS; // 40 = safety-circle
	      if (sqrDistance(status.asteroid[i].p.Pos(), status.shipPos.Pos()) < radiussum*radiussum)
		{
		  keys.hyperspace(true);
		  plan.valid=false;
		  status.ship_present=false;
		}
	    }

	  for (int i=0;i<status.nShots;i++)
	    {
	      int radiussum = 1.5*SAFETY_RADIUS; // 40 = safety-circle
	      if (status.shotPos[i].origin == AIShot::saucer &&
		  sqrDistance(status.shotPos[i].Pos(), status.shipPos.Pos()) < radiussum*radiussum)
		{
		  keys.hyperspace(true);
		  plan.valid=false;
		  status.ship_present=false;
		}
	    }
	}
    }

  // process shooting according to plan

  if (plan.valid)
    {
      if (!fired_last_frame)
	{
	  assert(!fired_last_frame);

	  if (plan.nRotate==0 && plan.nWaitFrames<=0 && (plan.nShotSeq&1))
	    {
	      keys.fire(true);
	      fired_this_frame=true;
	    }
	}

      if (plan.nRotate==0 && plan.nWaitFrames<=0)
	{
	  plan.nShotSeq--;
	  if (logg) cout << "PLAN ........................... " << plan.nShotSeq << endl;

	  if (plan.nShotSeq==0)
	    plan.valid=false;
	}
    }

  if (status.ship_present)
    {
      for (int i=0;i<status.nAsteroids;i++)
	{
	  if (status.asteroid[i].underFire < t-70)
	    {
	      if (checkCollisionTab(status.shipPos.Pos(), angle_byte,
				    status.asteroid[i]))
		//if (checkCollision(status.shipPos.Pos(), angle,
		//	 status.asteroid[i]))
		{
		  status.asteroid[i].underFire = t;
		  keys.fire(true);
		  //showExpectedFirePath();
		  fired_this_frame=true;

		  ///**/ if (status.asteroid[i].sf== 0) { nSalve=4; salve[0]=t+2; salve[1]=t+4; salve[2]=t+6; salve[3]=t+8; }
		  //else if (status.asteroid[i].sf==15) { nSalve=3; salve[0]=t+2; salve[1]=t+4; salve[2]=t+6;               }
		}
	    }
	  else
	    {} // cout << "already under fire " << status.asteroid[i].underFire << " / t=" << t << endl;
	}
    }

  Target* target=NULL;

  if (!plan.valid && status.saucer_present && status.ship_present)
    {
      target = &status.saucer;
      //plan = planShot(status.shipPos.Pos(), angle, *target);
      plan = planShotTab(status.shipPos.Pos(), angle_byte, *target);
      plan.nShotSeq=1;
    }

  if (!plan.valid && status.ship_present) // && (t%200)==0)
    {
      int min_impact_time = MAX_IMPACT_TIME;
      for (int i=0;i<status.nAsteroids;i++)
	{
	  int impact_time = collisionTime(status.asteroid[i], status.shipPos.Pos());
	  if (impact_time < min_impact_time)
	    {
	      min_impact_time=impact_time;
	      target = &status.asteroid[i];

	      //plan = planShot(status.shipPos.Pos(), angle, status.asteroid[i]);
	      plan = planShotTab(status.shipPos.Pos(), angle_byte, status.asteroid[i]);
	      plan.nShotSeq=asteroidSF2ShotSeq(status.asteroid[i].sf);

	      // cout << "collision detected, asteroid " << idx << endl;
	    }
	}
    }

  if (!plan.valid && status.ship_present) // && (t%200)==0)
    {
      for (int i=0;i<status.nAsteroids;i++)
	if (status.asteroid[i].underFire < t-70)
	  {
	    //Plan asterplan = planShot(status.shipPos.Pos(), angle, status.asteroid[i]);
	    Plan asterplan = planShotTab(status.shipPos.Pos(), angle_byte, status.asteroid[i], plan.time());
	    asterplan.nShotSeq=asteroidSF2ShotSeq(status.asteroid[i].sf);

	    if (asterplan.valid)
	      {
		if (!plan.valid || asterplan.time() < plan.time())
		  {
		    plan = asterplan;
		    target = &status.asteroid[i];
		  }
	      }
	  }
    }

  if (target)
    {
      target->underFire = t;
    }

  if (plan.valid)
    {
      if (plan.nRotate)
	{
	  /**/ if (plan.nRotate>0) { keys.left (true); angle_byte++; }
	  else if (plan.nRotate<0) { keys.right(true); angle_byte--; }

	  /**/ if (plan.nRotate<0) { plan.nRotate++; }
	  else if (plan.nRotate>0) { plan.nRotate--; }
	}
      else
	{
	  if (plan.nWaitFrames>0)
	    plan.nWaitFrames--;
	}
    }

  // rotate of no clue what to do
  if (!plan.valid)
    {
      keys.left (true); angle_byte++;
    }
#endif

#if 0
  // check for possible hits

  if (state==state_normalplay)
    {
      if (nSalve>0)
	{
	  if (t>=salve[0])
	    {
	      keys.fire(true);
	      showExpectedFirePath();
	      fired_this_frame=true;

	      nSalve--;
	      for (int i=0;i<nSalve;i++)
		salve[i]=salve[i+1];
	    }
	}
      else
	{
	  if (status.saucer_present)
	    {
	      if (status.saucer.underFire < t-70)
		{
		  if (checkCollision(status.shipPos.Pos(), angle, status.saucer))
		    {
		      status.saucer.underFire = t;
		      keys.fire(true);
		      showExpectedFirePath();
		      fired_this_frame=true;
		    }
		}
	    }

	  if (status.ship_present)
	    {
	      for (int i=0;i<status.nAsteroids;i++)
		{
		  if (status.asteroid[i].underFire < t-70)
		    {
		      if (checkCollision(status.shipPos.Pos(), angle,
					 status.asteroid[i]))
			{
			  status.asteroid[i].underFire = t;
			  keys.fire(true);
			  showExpectedFirePath();
			  fired_this_frame=true;

			  /**/ if (status.asteroid[i].sf== 0) { nSalve=4; salve[0]=t+2; salve[1]=t+4; salve[2]=t+6; salve[3]=t+8; }
			  else if (status.asteroid[i].sf==15) { nSalve=3; salve[0]=t+2; salve[1]=t+4; salve[2]=t+6;               }
			}
		    }
		  else
		    {} // cout << "already under fire " << status.asteroid[i].underFire << " / t=" << t << endl;
		}
	    }
	}

      int min_dx=9999,min_dy=9999;
      Target* target = NULL;

      if (!target)
	{
	  int min_impact_time = MAX_IMPACT_TIME;
	  for (int i=0;i<status.nAsteroids;i++)
	    {
	      int impact_time = collisionTime(status.asteroid[i], status.shipPos.Pos());
	      if (impact_time < min_impact_time)
		{
		  min_impact_time=impact_time;
		  target = &status.asteroid[i];

		  // cout << "collision detected, asteroid " << idx << endl;
		}
	    }
	}

      if (!target && status.saucer_present)
	{
	  target = &status.saucer;
	}

      if (!target)
	for (int i=0;i<status.nAsteroids;i++)
	  {
	    int dx = status.asteroid[i].p.Pos().x - status.shipPos.Pos().x;
	    int dy = status.asteroid[i].p.Pos().y - status.shipPos.Pos().y;

	    if (hypot(dx,dy) < hypot(min_dx,min_dy))
	      {
		min_dx=dx; min_dy=dy;
		target = & status.asteroid[i];
	      }
	  }


      if (target)
	{
	  int rotation, waitframes;
	  planShot(status.shipPos.Pos(), angle,
		   *target, rotation, waitframes);
	}

      // now turn the ship

      if (target && /* !fired_this_frame &&*/ nSalve==0)
	{
	  int dx = target->p.Pos().x - status.shipPos.Pos().x;
	  int dy = target->p.Pos().y - status.shipPos.Pos().y;

	  int towards;

	  if (cos(angle) * dy - sin(angle) * dx > 0)
	    towards=1;
	  else
	    towards=-1;

	  int turn=0;

	  double time_shot, time_asteroid;
	  if (aimCollision(status.shipPos.Pos(), angle,
			   *target,
			   time_shot, time_asteroid))
	    {
	      {
		if (time_shot > time_asteroid)
		  {
		    turn = -towards;
		  }
		else
		  {
		    turn = towards;
		  }
	      }
	    }
	  else
	    {
	      turn=towards;
	    }

	  if (turn>0)
	    {
	      keys.left(true);
	      angle_byte++; if (angle_byte>255) angle_byte=0;
	    }
	  else if (turn<0)
	    {
	      keys.right(true);
	      angle_byte--; if (angle_byte<0) angle_byte=255;
	    }
	}
    }
#endif

  angle=angletab[angle_byte]/180*M_PI;
  fired_last_frame=fired_this_frame;

  return keys;
}


void AIControl::showExpectedFirePath() const
{
  if (logg) cout << "FIRE\n";
  return;

  cout << "expected shot path:\n";
  for (int t=0;t<40;t++)
    {
      const double vx = cos(angle);
      const double vy = sin(angle);
  
      Position p = status.shipPos.Pos();
      p.x += vx*95/8.0;
      p.y += vy*95/8.0;
      p.x += vx*64/8.0;
      p.y += vy*64/8.0;

      p.x += t*vx*8.0;
      p.y += t*vy*8.0;

      cout << t << ": " << p.x << "; " << p.y << endl;
    }
}


#if 0
void AIControl::drawHist(Image<Pixel>& img,
			 const PositionHist& hist,
			 const Color<Pixel>& col)
{
  for (int i=0;i<hist.historyLen()-1;i++)
    {
      if (abs(game2scrX(hist.Pos(i  ).x)-game2scrX(hist.Pos(i+1).x))<300 &&
	  abs(game2scrY(hist.Pos(i  ).y)-game2scrY(hist.Pos(i+1).y))<300)
	{
	  DrawLine(img,
		   game2scrX(hist.Pos(i  ).x), game2scrY(hist.Pos(i  ).y),
		   game2scrX(hist.Pos(i+1).x), game2scrY(hist.Pos(i+1).y),
		   col);
	}
    }

  for (int i=0;i<hist.historyLen();i++)
    DrawCircle(img, game2scrX(hist.Pos(i).x), game2scrY(hist.Pos(i).y), 1, col);
}
#endif

void AIControl::showDebugImg()
{
#if 0
  if (!WINDOW_OUTPUT) return;

  // initialization

  bool centered=false;  // OPTION

  if (!centered)
    {
      offsetX = 0;
      offsetY = -128;
    }
  else
    {
      if (status.shipPos.historyLen()>0)
	{
	  offsetX = -status.shipPos.Pos().x;
	  offsetY = -status.shipPos.Pos().y;
	  offsetX += xMax/2;
	  offsetY += (yMax-yMin)/2;
	}
    }

  Clear(img, Color<Pixel>(0,0,0));

  // draw ship

  if (status.ship_present)
    {
      drawHist(img, status.shipPos, Color<Pixel>(180,0,0));
      DrawCircle(img,
		 game2scrX(status.shipPos.Pos().x),
		 game2scrY(status.shipPos.Pos().y),
		 10, Color<Pixel>(255,0,0), true);
    }

  // draw asteroids

  for (int i=0;i<status.nAsteroids;i++)
    {
      drawHist(img, status.asteroid[i].p, Color<Pixel>(0,120,0));

      Color<Pixel> col;
      if (status.asteroid[i].newcnt)
	{ col = Color<Pixel>(0,0,255); status.asteroid[i].newcnt--; }
      else if (status.asteroid[i].underFire > status.timestep-70)
	{ col = Color<Pixel>(200,200,0); }
      else
	{ col = Color<Pixel>(0,200,0); }

      const int x = game2scrX(status.asteroid[i].p.Pos().x);
      const int y = game2scrY(status.asteroid[i].p.Pos().y);

      DrawCircle(img, x,y,
		 status.asteroid[i].radius, col, true);

      DrawArrow(img,  x,y,
		x + status.asteroid[i].p.estimSpeed8().x,
		y - status.asteroid[i].p.estimSpeed8().y,
		Color<Pixel>(0,130,0));
    }

  // draw shots

  for (int i=0;i<status.nShots;i++)
    {
      int redvalue = 0;

      if (status.shotPos[i].origin == AIShot::saucer) redvalue=255;
      if (status.shotPos[i].origin == AIShot::ship)   redvalue=100;

      drawHist(img, status.shotPos[i], Color<Pixel>(redvalue,100,100));

      DrawCircle(img,
		 game2scrX(status.shotPos[i].Pos().x),
		 game2scrY(status.shotPos[i].Pos().y),
		 2, Color<Pixel>(200,200,255), true);
    }

  // draw saucer

  if (status.saucer_present)
    {
      DrawCircle(img,
		 game2scrX(status.saucer.p.Pos().x),
		 game2scrY(status.saucer.p.Pos().y),
		 status.saucer.radius, Color<Pixel>(255,0,0), true);
    }

  // draw marks

  if (nMarks>0 && markcnt>0)
    {
      markcnt--;
      for (int i=0;i<nMarks;i++)
	{
	  DrawLine(img,
		   game2scrX(mark[i].x)-5, game2scrY(mark[i].y)-5,
		   game2scrX(mark[i].x)+5, game2scrY(mark[i].y)+5,
		   Color<Pixel>(255,0,0));
	  DrawLine(img,
		   game2scrX(mark[i].x)+5, game2scrY(mark[i].y)-5,
		   game2scrX(mark[i].x)-5, game2scrY(mark[i].y)+5,
		   Color<Pixel>(255,0,0));
	}
    }

  // display image

  win.Display(img);
#endif
}
