// This file is part of "Omniroid", an Asteroids bot written for the 2008 c't anniversary contest
// Omniroid was written by Vladimir "CyberShadow" Panteleev <thecybershadow@gmail.com>
// This file is written in the D Programming Language ( http://digitalmars.com/d/ )

/// Module used for testing code fragments.
module mos6502;

struct Context
{
	ubyte _a, p;

	ubyte a()
	{
		return _a;
	}

	void a(ubyte v)
	{
		_a = v;
		z = v == 0;
		n = (v & 0x80) != 0;
	}

	static string mixinBit(string name, string mask)
	{
		return "
			bool " ~ name ~ "()
			{
				return (p & 0b" ~ mask ~ ") != 0;
			}
			
			/// ditto
			void " ~ name ~ "(bool value)
			{
				if (value)
					p |= 0b" ~ mask ~ ";
				else
					p &= ~ 0b" ~ mask ~ ";
			}";
	}
	
	// only a few of these are used, actually
	/** Carry      */ mixin(mixinBit("c", "00000001"));
	/** Zero       */ mixin(mixinBit("z", "00000010"));
	/** Interrupt  */ mixin(mixinBit("i", "00000100"));
	/** Decimal    */ mixin(mixinBit("d", "00001000"));
	/** Break      */ mixin(mixinBit("b", "00010000"));
	/** Expansion  */ mixin(mixinBit("e", "00100000"));
	/** Overflow   */ mixin(mixinBit("v", "01000000"));
	/** Negative   */ mixin(mixinBit("n", "10000000")); // also "s" in some references

	private static const hex = "0123456789ABCDEF";
	
	string toString()
	{
		return 
			 "A="~hex[a>>4]~hex[a&15]~
			" C="~hex[c]~
			" Z="~hex[z]~
		//	" V="~hex[v]~
			" N="~hex[n];
	}

	static Context opCall(ubyte a)  // also updates flags
	{
		Context c;
		c.a = a;
		return c;
	}
}

// ******************************************************************

/// Set Carry
Context sec(Context c)
{
	c.c = true;
	return c;
}

/// Clear Carry
Context clc(Context c)
{
	c.c = false;
	return c;
}

// ******************************************************************

/// Add with Carry
Context adc(Context c, ubyte value)
{
	ushort r = c.a;
	ushort b = cast(ubyte)c.c + value;
	r += b;
	c.c = (r & 0xFF00) != 0;
	c.a = cast(ubyte)r;

	return c;
}

unittest
{
	Context c;
	c = adc(c, 0x05);
	assert(c.a==0x05 && c.c==false && c.z==false && c.n==false);
	c = adc(c, 0x7F);
	assert(c.a==0x84 && c.c==false && c.z==false && c.n==true );
	c = adc(c, 0x7C);
	assert(c.a==0x00 && c.c==true  && c.z==true  && c.n==false);
}

/// Substract with Carry
Context sbc(Context c, ubyte value)
{
	ushort r = c.a;
	ushort b = (1-cast(ubyte)c.c) + value;
	r -= b;
	c.c = (r & 0xFF00) == 0;
	c.a = cast(ubyte)r;

	return c;
}

unittest
{
	Context c;
	c = sbc(c, 0x04);//+carry
	assert(c.a==0xFB && c.c==false && c.z==false && c.n==true);
	c = sbc(c, 0x7F);
	assert(c.a==0x7B && c.c==true  && c.z==false && c.n==false);
	c = sbc(c, 0x7B);
	assert(c.a==0x00 && c.c==true  && c.z==true  && c.n==false);
}

// ******************************************************************

/// And
Context and(Context c, ubyte value)
{
	c.a = c.a & value;
	return c;
}

/// Or
Context ora(Context c, ubyte value)
{
	c.a = c.a | value;
	return c;
}

/// Exclusive Or
Context eor(Context c, ubyte value)
{
	c.a = c.a ^ value;
	return c;
}

// ******************************************************************

/// Logic Shift Right
Context lsr(ref ubyte a, Context c)
{
	c.c = (a&1)!=0;
	a >>>= 1;
	return c;
}

/// ditto
Context lsr(Context c)
{
	c.c = (c.a&1)!=0;
	c.a = c.a >>> 1;
	return c;
}

/// Rotate Right
Context ror(ref ubyte a, Context c)
{
	bool newc = (a&1)!=0;
	a = c.c ? (a >>> 1) | 0x80 : a >>> 1;
	c.c = newc;
	return c;
}

/// ditto
Context ror(Context c)
{
	bool newc = (c.a&1)!=0;
	c.a = c.c ? (c.a >>> 1) | 0x80 : c.a >>> 1;
	c.c = newc;
	return c;
}

/// Arithmetic Shift Left
Context asl(ref ubyte a, Context c)
{
	c.c = (a&0x80)!=0;
	a <<= 1;
	return c;
}

/// ditto
Context asl(Context c)
{
	c.c = (c.a&0x80)!=0;
	c.a = c.a << 1;
	return c;
}

/// Rotate Right
Context rol(ref ubyte a, Context c)
{
	bool newc = (a&0x80)!=0;
	a = (a<<1) | c.c;
	c.c = newc;
	return c;
}

/// ditto
Context rol(Context c)
{
	bool newc = (c.a&0x80)!=0;
	c.a = (c.a<<1) | c.c;
	c.c = newc;
	return c;
}

// ******************************************************************
