// ============================================================================
// File:               $File$
//
// Project:            
//
// Purpose:            
//
// Author:             Rammi
//
// Copyright Notice:   (c) 2008  Rammi (rammi@caff.de)
//                     This code is in the public domain.
//                     Use at own risk.
//                     No guarantees given.
//
// Latest change:      $Date$
//
// History:	       $Log$
//=============================================================================
package de.caff.asteroid;

import java.awt.*;
import java.awt.geom.AffineTransform;
import java.awt.geom.GeneralPath;
import java.util.Collection;

/**
 *
 *  A moving game object has a position and a velocity and it can be drawn.
 * 
 *  This is part of a solution for a competition
 *  <a href="http://www.heise.de/ct/creativ/08/02/details/">by the German computer magazine c't</a>
 */
public abstract class MovingGameObject
        extends GameObject
{
  /** TEST: calculate maximum velocities? */
  protected static final boolean WATCH_VELOCITIES = false;

  // if you are interested which y coordinates are used change the "if (false)" line in contructor to "if (true)"
  private static int minY = Integer.MAX_VALUE;
  private static int maxY = Integer.MIN_VALUE;

  /** Size of head of velocity arrow. */
  private static final int ARROW_HEAD_SIZE = 8;

  /** The arrow head for velocity drawings. */
  private static final GeneralPath ARROW_HEAD = new GeneralPath();
  static {
    ARROW_HEAD.moveTo(0, 0);
    ARROW_HEAD.lineTo(-ARROW_HEAD_SIZE, ARROW_HEAD_SIZE);
    ARROW_HEAD.lineTo(-ARROW_HEAD_SIZE, -ARROW_HEAD_SIZE);
    ARROW_HEAD.closePath();
  }

  /** The x coordinate of the velocity vector. */
  private int vx;
  private double vxd;
  /** The y coordinate of the velocity vector. */
  private int vy;
  private double vyd;
  
  private int myLifeTime;
  private int myBulletLifeTime;

  /**
   *  Constructor.
   *  @param x  x coordinate
   *  @param y  y coordinate
   */
  protected MovingGameObject(int x, int y)
  {
    super(x, y);
    if (false) {
      if (y < minY) {
        minY = y;
        System.out.println("new min/max: "+minY+","+maxY);
      }
      if (y > maxY) {
        maxY = y;
        System.out.println("new min/max: "+minY+","+maxY);
      }
    }
  }

  /**
   *  Get the center of the object.
   *  @return center point
   */
  public Point getCenter()
  {
    return new Point(x, y);
  }

  /**
   *  Get the size of the object.
   *
   *  The size returned by this method is half the length of a square which contains the object,
   *  so the object's bounding box is between (x - size, y - size) and (x + size, y + size).
   *  @return object size
   */
  public abstract int getSize();

  /**
   *  Draw the object.
   *  @param g graphics context
   */
  public void draw(Graphics2D g)
  {
    g.setColor(Color.white);
    int size = getSize();
    g.drawOval(x - size, y - size, 2*size, 2*size);

    drawVelocityVector(g, Color.red);
  }

  /**
   *  Get the shortest vector from this to another game object.
   *
   *  Because of the torus topology it may be shorter going across the borders.
   *  @param obj other object
   *  @return shortest vector from this to the other object
   */
  public Point getDelta(MovingGameObject obj)
  {
    int dx = obj.x - x;
    if (dx < -EXTENT_X/2) {
      dx += EXTENT_X;
    }
    else if (dx >= EXTENT_X/2) {
      dx -= EXTENT_X;
    }
    int dy = obj.y - y;
    if (dy < -EXTENT_Y/2) {
      dy += EXTENT_Y;
    }
    else if (dy >= EXTENT_Y/2) {
      dy -= EXTENT_Y;
    }
    return new Point(dx, dy);
  }

  /**
   *  Get the shortest vector from this to another game object.
   *
   *  Because of the torus topology it may be shorter going across the borders.
   *  @param obj other object
   *  @param framesInFuture frames in the future (0=now)
   *  @return shortest vector from this to the other object
   */
  public Point getDelta(MovingGameObject obj, int framesInFuture)
  {
    int dx = (int)(obj.x+(obj.getVxd()*framesInFuture) - x);
    if (dx < -EXTENT_X/2) {
      dx += EXTENT_X;
    }
    else if (dx >= EXTENT_X/2) {
      dx -= EXTENT_X;
    }
    int dy = (int)(obj.y+(obj.getVyd()*framesInFuture) - y);
    if (dy < -EXTENT_Y/2) {
      dy += EXTENT_Y;
    }
    else if (dy >= EXTENT_Y/2) {
      dy -= EXTENT_Y;
    }
    return new Point(dx, dy);
  }

  /**
   *  Get the squared size of this object.
   *  @return squared size
   */
  public int getSquaredSize()
  {
    int size = getSize();
    return size*size;
  }

  /**
   *  Set the velocity.
   *
   *  The velocity is the step between frames.
   *
   *  It is not calculated internally but has to be set from outside. For an example see
   *  {@link de.caff.asteroid.SimpleVelocityPreparer#prepareFrames(java.util.LinkedList)}.
   *  @param x x coordinate of velocity
   *  @param y y coordinate of velocity
   */
  public void setVelocity(int x, int y)
  {
    vx = x;
    vy = y;
  }

  /**
   *  Set the velocity.
   *
   *  The velocity is the step between frames.
   *  @param v velocity vector
   */
  public void setVelocity(Point v)
  {
    setVelocity(v.x, v.y);
  }

  /**
   *  Set the velocity assuming that the given object is at the same place as
   *  this object in the last frame.
   *
   *  @param obj comparision object (<code>null</code> allowed, but then nothing happens)
   */
  public void setVelocityFromDelta(MovingGameObject obj)
  {
    if (obj != null) {
      setVelocity(obj.getDelta(this));
      setMyLifeTime(obj.getMyLifeTime());
      setVxd(obj.getVxd());
      setVyd(obj.getVyd());
      
      //System.out.println(obj.getMyBulletLifeTime());
      
/*      if(obj.getMyBulletLifeTime() > 0)
        setMyBulletLifeTime(obj.getMyBulletLifeTime()-1);
      else
    	setMyBulletLifeTime(0); 
*/      
      if(this.getMyLifeTime()<1)
      {
		  vxd= vx;
		  vyd= vy;
      }
      
    }
  }

  public void setVelocityDoubles()
  {
	  if( myLifeTime>0 /*&& myLifeTime<500*/ )
	  {
		  //System.out.println(myLifeTime+")"+vxd+":"+vyd+"|"+vx+":"+vy);

		  vxd= (vxd*myLifeTime+vx)/(myLifeTime+1);
		  vyd= (vyd*myLifeTime+vy)/(myLifeTime+1);
		  myLifeTime++;
		  
	  }
	  else
	  {
		  myLifeTime= 1;
		  vxd= vx;
		  vyd= vy;		  
	  }
  }
  
  /**
   *  Get the velocity in x.
   *  @return x component of velocity vector
   */
  public int getVelocityX()
  {
    return vx;
  }

  /**
   *  Get the velocity in y.
   *  @return y component of velocity vector
   */
  public int getVelocityY()
  {
    return vy;
  }

  /**
   *  Get the velocity vector.
   *  @return velocity vector (movement between frames)
   */
  public Point getVelocity()
  {
    return new Point(vx, vy);
  }

  /**
   *  Get the velocity angle.
   *
   *  The angle is measured counterclockwise, with <code>0</code> pointing to the right
   *  @return velocity in radians, or -Math.PI if ship has no velocity
   */
  public double getVelocityAngle()
  {
    return vx != 0 || vy != 0  ?  Math.atan2(vy, vx)  :  -Math.PI;
  }

  /**
   *  Has this object a velocity.
   *  @return velocity
   */
  public boolean hasKnownVelocity()
  {
    return vx != 0 || vy != 0;
  }

  /**
   *  Draw a vector displaying the current velocity.
   *  @param g graphics context
   *  @param color color to use
   */
  protected void drawVelocityVector(Graphics2D g, Color color)
  {
    if (vx != 0  ||  vy != 0) {
      int scale = 16;
      g.setColor(color);
      g.drawLine(x, y, x + scale*vx, y + scale*vy);
      double angle = Math.atan2(vy, vx);
      AffineTransform at = AffineTransform.getTranslateInstance(x + scale*vx, y + scale*vy);
      at.concatenate(AffineTransform.getRotateInstance(angle));
      g.fill(at.createTransformedShape(ARROW_HEAD));
    }
  }
  /**
   * 
   */
  public Rectangle getBounds(int framesInFuture)
  {
	    Point center;
	    
	    if( hasKnownVelocity() )
	      center = new Point(x+(int)(vx*framesInFuture), y+(int)(vy*framesInFuture));
	    else
	      center = getCenter();
	    
	    int size = getSize();
	    return new Rectangle(center.x - size, center.y - size, 2*size, 2*size);
  }
  
  /**
   * Get the bounding box of this rectangle.
   *
   * @return the bounding box
   */
  public Rectangle getBounds()
  {
    Point center = getCenter();
    int size = getSize();
    return new Rectangle(center.x - size, center.y - size, 2*size, 2*size);
  }

  /**
   * Get the properties of this object.
   *
   * @return collection of properties
   */
  @Override
  public Collection<Property> getProperties()
  {
    Collection<Property> props = super.getProperties();
    props.add(new Property<Integer>("Size", getSize()));
    props.add(new Property<Rectangle>("Bounds", getBounds()));
    props.add(new Property<Point>("Velocity", getVelocity()));
    return props;
  }

public double getVxd() {
	return vxd;
}

public double getVyd() {
	return vyd;
}

public void setVxd(double vxd) {
	this.vxd = vxd;
}

public void setVyd(double vyd) {
	this.vyd = vyd;
}


public int getMyLifeTime() {
	return myLifeTime;
}

public void setMyLifeTime(int lifeTime) {
	this.myLifeTime = lifeTime;
}

public int getMyBulletLifeTime() {
	return myBulletLifeTime;
}

public void setMyBulletLifeTime(int myBulletLifeTime) {
	this.myBulletLifeTime = myBulletLifeTime;
}
}
