// ============================================================================
// File:               $File$
//
// Project:            
//
// Purpose:            
//
// Author:             Rammi
//
// Copyright Notice:   (c) 2008  Rammi (rammi@caff.de)
//                     This code is in the public domain.
//                     Use at own risk.
//                     No guarantees given.
//
// Latest change:      $Date$
//
// History:	       $Log$
//=============================================================================
package de.caff.asteroid;

import java.awt.*;
import java.awt.geom.AffineTransform;
import java.awt.geom.GeneralPath;
import java.util.Collection;

/**
 * Space ship in automated player of Asteroid game.
 *
 * This is part of a solution for a competition
 * <a href="http://www.heise.de/ct/creativ/08/02/details/">by the German computer magazine c't</a>
 */
public class SpaceShip
        extends MovingGameObject
{
  /** Ship color. */
  private static final Color COLOR = new Color(0xc0, 0xc0, 0xc0); // not as white as bullets

  /** The ship shape for drawing. */
  private static final GeneralPath SHIP_SHAPE = new GeneralPath();
  static {
    // defined for 0 angle
    SHIP_SHAPE.moveTo(16, 0);
    SHIP_SHAPE.lineTo(-16, 8);
    SHIP_SHAPE.lineTo(-8, 0);
    SHIP_SHAPE.lineTo(-16, -8);
    SHIP_SHAPE.closePath();
  }

  /** The direction of the ship, x coordinate. */
  private final int dx;
  /** The direction of the ship, y coordinate. */
  private final int dy;

  /**
   *  Constructor.
   *  @param x  x coordinate of ship center
   *  @param y  y coordinate of ship center
   *  @param dx direction x
   *  @param dy direction y
   */
  public SpaceShip(int x, int y, int dx, int dy)
  {
    super(x, y);
    this.dx = dx;
    this.dy = dy;
  }

  /**
   *  Get the size of the object.
   *
   *  The size returned by this method is half the length of a square which contains the object,
   *  so the object's bounding box is between (x - size, y - size) and (x + size, y + size).
   *  @return object size
   */
  public int getSize()
  {
    return 16;
  }

  /**
   *  Get the x coordinate of the ship's direction.
   *  @return ship direction, x coordinate
   */
  public int getDirX()
  {
    return dx;
  }

  /**
   *  Get the y coordinate of the ship's direction.
   *  @return ship direction, y coordinate
   */
  public int getDirY()
  {
    return dy;
  }

  /**
   *  Get the direction angle.
   *
   *  The angle is measured counterclockwise, with <code>0</code> pointing to the right
   *  @return ship direction in radians
   */
  public double getDirectionAngle()
  {
    return Math.atan2(dy, dx);
  }

  /**
   *  Get the direction.
   *  @return direction
   */
  public Point getDirection()
  {
    return new Point(dx, dy);
  }

  /**
   * Draw the object.
   *
   * @param g graphics context
   */
  @Override
  public void draw(Graphics2D g)
  {
    g.setColor(COLOR);
    AffineTransform at = AffineTransform.getTranslateInstance(x, y);
    at.concatenate(AffineTransform.getRotateInstance(getDirectionAngle()));
    g.fill(at.createTransformedShape(SHIP_SHAPE));

    drawVelocityVector(g, Color.green);
  }

  /**
   * Returns a string representation of the object.
   *
   * @return a string representation of the object.
   */
  @Override
  public String toString()
  {
    return String.format("SpaceShip@(%d,%d)>(%-3.1f)", x, y, 180*getDirectionAngle()/Math.PI);
  }

  private static int minVX = 0;
  private static int maxVX = 0;
  private static int minVY = 0;
  private static int maxVY = 0;
  private static int maxVSquared   = 0;

  /**
   * Set the velocity.
   * <p/>
   * The velocity is the step between frames.
   *
   * @param x x coordinate of velocity
   * @param y y coordinate of velocity
   */
  @Override
  public void setVelocity(int x, int y)
  {
    super.setVelocity(x, y);
    if (WATCH_VELOCITIES) {
      boolean print = false;
      if (x < minVX) {
        minVX = x;
        print = true;
      }
      else if (x > maxVX) {
        maxVX = x;
        print = true;
      }
      if (y < minVY) {
        minVY = y;
        print = true;
      }
      else if (y > maxVY) {
        maxVY = y;
        print = true;
      }
      int len2 = x*x + y*y;
      if (len2 > maxVSquared) {
        maxVSquared = len2;
        print = true;
      }
      if (print) {
        System.out.println(String.format("SHP: minV=(%d,%d), maxV=(%d,%d), maxV^2=%d",
                                         minVX, minVY, maxVX, maxVY, maxVSquared));
      }
    }
  }

  /**
   * Get the properties of this object.
   *
   * @return collection of properties
   */
  @Override
  public Collection<Property> getProperties()
  {
    Collection<Property> props = super.getProperties();
    props.add(new Property<Point>("Direction", getDirection()));
    props.add(new Property<Double>("Direction Angle ()", 180*getDirectionAngle()/Math.PI));
    return props;
  }

  /**
   * Get the type of game object.
   *
   * @return game object type
   */
  public String getObjectType()
  {
    return "Space Ship";
  }
}
