package gui;

import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.text.SimpleDateFormat;
import java.util.Date;

import javax.swing.BoxLayout;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.border.EtchedBorder;
import javax.swing.border.TitledBorder;

public class StatusPanel extends JPanel {
	 
	private static final long serialVersionUID = 1L;
	
	public static final int STATE_ERROR = 0;
	
	public static final int STATE_WAITING = 1;
	
	public static final int STATE_CONNECTED = 2;
	
	public static final int STATE_RUNNING = 3;

	private SimpleDateFormat timeFormat = new SimpleDateFormat("mm:ss");
	
	private Date time = new Date();
	
	private JLabel[] timeLabels;
	
	private JLabel[] frameLabels;
	
	private JLabel[] lossLabels;
	
	private JLabel[] stateLabels;
	
	private JLabel[] serverLabels;
	
	private JLabel[] portLabels;


	public StatusPanel()
	{
		super();
		setBorder(new TitledBorder(new EtchedBorder() , "Status"));
		int lWidth = 45;
		int vWidth = 125;
		setLayout(new BoxLayout(this, BoxLayout.PAGE_AXIS));
		timeLabels = createLabelPair("Time:", lWidth, "00:00", vWidth);
		frameLabels = createLabelPair("Frame:", lWidth, "00001", vWidth);
		lossLabels = createLabelPair("Loss:", lWidth, "000", vWidth);
		stateLabels = createLabelPair("State:", lWidth,  "Waiting", vWidth);
		stateLabels[1].setForeground(Color.ORANGE.darker());
		serverLabels = createLabelPair("Server:", lWidth, "localhost", vWidth);
		portLabels = createLabelPair("Port:", lWidth, "1979", vWidth);
	}
	
	public void updateTime(long refTime)
	{
		time.setTime((refTime * 100) / 6);
		timeLabels[1].setText(timeFormat.format(time));
	}
	
	public void updateFrame(int frame)
	{
		String frameFormat = "%05d";
		frameLabels[1].setText(String.format(frameFormat, frame));
	}
	
	public void updateLoss(int loss)
	{
		String frameFormat = "%03d";
		lossLabels[1].setText(String.format(frameFormat, loss));
	}
	
	public void updateConnection(String server, int port)
	{
		serverLabels[1].setText(server);
		portLabels[1].setText(Integer.toString(port));
	}
	
	public void updateState(int state)
	{
		switch(state)
		{
		case STATE_ERROR:
			stateLabels[1].setForeground(Color.RED);
			stateLabels[1].setText("Disconnected");
			break;
		case STATE_WAITING:
			stateLabels[1].setForeground(Color.ORANGE.darker());
			stateLabels[1].setText("Waiting");
			break;
		case STATE_CONNECTED:
			stateLabels[1].setForeground(Color.GREEN.darker());
			stateLabels[1].setText("Connected");
			break;
		case STATE_RUNNING:
			stateLabels[1].setForeground(Color.BLUE);
			stateLabels[1].setText("Running");
			break;
		default:
			stateLabels[1].setForeground(Color.BLACK);
			stateLabels[1].setText("Undefined");
			break;
		}		
	}
	
	public JLabel[] createLabelPair(String label, int lWidth, String value, int vWidth)
	{
		JPanel result = new JPanel();		
		JLabel[] labels = new JLabel[2];
		labels[0] = new JLabel(label);
		Dimension prefSize = labels[0].getPreferredSize();
		labels[0].setPreferredSize(new Dimension(lWidth, prefSize.height));
		labels[0].setAlignmentX(Component.LEFT_ALIGNMENT);
		labels[1] = new JLabel(value);	
		prefSize = labels[1].getPreferredSize();
		labels[1].setPreferredSize(new Dimension(vWidth, prefSize.height));
		result.setLayout(new FlowLayout(FlowLayout.LEADING, 5, 0));
		result.setAlignmentX(Component.LEFT_ALIGNMENT);
		result.add(labels[0]);
		result.add(labels[1]);
		add(result);
		return labels;
	}
}
