package helpers;

import io.CombatState;
import objects.Asteroid;
import objects.CombatObject;
import objects.Saucer;
import objects.Ship;
import objects.Shootable;
import objects.TargetObject;
import structures.TargetConfiguration;
import structures.TargetInformation;
import utils.Tables;
import utils.Utils;


public class CombatHeuristics {
	
	public static double getSaucerPrioritization(CombatState state, TargetConfiguration config, Saucer saucer, int asteroidNum)
	{
		if(isSaucerCampingReasonable(state, config))
		{
			return 0.0;
		} else {
			double prio = getCollidingPrioritization(config);
			if(saucer.getScale() == CombatObject.SMALL_SCALE)
			{
				if(asteroidNum > 3)
				{
					prio *= 0.25;
				} else {
					prio *= 0.75;
				}
			} else {
				if(asteroidNum > 3)
				{
					prio *= 0.75;
				} else {
					prio *= 1.25;
				}
			}
			return prio;
		}
	}
	
	public static double getCollidingPrioritization(TargetConfiguration config)
	{
		if(config.collisionTime > 4 && config.collisionTime < Utils.EVASION_MIN_TIME)
		{
			return 0.5 + config.collisionTime / (2 * Utils.EVASION_MIN_TIME); 
		} else {
			return 1.0;
		}
	}
	
	public static boolean isMaxShotCountReached(TargetObject object)
	{
		int shotsFired = object.getTargetedStatus();
		switch(object.getScale())
		{
			case 0: 
				if(shotsFired > 3) return true;
				break;
			case 14: 
				if(shotsFired > 0) return true;
				break;
			case 15: 
				if(object instanceof Saucer) {
					if(shotsFired > 0) return true;
				} else {
					if(shotsFired > 2) return true;
				}
				break;
		}
		return false;
	}
	
	public static boolean isTargetSwitchReasonable(TargetObject oldTarget, TargetConfiguration newTarget)
	{
		if(newTarget.object != oldTarget)
		{
			if(newTarget.collisionTime < Utils.EVASION_MIN_TIME)
			{
				if(oldTarget.getTargetedStatus() == 0)
				{
					return true;
				} else {
					Ship ship = newTarget.ship;
					Shootable fastestTorpedo = ship.getFastestTorpedo();
					if(ship.getAvailableTorpedos() == 1 &&
						fastestTorpedo != null &&
						fastestTorpedo.getTimeToDestination() > newTarget.collisionTime - 3)
					{
						return true;
					}
				}
			} else if(newTarget.object.getMovementPrecision() < Utils.MAX_MOVE_PRECISION)
			{
				return (oldTarget.getTargetedStatus() == 0);
			}
		}
		return false;
	}
	
	public static boolean isSaucerCampingReasonable(CombatState state, TargetConfiguration config)
	{
		
		if(CombatStatistics.getSectorSaucerCount() > 0 && state.isLastTargetLeft())
		{
			int timeToNextSaucer = Tables.SAUCER_SPAWN_DELAY[Math.min(19, CombatStatistics.getGlobalSaucerCount())];
			int timePassed = CombatStatistics.getPassedSaucerWaitTime();
			timePassed += config.getFinalTimeTaken() + Utils.EXPLOSION_DELAY - 2;
			if(timeToNextSaucer - timePassed > 0 && timeToNextSaucer - timePassed < Utils.MAX_SAUCER_CAMP_TIME)
			{
				return true;
			}
		}
		return false;
	}
	
	public static boolean isTargetDifficult(TargetConfiguration config)
	{
		// Low effort check for "difficult" targets
		if(config.object.isMovementPrecise())
		{
			// Check 1st criteria: distance
			int minDist;
			switch(config.object.getScale())
			{
			case CombatObject.LARGE_SCALE:
				minDist = 560;
				break;
			case CombatObject.MEDIUM_SCALE:
				minDist = 272;
				break;
			default:
				minDist = 200;
				break;
			}			
			if(config.preCalc.distance > minDist)
			{
				// Check 2nd criteria: direction
				double directionDif = Math.abs(config.object.getMovementAngle() - Utils.getCounterAngle(config.info.angle));
				while(directionDif > Math.PI)
				{
					directionDif -= Math.PI;
				}	
				if(directionDif > Math.PI / 2.0)
				{
					directionDif = Math.PI - directionDif;
				}				
				if(directionDif < Utils.AVERAGE_TURN_ANGLE / 2)
				{					
					// Check 3rd criteria: position
					double angleDif = Math.abs(Utils.getAngleDif(Utils.TABLES.getAngle(config.angleIndex), config.preCalc.angle));
					if(angleDif < Math.PI / 4.0){			
						double hitZone = CombatHeuristics.getHitZone(config.object);
						double deviation = Utils.TABLES.sin(angleDif) * config.preCalc.distance;
						if(deviation >= hitZone)
						{
							return true;
						}
					}
				}
			}
		}
		return false;
	}
	
	public static boolean isTargetHittable(TargetConfiguration config, TargetInformation info)
	{
		if(Math.abs(info.angleDif) < Math.PI / 4.0){			
			double hitZone = CombatHeuristics.getHitZone(config.object);
			double deviation = Utils.TABLES.sin(Math.abs(info.angleDif)) * info.distance;
			return deviation < hitZone || config.object.getTargetedStatus() > 0;
		}
		return false;
	}

	public static double getHitZone(TargetObject object)
	{
		double hitZone = object.getRadius();
		if(hitZone > 8.0)
		{
			hitZone *= 0.666;
		} else {
			hitZone *= 1.0;
		}
		return hitZone;
	}
	
	public static double getImprovedHitZone(TargetObject object)
	{
		double hitZone = object.getRadius();
		if(hitZone > 8.0)
		{
			hitZone *= 0.333;
		} else {
			hitZone -= 1.0;
		}
		return hitZone;
	}
	
	public static int getPrecisionDelay(double distance, TargetObject target)
	{	
		int precision = 0;
		if(target != null)
		{
			precision = target.getMovementPrecision();
		}
		if(distance < 80) {
			return 0;
		} else if(distance < 160) {
			return Math.max(0, 2 - precision);
		} else if(distance < 240) {
			return Math.max(0, 4 - precision);
		} else {
			return Math.max(0, 8 - precision);
		}		
	}
	
	public static boolean willLoseChildren(TargetObject object, int asteroidNumber)
	{
		if(asteroidNumber > 24)
		{
			switch(object.getScale())
			{
				case 0: 
					return true;
				case 14: 
					return false;
				case 15: 
					return true;
			}
		}
		return false;
	}
	
	public static boolean willSpawnChildren(TargetObject target, CombatState state)
	{
		if(target != null && target instanceof Asteroid) {
			int targetedStatus = target.getTargetedStatus();
			switch(target.getScale())
			{
			case Asteroid.LARGE_SCALE:
				return targetedStatus > 0 && state.getAsteroidNumber() < 5 && state.areTargetsLeft() == false; 
			case Asteroid.MEDIUM_SCALE:
				return targetedStatus > 0 && state.getAsteroidNumber() < 5 && state.areTargetsLeft() == false; 
			default: 
				return targetedStatus > 0 && state.getAsteroidNumber() < 5 && state.areTargetsLeft() == false;
			}
		}
		return false;
	}
}
