package objects;


import java.awt.Rectangle;

import utils.Utils;

public abstract class CombatObject 
{
	public static final int NO_SCALE = -1;
	
	public static final int LARGE_SCALE = 0;
	
	public static final int MEDIUM_SCALE = 15;
	
	public static final int SMALL_SCALE = 14;
	
	protected static int RUNNING_ID = 0; 
	
	protected int posX;
	
	protected int posY;
	
	protected double moveX;
	
	protected double moveY;
	
	protected int id;
	
	protected int width;
	
	protected int height;
	
	protected int scale;
	
	protected int movePrecision;
	
	protected boolean isMovementPrecise;
	
	protected boolean isAlive;
	
	protected int pos0X;
	
	protected int pos0Y;
	
	public CombatObject(int x, int y) {
		pos0X = x;
		pos0Y = y;
		posX = x;
		posY = y;		
		moveX = 0.0;
		moveY = 0.0;
		id = RUNNING_ID++;
		movePrecision = 0;
		isMovementPrecise = false;
		width = 0;
		height = 0;
		scale = 0;	
		isAlive = true;
	}
	
	public int getScale()
	{
		return scale;
	}
	
	public String toString() {
		String result = "Combat Object #%d {%d, %d / %.3f, %.3f}";
		return String.format(result, id, posX, posY, moveX, moveY);
	}
	
	public boolean isMovementPrecise()
	{
		return isMovementPrecise;
	}
	
	public int getMovementPrecision()
	{
		if(isMovementPrecise)
		{
			return Utils.MAX_MOVE_PRECISION;
		} else {
			return movePrecision;
		}
	}
	
	public boolean matches(int otherX, int otherY, int frameDif) {
		int distX;
		int distY;
		int maxDist;
		if(isMovementPrecise || movePrecision > 0) {
			distX = (int)Utils.getXDist(otherX, this.pos0X + moveX * (movePrecision + frameDif));
			distY = (int)Utils.getYDist(otherY, this.pos0Y + moveY * (movePrecision + frameDif));
			if(isMovementPrecise)
			{
				maxDist = 0;
			} else {
				maxDist = (int)Math.ceil(getMaximumSpeed() / (double)(getMovementPrecision() + 1));
			}
		} else {
			distX = (int)Utils.getXDist(otherX, this.posX);
			distY =  (int)Utils.getYDist(otherY, this.posY);
			maxDist = getMaximumSpeed() * frameDif;			
		}
		if(distX <= maxDist && distY <= maxDist)
		{
			return true;
		}
		return false; 
	}
	
	public void update(int newX, int newY, int frameDif) {
		movePrecision += frameDif;
		if(movePrecision % Utils.MAX_MOVE_PRECISION == 0)
		{
			movePrecision = 0;
		}
		if(isMovementPrecise == false)
		{
			if(movePrecision == 0)
			{
				moveX = Utils.getXDif(pos0X, newX) / (double)Utils.MAX_MOVE_PRECISION;			
				moveY = Utils.getYDif(pos0Y, newY) / (double)Utils.MAX_MOVE_PRECISION;
				isMovementPrecise = true;
			} else {
				moveX = Utils.getXDif(pos0X, newX) / (double)movePrecision;			
				moveY = Utils.getYDif(pos0Y, newY) / (double)movePrecision;
			}
			//moveX = (movePrecision * moveX + Utils.getXDif(posX, newX) / (double)frameDif) / (double)(movePrecision + 1);
			//moveY = (movePrecision * moveY + Utils.getYDif(posY, newY) / (double)frameDif) / (double)(movePrecision + 1);
		}
		if(movePrecision == 0)
		{
			pos0X = newX;
			pos0Y = newY;
		}
		posX = newX;
		posY = newY;
	}
	
	public int getRadius() {
		return (width + height) / 4;
	}
	
	public double getSpeed() {
		return Utils.TABLES.getSpeed(moveX, moveY);
	}
	
	public double getMovementAngle() {
		return Utils.TABLES.getAngle(moveX, moveY);
	}
	
	public double getMovementX() {
		return moveX;
	}
	
	public double getMovementY() {
		return moveY;
	}
	
	public int getPosX() {
		return posX;
	}
	
	public int getPosY() {
		return posY;
	}

	public int getID() {
		return id;
	}
	
	public int getWidth()
	{
		return width;
	}
	
	public int getHeight()
	{
		return height;
	}
	
	public Rectangle getBounds() {
		return new Rectangle(posX - width / 2, posY - height / 2, width, height);
	}
	
	public boolean isAlive()
	{
		return isAlive;
	}
	
	public void spawn(int x, int y)
	{
		pos0X = x;
		pos0Y = y;
		posX = x;
		posY = y;		
		moveX = 0.0;
		moveY = 0.0;
		movePrecision = 0;
		isMovementPrecise = false;
		isAlive = true;
	}
	
	public void destroy()
	{
		isAlive = false;
	}
	
	public abstract int getMaximumSpeed();
}
