///////////////////////////////////////////////////////////////////////////////
// @ Eduard Heidt                                                            //
///////////////////////////////////////////////////////////////////////////////


#ifndef MATH2DH
#define MATH2DH

#include <stdio.h>
#include <math.h>
#include <float.h>
#include <limits.h>
#include <algorithm>

#ifndef PI
#define PI (VDC)3.14159265358979323846
#define DEG2RAD(x) ((VDC)((x)/180.0*PI))
#define RAD2DEG(x) ((VDC)((x)*180.0/PI))
#endif

typedef unsigned char byte;

typedef float VDC;
static const VDC VDC_MAX = FLT_MAX;
static const VDC VDC_MIN = FLT_MIN;


class Vector2D
{
public: 
	VDC x, y;

	Vector2D():x(0),y(0){}
	Vector2D(VDC x, VDC y):x(x),y(y){}
	Vector2D(int x, int y):x((VDC)x),y((VDC)y){}
	Vector2D(const Vector2D &b):x(b.x),y(b.y){/*ASSERT(!IsNaN());*/}

	inline bool IsZero(){return fabs(x*x+y*y)<0.001f;}

	inline Vector2D Normalized() const
	{
		VDC len = Len();
		if(len > 0.0001)
			return (*this) * (1.f/len);
		else
			return Vector2D();
	}

	inline Vector2D Translated(VDC _x, VDC _y) const
	{
		return Vector2D(x+_x, y+_y);
	}
	inline Vector2D Translated(int _x, int _y) const
	{
		return Translated((VDC)_x,(VDC)_y);
	}

	inline bool operator==(const Vector2D &v2) const
	{
		return this->x == v2.x && this->y == v2.y;
	}

	inline bool operator!=(const Vector2D &v2) const
	{
		return this->x != v2.x || this->y != v2.y;
	}

	inline VDC Len() const { return sqrt(SqLen()); } 
	inline VDC SqLen() const { return ((x*x)+(y*y)); }
	inline VDC operator*(const Vector2D &v2) const { return ((x*v2.x)+(y*v2.y)); }
	inline VDC operator/(const Vector2D &v2) const { return ((x/v2.x)+(y/v2.y)); }

	inline Vector2D operator*(const VDC &f) const { return Vector2D((x*f),(y*f)); }
	inline bool IsNaN(){return (_isnan(x) && _isnan(y));}
	inline Vector2D operator+(const Vector2D &v) const { return Vector2D(x+v.x, y+v.y); }
	inline Vector2D operator-(const Vector2D &v) const { return Vector2D(x-v.x, y-v.y); }
	inline VDC Cross(const Vector2D &v2) const { return y*v2.x-x*v2.y; }
	
	inline VDC DegreeBetween360(const Vector2D &v2) const 
	{
		VDC deg = DegreeBetween180(v2);
		
		if(Cross(v2) < 0)
			deg = 360-deg;

		return deg;
	}
	inline VDC DegreeBetween180(const Vector2D &v2) const 
	{
		VDC c = ((*this)*v2)/(this->Len()*v2.Len());
		
		if(c<-1) c = -1;
		if(c> 1) c = 1;

		return RAD2DEG(acos(c));
	}
	inline bool IsPrallelTo(const Vector2D &v2){return Cross(v2) == 0;}
	inline VDC DistanceTo(const Vector2D &v2) const {return (v2-*this).Len();}
	inline VDC SqDistanceTo(const Vector2D &v2) const {return (v2-*this).SqLen();}

	inline Vector2D GetMod(Vector2D min, Vector2D max)
	{
		Vector2D v = *this;

		while(v.x > max.x) 
			v.x -= (max.x-min.x);
		while(v.y > max.y) 
			v.y -= (max.y-min.y);

		while(v.x < min.x) 
			v.x += (max.x-min.x);
		while(v.y < min.y) 
			v.y += (max.y-min.y);

		return v;
	}

	void print()
	{
		printf("[%.2f,%.2f]", x, y);
	}
};


class Ray2D 
{
public:
	Vector2D p, d;
	Ray2D(){}
	Ray2D(const Vector2D &_p, const Vector2D &_d):p(_p),d(_d){}
	
	Vector2D GetPosAt(VDC t) const
	{
		return p + d*t;
	}
	
	bool GetIntersection(const Ray2D& r2, VDC& t, VDC& t2) const
	{
		VDC vec[2];
		VDC mat[2][2];
		VDC r[2];

		VDC cross = d.Cross(r2.d);

		if(cross == 0.f) //  sind prallel
		{
			if((p-r2.p).IsPrallelTo(d)) //liegen genau aufeinander
			{
				if(d.DegreeBetween180(r2.d) > 170 && d.DegreeBetween180(r2.d) < 190) //laufen gegeneinander
					t = t2 = p.DistanceTo(r2.p)/(d.Len()+r2.d.Len());
				else
					t = t2 = p.DistanceTo(r2.p)/(d.Len()-r2.d.Len());
			}
			else
				return false; //verlaufen parallel..
		}
		else
		{
			mat[0][0] = r2.d.y/cross;	mat[0][1] = -r2.d.x/cross;
			mat[1][0] = d.y/cross; 		mat[1][1] = -d.x/cross;
			
			vec[0] = p.x-r2.p.x;
			vec[1] = p.y-r2.p.y;
			
			for ( int i = 0; i < 2; ++i ) 
			{
				r[i] = 0;
				for ( int j = 0; j < 2; ++j )
					r[i] += mat[i][j] * vec[j];
			}

			t   = r[0];
			t2  = r[1];
		}
		
		return true;
	}

	
	bool GetIntersectionWithCircle(const Vector2D& CircleCenter, const VDC Radius, VDC& t1, VDC &t2) const
	{
		const Vector2D& LineStart = p;
		const Vector2D& LineEnd = p + d;
		
		const VDC RadiusSq = Radius * Radius;	// r vorberechnen
		const Vector2D PMinusM(LineStart - CircleCenter); // (p - m) vorberechnen
		
	
		const Vector2D LineDir(LineEnd - LineStart);  // Richtungsvektor der Linie berechnen (u)	
		const VDC UDotPMinusM = LineDir*PMinusM; // u * (p - m) vorberechnen
		const VDC LineDirSq = LineDir.SqLen(); // u vorberechnen

		// Die Diskriminante berechnen:
		//   (u * (p - m))
		// - (u * ((p - m) - r))
		const VDC d =   (UDotPMinusM * UDotPMinusM) - (LineDirSq * (PMinusM.SqLen() - RadiusSq));

		// Ist die Diskriminante negativ, null oder positiv?
		if(d < 0.0f) 
			return false;

		else if(d < 0.0001f) // nur 1 Schnittpunkt
		{
			t1 = t2 = -UDotPMinusM / LineDirSq;
		}
		else	// Zwei Schnittpunkte
		{
			t1 = (-UDotPMinusM - sqrtf(d)) / LineDirSq;
			t2 = (-UDotPMinusM + sqrtf(d)) / LineDirSq;
		}

		if(PMinusM.SqLen() <= RadiusSq)
		{
			// Als Schnittpunkt nehmen wir einfach
			// den Startpunkt der Linie.
			t1 = 0;
		}

		return true;
	}
};


class Circle2D
{
public:
	Vector2D c;
	VDC		rad;

	Circle2D(const Vector2D& _c, VDC _rad):c(_c), rad(_rad){}
	bool GetIntersectionWithRay(const Ray2D& r, VDC& t1, VDC& t2)
	{
		return r.GetIntersectionWithCircle(c, rad, t1, t2);
	}
};

typedef Ray2D 		Ray;
typedef Vector2D 	Vector;
typedef Circle2D 	Circle;

#endif

