/*
 *
 * Asteroids Player
 *
 * for the c't Competition 2008 (Creativ '08) 
 *
 * Copyright 2008, Volker Raum, Erlangen, Germany
 *
 */
package de.volkerraum.asteroids.model;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Observable;

import de.volkerraum.asteroids.connection.Connection;

/**
 * @author Volker Raum (C) 2007
 */
public class AsteroidsModel extends Observable
{
   public final static int BIG_ASTEROID_SIZE = 64;
   public final static int MIDDLE_ASTEROID_SIZE = 32;
   public final static int SMALL_ASTEROID_SIZE = 16;

   public final static int SHIP_SIZE = 28;
   public final static int SHIP_RADIUS = 14;

   final static int MAX_ASTEROIDS = 100;
   final static int MAX_SHOTS = 10;

   public static final int SHOT_REACH = 570;

   public final static double COLLISION_SAFETY = 0;

   public final static double SHOT_SPEED = 8;

   public static final Integer MSG_GAME_STARTED = 1;
   public static final Integer MSG_GAME_FINISHED = 2;
   public static final Integer MSG_NEW_ROUND = 3;
   public static final Integer MSG_NEW_FRAME = 4;

   Asteroid[] asteroids = new Asteroid[MAX_ASTEROIDS];
   Asteroid[] oldAsteroids = new Asteroid[MAX_ASTEROIDS];

   Shot[] shots = new Shot[MAX_SHOTS];
   Shot[] oldShots = new Shot[MAX_SHOTS];

   HashMap<Integer, Integer> numbers = new HashMap<Integer, Integer>();
   static final int[] routineNumbers = { 0xadd, 0xb2e, 0xb32, 0xb3a, 0xb41, 0xb48, 0xb4f, 0xb56, 0xb5b, 0xb63 };

   {
      for (int i = 0; i < routineNumbers.length; ++i)
      {
         numbers.put(routineNumbers[i], i);
      }
   }

   /**
    * @uml.property name="allObjectsInScene"
    */
   ArrayList<BaseObject> allObjectsInScene = new ArrayList<BaseObject>(MAX_ASTEROIDS + 10);

   /**
    * @uml.property name="asteroidCount"
    */
   int asteroidCount = 0;
   int oldAsteroidCount = 0;

   /**
    * @uml.property name="shotCount"
    */
   int shotCount = 0;
   int oldShotCount = 0;

   boolean inGame = false;
   Ship theShip = null;
   Ship theOldShip = null;
   Saucer theSaucer = null;
   Saucer theOldSaucer = null;

   public static double originalSizeY = 768;
   public static double originalSizeX = 1024;

   Connection connection = null;

   List<Shot> identifiedShots = new ArrayList<Shot>(MAX_SHOTS);

   // the 5 digits from asteroids
   private int internalScore = 0;
   private int internalScoreOverflows = 0;

   private int roundCounter = 0;
   public long processedFramesCount = 0;

   // how many Times is the count of the asteroids identical ?
   public int constantAsteroidsCount = 0;

   public boolean startMessageWasDisplayed = false;

   byte leftCommandFrame = -1;
   byte rightCommandFrame = -1;
   boolean leftCommand = false;
   boolean rightCommand = false;

   HashMap<Double, Integer> leftMovementAngleToAnglebyte = new HashMap<Double, Integer>();
   HashMap<Double, Integer> rightMovementAngleToAnglebyte = new HashMap<Double, Integer>();

   double TwoPI = Math.PI * 2;
   double internalAngleStep = TwoPI * 3 / 256;

   byte lastFrameNumber = 0;
   boolean shipsScreenAngleChanged = false;

   int angleByte = 0;

   public int saucerMovementCount = 0;

   List<Asteroid> trackedAsteroids = new ArrayList<Asteroid>(MAX_ASTEROIDS);

   long startFrameForRound = 0;
   long endFrameForRound = 0;
   int lossMinus1Count = 0;

   double speedSmall = 0;
   double speedMid = 0;
   double speedBig = 0;

   boolean thrustPossible = false;
   boolean slowdown = false;

   long maxProcessTime = 0;
   long minProcessTime = Long.MAX_VALUE;

   public static final long SCORE_REPORT_INTERVALL = 3600; // 5 Mins
   long lastIntervallScore = 0;
   long nextReport = SCORE_REPORT_INTERVALL;

   public AsteroidsModel()
   {
      for (int i = 0; i < MAX_ASTEROIDS; ++i)
      {
         asteroids[i] = new Asteroid();
         oldAsteroids[i] = new Asteroid();
      }
      for (int i = 0; i < MAX_SHOTS; ++i)
      {
         shots[i] = new Shot();
         oldShots[i] = new Shot();
      }

      theShip = new Ship();
      theOldShip = new Ship();
      theSaucer = new Saucer();
      theOldSaucer = new Saucer();

      init();
   }

   public void sendCommand(Command command)
   {
      if (connection != null)
         connection.sendCommand(command.command);
   }

   public void sendCommand(byte command)
   {
      connection.sendCommand(command);
   }

   public void reset()
   {
      for (int i = 0; i < MAX_ASTEROIDS; ++i)
      {
         asteroids[i] = new Asteroid();
         oldAsteroids[i] = new Asteroid();
      }
      for (int i = 0; i < MAX_SHOTS; ++i)
      {
         shots[i] = new Shot();
         oldShots[i] = new Shot();
      }

      theShip = new Ship();
      theOldShip = new Ship();
      theSaucer = new Saucer();
      theOldSaucer = new Saucer();

      asteroidCount = 0;
      oldAsteroidCount = 0;

      shotCount = 0;
      oldShotCount = 0;

      inGame = false;

      // the 5 digits from asteroids
      internalScore = 0;
      internalScoreOverflows = 0;

      roundCounter = 0;
      processedFramesCount = 0;

      // how many Times is the count of the asteroids identical ?
      constantAsteroidsCount = 0;
      lastIntervallScore = 0;
      startMessageWasDisplayed = false;

      leftCommandFrame = -1;
      rightCommandFrame = -1;
      leftCommand = false;
      rightCommand = false;
      lastFrameNumber = 0;
      shipsScreenAngleChanged = false;
      angleByte = 0;
      saucerMovementCount = 0;
      startFrameForRound = 0;
      lossMinus1Count = 0;
      thrustPossible = false;
      slowdown = false;
      maxProcessTime = 0;
      minProcessTime = Long.MAX_VALUE;

   }

   public void analyseIncomingBytes(byte[] incoming)
   {
      long startProcess = System.currentTimeMillis();

      processedFramesCount++;

      if (processedFramesCount == 18000)
      {
         System.err.println(this.getScore() + " AFTER 5 MINUTES");
      }
      if (processedFramesCount >= nextReport)
      {
         System.err.println("Score : " + (this.getScore() - lastIntervallScore) + "(delta) - " + this.getScore() + " abs");
         lastIntervallScore = this.getScore();
         nextReport += SCORE_REPORT_INTERVALL;
      }
      byte newFramecount = incoming[1024];
      if (newFramecount != ++lastFrameNumber)
      {
         System.err.println("FrameLoss " + (newFramecount - lastFrameNumber));

         if ((newFramecount - lastFrameNumber) == -1 && processedFramesCount > 17000)
         {
            this.setChanged();
            this.notifyObservers(MSG_GAME_FINISHED);
         }

      }
      lastFrameNumber = newFramecount;

      if ((incoming[1] & 0xff) == 0xE2)
      {
         thrustPossible = false;
      }
      else
      {
         thrustPossible = true;
      }

      if (!theShip.present)
      {
         theShip.internalDX = theShip.internalDX - ((int) theShip.internalDX);
         theShip.internalDY = theShip.internalDY - ((int) theShip.internalDY);
      }
      else
      {
         if ((incoming[1025] & Command.ACTION_THRUST) != 0)
         {
            if (!thrustPossible)
            {
               // we had a thrust last Frame
               theShip.internalDX += ((double) Math.round(Math.cos(theShip.internalAngle) * 127)) / 128;
               theShip.internalDY += ((double) Math.round(Math.sin(theShip.internalAngle) * 127)) / 128;
            }
         }
         else if (!thrustPossible)
         {
            if (theShip.internalDX > 0)
            {
               theShip.internalDX = theShip.internalDX - ((double) Math.floor(theShip.internalDX)) / 128.0 - 1.0 / 256.0;
               if (theShip.internalDX < 0)
                  theShip.internalDX = 0;
            }
            else
            {
               theShip.internalDX = theShip.internalDX + ((double) Math.floor(theShip.internalDX)) / 128.0;
               if (theShip.internalDX > 0)
                  theShip.internalDX = 0;
            }
            if (theShip.internalDY > 0)
            {
               theShip.internalDY = theShip.internalDX - ((double) Math.floor(theShip.internalDY)) / 128.0 - 1.0 / 256.0;
               if (theShip.internalDY < 0)
                  theShip.internalDY = 0;
            }
            else
            {
               theShip.internalDY = theShip.internalDY + ((double) Math.floor(theShip.internalDY)) / 128.0;
               if (theShip.internalDY > 0)
                  theShip.internalDY = 0;
            }
         }
      }

      theShip.internalX += Math.floor(theShip.internalDX);
      theShip.internalY += Math.floor(theShip.internalDY);

      int[] vector_ram = toIntArray(incoming);
      analyse(vector_ram);

      // Try to pick ships internal Angle.

      boolean syncTookPlace = checkInternalShipAngle(incoming[1025]);
      theShip.internalAngle = Helper.winkelByteToAngle.get(angleByte);
      theShip.angleByte = angleByte;

      // next Round ?
      if (oldAsteroidCount == 0 && asteroidCount > 0)
      {
         roundCounter++;
         // System.err.println("Round " + roundCounter + " : min/max " + minProcessTime + "/" + maxProcessTime + " ms - Frames = " + (endFrameForRound - startFrameForRound));
         System.err.println("Round " + roundCounter + " Frames = " + (endFrameForRound - startFrameForRound));

         startFrameForRound = processedFramesCount;
         maxProcessTime = 0;
         minProcessTime = Long.MAX_VALUE;

         this.setChanged();
         this.notifyObservers(MSG_NEW_ROUND);

      }

      if (oldAsteroidCount > 0 && asteroidCount == 0)
      {
         endFrameForRound = processedFramesCount;
         // System.err.println("Frames for this round " + endFrameForRound);
      }

      // Objects with an analysed Speed
      allObjectsInScene.clear();

      // Check The Asteroids Movements
      checkAsteroidMovements(true);

      // Now check the shots movements
      checkShotMovements();

      // Normalize Saucer
      checkSaucerMovement();

      if (theSaucer.present)
         allObjectsInScene.add(theSaucer);
      else
         theSaucer.shotsFiredAt = 0;

      double sdx = calcDiffWithTorus(theShip.rawX, theOldShip.rawX, 1024, 800);
      double sdy = calcDiffWithTorus(theShip.rawY, theOldShip.rawY, 768, 600);
      theShip.dx = sdx; // Mean of the current and the

      // Last Value
      theShip.dy = sdy;
      theShip.speed = Math.sqrt(sdx * sdx + sdy * sdy);

      theOldShip.copyData(theShip);

      Helper.calcHighLevelInformations(allObjectsInScene, identifiedShots, theShip, isLeftCommand(), isRightCommand());

      this.setChanged();
      this.notifyObservers(MSG_NEW_FRAME);

      long processTime = System.currentTimeMillis() - startProcess;
      if (processTime > this.maxProcessTime)
         maxProcessTime = processTime;
      if (processTime < this.minProcessTime)
         minProcessTime = processTime;

   }

   private void checkSaucerMovement()
   {
      if (theOldSaucer.present && theSaucer.present)
      {
         double dx = calcDiffWithTorus(theSaucer.rawX, theOldSaucer.rawX, 1024, 800);
         double dy = calcDiffWithTorus(theSaucer.rawY, theOldSaucer.rawY, 768, 600);

         theSaucer.dx = dx;
         theSaucer.dy = dy;

      }

      if (theSaucer.present)
      {
         Helper.normalizeObject(theShip, theSaucer);
      }

      theOldSaucer.copyData(theSaucer);
   }

   private void checkShotMovements()
   {
      identifiedShots.clear();
      boolean shotProblem = false;
      if (oldShotCount == shotCount && shotCount != 0)
      {
         for (int i = 0; i < shotCount; ++i)
         {
            double dx = calcDiffWithTorus(shots[i].rawX, oldShots[i].rawX, 1024, 800);
            double dy = calcDiffWithTorus(shots[i].rawY, oldShots[i].rawY, 768, 600);

            double speed = Helper.calcPythagoras(dx, dy);

            if (speed > 13)
            {
               // one shot exploded and another was born
               shotProblem = true;
               break;
            }

            int interpolationSteps = oldShots[i].interpolationSteps;

            if (interpolationSteps == 0)
            {
               shots[i].dx = dx;
               shots[i].dy = dy;
               shots[i].speed = speed;
               shots[i].interpolationSteps = 1;
            }
            else
            {

               shots[i].dx = (oldShots[i].dx * interpolationSteps + dx) / (interpolationSteps + 1);
               shots[i].dy = (oldShots[i].dy * interpolationSteps + dy) / (interpolationSteps + 1);

               shots[i].speed = Helper.calcPythagoras(shots[i].dx, shots[i].dy);
               shots[i].interpolationSteps++;
            }

            Shot newShot = new Shot();
            newShot.copyData(shots[i]);
            identifiedShots.add(newShot);
         }
      }
      else
      {
         for (int i = 0; i < shotCount; ++i)
         {
            shots[i].dx = 0;
            shots[i].dy = 0;
            shots[i].speed = 0;
            shots[i].interpolationSteps = 0;
         }
      }

      if (shotProblem)
      {

         for (int i = 0; i < shotCount; ++i)
         {
            shots[i].dx = 0;
            shots[i].dy = 0;
            shots[i].speed = 0;
            shots[i].interpolationSteps = 0;
         }
      }

      for (int i = 0; i < shotCount; ++i)
      {
         Helper.normalizeObject(theShip, shots[i]);
         oldShots[i].copyData(shots[i]);
      }
      oldShotCount = shotCount;
   }

   private double calcDiffWithTorus(double newValue, double oldValue, int torusSize, int maxDiff)
   {
      double diff = newValue - oldValue;

      // Eliminate Torus Effekt
      if (diff > maxDiff)
      {
         diff = (newValue - torusSize) - oldValue;
      }
      if (diff < -maxDiff)
      {
         diff = (newValue + torusSize) - oldValue;
      }
      return diff;
   }

   private double compensateTorus(double newValue, double oldValue, int torusSize, int maxDiff)
   {
      double diff = newValue - oldValue;

      // Eliminate Torus Effekt
      if (diff > maxDiff)
      {
         oldValue += torusSize;
      }
      if (diff < -maxDiff)
      {
         oldValue -= torusSize;
      }
      return oldValue;
   }

   private void calcInterpolatedMovement(Asteroid tracked, Asteroid newAsteroid)
   {

      double newDx = calcDiffWithTorus(newAsteroid.rawX, tracked.rawX, 1024, 800);
      double newDy = calcDiffWithTorus(newAsteroid.rawY, tracked.rawY, 768, 600);
      tracked.dx = (tracked.dx * tracked.interpolationSteps + newDx) / (tracked.interpolationSteps + 1);
      tracked.dy = (tracked.dy * tracked.interpolationSteps + newDy) / (tracked.interpolationSteps + 1);
      tracked.speed = Helper.calcPythagoras(tracked.dx, tracked.dy);

      tracked.interpolationSteps++;

      tracked.rawX = newAsteroid.rawX;
      tracked.rawY = newAsteroid.rawY;

      tracked.x = newAsteroid.x;
      tracked.y = newAsteroid.y;
      tracked.addHistory();

   }

   private void trackAsteroids()
   {
      if (oldAsteroidCount == 0 && asteroidCount > 0)
      {

         // new Round => fill all Asteroids
         for (int i = 0; i < asteroidCount; ++i)
         {
            Asteroid newAsteroid = new Asteroid();
            newAsteroid.copyData(asteroids[i]);
            newAsteroid.addHistory();
            trackedAsteroids.add(newAsteroid);
         }
      }
      else if (oldAsteroidCount == asteroidCount && asteroidCount != 0)
      {

         // a Good Time to calc Deltas
         for (Asteroid trackedAst : trackedAsteroids)
         {
            for (int i = 0; i < oldAsteroidCount; ++i)
            {
               Asteroid oldAst = oldAsteroids[i];
               if (trackedAst.rawX == oldAst.rawX && trackedAst.rawY == oldAst.rawY && trackedAst.sizeType == oldAst.sizeType)
               {
                  calcInterpolatedMovement(trackedAst, asteroids[i]);

               }
            }
         }
      }
      else if (oldAsteroidCount != asteroidCount && asteroidCount != 0)
      {

         HashMap<Integer, Integer> trackedToIndexed = new HashMap<Integer, Integer>();

         // a Good Time to calc Deltas
         for (int trackedIndex = 0; trackedIndex < trackedAsteroids.size(); trackedIndex++)
         {
            for (int indexed = 0; indexed < asteroidCount; ++indexed)
            {
               Asteroid trackedAst = trackedAsteroids.get(trackedIndex);
               Asteroid indexedAst = asteroids[indexed];

               double deltaX = calcDiffWithTorus(indexedAst.rawX, trackedAst.rawX, 1024, 800) - trackedAst.dx;
               double deltaY = calcDiffWithTorus(indexedAst.rawY, trackedAst.rawY, 768, 600) - trackedAst.dy;

               if (trackedAst.sizeType == indexedAst.sizeType && Math.abs(deltaX) < 2 && Math.abs(deltaY) < 2)
               {

                  // Found it ?
                  trackedToIndexed.put(trackedIndex, indexed);
                  calcInterpolatedMovement(trackedAst, indexedAst);
               }

            }
         }

         // Now check the missing Tracked and Kick them.
         for (int i = trackedAsteroids.size() - 1; i >= 0; --i)
         {
            if (trackedToIndexed.get(i) == null)
            {
               trackedAsteroids.remove(i);
            }
         }
         // not create the newAsteroids
         for (int i = 0; i < asteroidCount; ++i)
         {
            if (!trackedToIndexed.values().contains(i))
            {
               Asteroid newAsteroid = new Asteroid();
               newAsteroid.copyData(asteroids[i]);
               newAsteroid.addHistory();
               trackedAsteroids.add(newAsteroid);
            }
         }

      }
      else
      {
         trackedAsteroids.clear();
      }

   }

   private void checkAsteroidMovements(boolean buildMeanValue)
   {
      for (int i = 0; i < asteroidCount; ++i)
      {

         Helper.normalizeObject(theShip, asteroids[i]);
      }

      trackAsteroids();
      allObjectsInScene.addAll(trackedAsteroids);

      for (int i = 0; i < asteroidCount; ++i)
      {

         oldAsteroids[i].copyData(asteroids[i]);
      }
      oldAsteroidCount = asteroidCount;

      for (Asteroid currAst : trackedAsteroids)
      {
         currAst.dx = currAst.getMeanDX();
         currAst.dy = currAst.getMeanDY();
      }

   }

   public void analyse(int[] vector_ram)
   {
      // System.err.println("-----------------------------");

      int digitValue = 10000;
      boolean evaluateScoreNumbers = false;
      boolean checkForPlayerMessage = false;

      int newInternalScore = 0;

      theSaucer.present = false;
      theShip.present = false;
      asteroidCount = 0;
      shotCount = 0;

      int dx = 0, dy = 0, sf = 0, vx = 0, vy = 0, vz = 0, vs = 0;
      int v1x = 0;
      int v1y = 0;
      int shipdetect = 0;
      int size = 0;
      Asteroid.SIZES sizeType = Asteroid.SIZES.SMALL;

      if ((vector_ram[0] & 0xff00) != 0xe000 && (vector_ram[0] & 0xff00) != 0xe200)
         return; // sollte nicht vorkommen; erster Befehl ist immer ein JMPL

      int pc = 1;
      while (1 == 1)
      {
         int op = (vector_ram[pc] >> 12) & 0x0f;
         switch (op)
         {
            case 0x0a: // LABS
               vy = ((vector_ram[pc] & 0x3ff) - 0x80);
               vx = (vector_ram[pc + 1] & 0x3ff);
               vs = (vector_ram[pc + 1] >> 12);

               if (vector_ram[pc] == 41836) // Position of the Score
               {
                  evaluateScoreNumbers = true; // now we receive numbers
               }
               else
               {
                  evaluateScoreNumbers = false; // no score numbers
               }

               if (vector_ram[pc] == 41688) // GAMESTART ?
               {
                  checkForPlayerMessage = true;
               }
               else
               {
                  checkForPlayerMessage = false;

               }

               switch (vs)
               {
                  case 0:
                     size = BIG_ASTEROID_SIZE;
                     sizeType = Asteroid.SIZES.BIG;
                     break;
                  case 14:
                     size = SMALL_ASTEROID_SIZE;
                     sizeType = Asteroid.SIZES.SMALL;
                     break;
                  case 15:
                     size = MIDDLE_ASTEROID_SIZE;
                     sizeType = Asteroid.SIZES.MEDIUM;
                     break;
                  default:
                     break;
               }

               break;
            case 0x0b: // HALT

               return;
            case 0x0c: // JSRL
               int asteroidShort = vector_ram[pc] & 0x0fff;

               switch (asteroidShort)
               {

                  case 0x8f3:
                     asteroids[asteroidCount++].setAll(vx, vy, 1, size, sizeType);
                     break;
                  case 0x8ff:
                     asteroids[asteroidCount++].setAll(vx, vy, 2, size, sizeType);
                     break;
                  case 0x90d:
                     asteroids[asteroidCount++].setAll(vx, vy, 3, size, sizeType);
                     break;
                  case 0x91a:
                     asteroids[asteroidCount++].setAll(vx, vy, 4, size, sizeType);
                     break;
                  case 0x929:
                     theSaucer.present = true;
                     theSaucer.x = vx;
                     theSaucer.y = vy;
                     theSaucer.rawX = vx;
                     theSaucer.rawY = vy;
                     theSaucer.size = size;
                     break;

                  default:

                     if (evaluateScoreNumbers)
                     {
                        Integer number = numbers.get(asteroidShort);

                        if (number != null)
                        {
                           newInternalScore += digitValue * number;
                           digitValue /= 10;
                        }
                        else if (asteroidShort == 2860)
                        {
                           digitValue /= 10; // space => no digit
                        }

                        if (digitValue == 0)
                        {
                           if (newInternalScore < internalScore)
                           {
                              internalScoreOverflows++;
                           }
                           internalScore = newInternalScore;
                        }

                     }
                     if (checkForPlayerMessage)
                     {
                        if (asteroidShort == 0xafb)
                        {
                           startMessageWasDisplayed = true;
                           roundCounter = 0;
                           internalScoreOverflows = 0; // new Start => reset
                           // overflow Counter

                           startMessageWasDisplayed = false;
                           this.setChanged();
                           this.notifyObservers(MSG_GAME_STARTED);
                           processedFramesCount = 0;

                        }
                     }

                     break;
               }

               break;
            case 0x0d: // RTSL
               return;
            case 0x0e: // JMPL
               /*
                * pc = vector_ram[pc] & 0xfff; break;
                */
               return;
            case 0x0f: // SVEC
               /*
                * dy = vector_ram[pc] & 0x300; if ((vector_ram[pc] & 0x400) != 0) dy = -dy; dx = (vector_ram[pc] & 3) << 8; if ((vector_ram[pc] & 4) != 0) dx = -dx; sf = (((vector_ram[pc] & 8) >> 2) | ((vector_ram[pc] & 0x800) >> 11)) + 2; vz = (vector_ram[pc] & 0xf0) >> 4;
                */
               break;
            default:
               dy = vector_ram[pc] & 0x3ff;
               if ((vector_ram[pc] & 0x400) != 0)
                  dy = -dy;
               dx = vector_ram[pc + 1] & 0x3ff;
               if ((vector_ram[pc + 1] & 0x400) != 0)
                  dx = -dx;
               sf = op;
               vz = vector_ram[pc + 1] >> 12;
               if (dx == 0 && dy == 0 && vz == 15)
                  shots[shotCount++].setAll(vx, vy);
               if (op == 6 && vz == 12 && dx != 0 && dy != 0)
               {
                  switch (shipdetect)
                  {
                     case 0:
                        v1x = dx;
                        v1y = dy;
                        ++shipdetect;
                        break;
                     case 1:
                        theShip.present = true;
                        theShip.x = vx;
                        theShip.y = vy;
                        theShip.rawX = vx;
                        theShip.rawY = vy;
                        theShip.dirX = v1x - dx;
                        theShip.dirY = v1y - dy;

                        theShip.angleOfDirection = ((double) Math.round(Math.atan2(theShip.dirY, theShip.dirX) * 1000)) / 1000.0;

                        ++shipdetect;
                        break;
                  }
               }
               else if (shipdetect == 1)
                  shipdetect = 0;

               break;
         }
         if (op <= 0x0a)
            ++pc;
         if (op != 0x0e) // JMPL
            ++pc;
      }
   }

   public static int[] toIntArray(byte[] b)
   {
      final int[] ret = new int[b.length / 2];
      for (int i = 0, ptr = 0; i < ret.length; i++, ptr += 2)
      {
         ret[i] = (int) (((int) b[ptr + 1] & 0xff) << 8 | ((int) b[ptr] & 0xff));
      }
      return ret;
   }

   public void printBytes(byte[] bytes)
   {
      for (int i = 0; i < 10; ++i)
      {
         System.err.print(" " + Integer.toHexString(bytes[i] & 0xff));
      }
      System.err.println();
   }

   public void printInts(int[] ints)
   {
      String zeros = "0000";

      for (int i = 0; i < ints.length; ++i)
      {
         String hex = Integer.toHexString(ints[i]);
         System.err.print(" " + zeros.substring(0, 4 - hex.length()) + hex);
      }
      System.err.println();
   }

   /**
    * @return
    */
   public Asteroid[] getAsteroids()
   {
      return asteroids;
   }

   /**
    * @return
    */
   public Shot[] getShots()
   {
      return shots;
   }

   /**
    * @return
    */
   public Ship getTheShip()
   {
      return theShip;
   }

   /**
    * @return
    */
   public int getAsteroidCount()
   {
      return asteroidCount;
   }

   /**
    * @return
    */
   public int getShotCount()
   {
      return shotCount;
   }

   /**
    * @return
    */
   public Saucer getTheSaucer()
   {
      return theSaucer;
   }

   /**
    * @param connection
    */
   public void setConnection(Connection connection)
   {
      this.connection = connection;
   }

   /**
    * @return
    */
   public ArrayList<BaseObject> getAllObjectsInScene()
   {
      return allObjectsInScene;
   }

   /**
    * @return
    */
   public List<Shot> getIdentifiedShots()
   {
      return identifiedShots;
   }

   public int getScore()
   {
      return internalScore + internalScoreOverflows * 100000;
   }

   /**
    * @return
    * @uml.property name="roundCounter"
    */
   public int getRoundCounter()
   {
      return roundCounter;
   }

   public void init()
   {

      leftMovementAngleToAnglebyte.put(-0.099, 252);
      leftMovementAngleToAnglebyte.put(-0.194, 248);
      leftMovementAngleToAnglebyte.put(-0.29, 244);
      leftMovementAngleToAnglebyte.put(-0.391, 240);
      leftMovementAngleToAnglebyte.put(-0.487, 236);
      leftMovementAngleToAnglebyte.put(-0.589, 232);
      leftMovementAngleToAnglebyte.put(-0.686, 228);
      leftMovementAngleToAnglebyte.put(-0.785, 224);
      leftMovementAngleToAnglebyte.put(-0.885, 220);
      leftMovementAngleToAnglebyte.put(-0.981, 216);
      leftMovementAngleToAnglebyte.put(-1.084, 212);
      leftMovementAngleToAnglebyte.put(-1.18, 208);
      leftMovementAngleToAnglebyte.put(-1.28, 204);
      leftMovementAngleToAnglebyte.put(-1.376, 200);
      leftMovementAngleToAnglebyte.put(-1.472, 196);
      leftMovementAngleToAnglebyte.put(-1.67, 191);
      leftMovementAngleToAnglebyte.put(-1.765, 187);
      leftMovementAngleToAnglebyte.put(-1.861, 183);
      leftMovementAngleToAnglebyte.put(-1.962, 179);
      leftMovementAngleToAnglebyte.put(-2.058, 175);
      leftMovementAngleToAnglebyte.put(-2.16, 171);
      leftMovementAngleToAnglebyte.put(-2.257, 167);
      leftMovementAngleToAnglebyte.put(-2.356, 163);
      leftMovementAngleToAnglebyte.put(-2.455, 159);
      leftMovementAngleToAnglebyte.put(-2.552, 155);
      leftMovementAngleToAnglebyte.put(-2.655, 151);
      leftMovementAngleToAnglebyte.put(-2.75, 147);
      leftMovementAngleToAnglebyte.put(-2.851, 143);
      leftMovementAngleToAnglebyte.put(-2.947, 139);
      leftMovementAngleToAnglebyte.put(-3.042, 135);
      leftMovementAngleToAnglebyte.put(0.099, 7);
      leftMovementAngleToAnglebyte.put(0.194, 11);
      leftMovementAngleToAnglebyte.put(0.29, 15);
      leftMovementAngleToAnglebyte.put(0.391, 19);
      leftMovementAngleToAnglebyte.put(0.487, 23);
      leftMovementAngleToAnglebyte.put(0.589, 27);
      leftMovementAngleToAnglebyte.put(0.686, 31);
      leftMovementAngleToAnglebyte.put(0.785, 35);
      leftMovementAngleToAnglebyte.put(0.885, 39);
      leftMovementAngleToAnglebyte.put(0.981, 43);
      leftMovementAngleToAnglebyte.put(1.084, 47);
      leftMovementAngleToAnglebyte.put(1.18, 51);
      leftMovementAngleToAnglebyte.put(1.28, 55);
      leftMovementAngleToAnglebyte.put(1.376, 59);
      leftMovementAngleToAnglebyte.put(1.472, 63);
      leftMovementAngleToAnglebyte.put(1.67, 68);
      leftMovementAngleToAnglebyte.put(1.765, 72);
      leftMovementAngleToAnglebyte.put(1.861, 76);
      leftMovementAngleToAnglebyte.put(1.962, 80);
      leftMovementAngleToAnglebyte.put(2.058, 84);
      leftMovementAngleToAnglebyte.put(2.16, 88);
      leftMovementAngleToAnglebyte.put(2.257, 92);
      leftMovementAngleToAnglebyte.put(2.356, 96);
      leftMovementAngleToAnglebyte.put(2.455, 100);
      leftMovementAngleToAnglebyte.put(2.552, 104);
      leftMovementAngleToAnglebyte.put(2.655, 108);
      leftMovementAngleToAnglebyte.put(2.75, 112);
      leftMovementAngleToAnglebyte.put(2.851, 116);
      leftMovementAngleToAnglebyte.put(2.947, 120);
      leftMovementAngleToAnglebyte.put(3.042, 124);
      rightMovementAngleToAnglebyte.put(-0.099, 249);
      rightMovementAngleToAnglebyte.put(-0.194, 245);
      rightMovementAngleToAnglebyte.put(-0.29, 241);
      rightMovementAngleToAnglebyte.put(-0.391, 237);
      rightMovementAngleToAnglebyte.put(-0.487, 233);
      rightMovementAngleToAnglebyte.put(-0.589, 229);
      rightMovementAngleToAnglebyte.put(-0.686, 225);
      rightMovementAngleToAnglebyte.put(-0.785, 221);
      rightMovementAngleToAnglebyte.put(-0.885, 217);
      rightMovementAngleToAnglebyte.put(-0.981, 213);
      rightMovementAngleToAnglebyte.put(-1.084, 209);
      rightMovementAngleToAnglebyte.put(-1.18, 205);
      rightMovementAngleToAnglebyte.put(-1.28, 201);
      rightMovementAngleToAnglebyte.put(-1.376, 197);
      rightMovementAngleToAnglebyte.put(-1.472, 193);
      rightMovementAngleToAnglebyte.put(-1.67, 188);
      rightMovementAngleToAnglebyte.put(-1.765, 184);
      rightMovementAngleToAnglebyte.put(-1.861, 180);
      rightMovementAngleToAnglebyte.put(-1.962, 176);
      rightMovementAngleToAnglebyte.put(-2.058, 172);
      rightMovementAngleToAnglebyte.put(-2.16, 168);
      rightMovementAngleToAnglebyte.put(-2.257, 164);
      rightMovementAngleToAnglebyte.put(-2.356, 160);
      rightMovementAngleToAnglebyte.put(-2.455, 156);
      rightMovementAngleToAnglebyte.put(-2.552, 152);
      rightMovementAngleToAnglebyte.put(-2.655, 148);
      rightMovementAngleToAnglebyte.put(-2.75, 144);
      rightMovementAngleToAnglebyte.put(-2.851, 140);
      rightMovementAngleToAnglebyte.put(-2.947, 136);
      rightMovementAngleToAnglebyte.put(-3.042, 132);
      rightMovementAngleToAnglebyte.put(0.099, 4);
      rightMovementAngleToAnglebyte.put(0.194, 8);
      rightMovementAngleToAnglebyte.put(0.29, 12);
      rightMovementAngleToAnglebyte.put(0.391, 16);
      rightMovementAngleToAnglebyte.put(0.487, 20);
      rightMovementAngleToAnglebyte.put(0.589, 24);
      rightMovementAngleToAnglebyte.put(0.686, 28);
      rightMovementAngleToAnglebyte.put(0.785, 32);
      rightMovementAngleToAnglebyte.put(0.885, 36);
      rightMovementAngleToAnglebyte.put(0.981, 40);
      rightMovementAngleToAnglebyte.put(1.084, 44);
      rightMovementAngleToAnglebyte.put(1.18, 48);
      rightMovementAngleToAnglebyte.put(1.28, 52);
      rightMovementAngleToAnglebyte.put(1.376, 56);
      rightMovementAngleToAnglebyte.put(1.472, 60);
      rightMovementAngleToAnglebyte.put(1.67, 65);
      rightMovementAngleToAnglebyte.put(1.765, 69);
      rightMovementAngleToAnglebyte.put(1.861, 73);
      rightMovementAngleToAnglebyte.put(1.962, 77);
      rightMovementAngleToAnglebyte.put(2.058, 81);
      rightMovementAngleToAnglebyte.put(2.16, 85);
      rightMovementAngleToAnglebyte.put(2.257, 89);
      rightMovementAngleToAnglebyte.put(2.356, 93);
      rightMovementAngleToAnglebyte.put(2.455, 97);
      rightMovementAngleToAnglebyte.put(2.552, 101);
      rightMovementAngleToAnglebyte.put(2.655, 105);
      rightMovementAngleToAnglebyte.put(2.75, 109);
      rightMovementAngleToAnglebyte.put(2.851, 113);
      rightMovementAngleToAnglebyte.put(2.947, 117);
      rightMovementAngleToAnglebyte.put(3.042, 121);

   }

   private double findBestFittingAngle(double wantedAngle)
   {
      return ((double) Math.round(wantedAngle / this.internalAngleStep)) * this.internalAngleStep;
   }

   private boolean checkInternalShipAngle(byte lastCommand)
   {
      boolean syncHappened = false;
      byte deltaFrames;

      if (Math.abs(theShip.angleOfDirection - theOldShip.angleOfDirection) < 0.01)
         shipsScreenAngleChanged = false;
      else

      {
         shipsScreenAngleChanged = true;
      }

      if (leftCommand)
      {
         deltaFrames = (byte) (lastFrameNumber - leftCommandFrame);

         if (deltaFrames == 1)
         {
            angleByte += 3;
            // try sync
            Integer syncAngleByte = leftMovementAngleToAnglebyte.get(Helper.roundToDigits(theShip.angleOfDirection, 1000));
            if (syncAngleByte != null && !shipsScreenAngleChanged)
            {
               // System.err.println("L Sync " + syncAngleByte + " " + theShip.angleOfDirection);

               if (Math.abs(syncAngleByte - angleByte) != 0)
               {
                  // System.err.println("SYNC !!!");
                  angleByte = syncAngleByte;
                  syncHappened = true;
               }
            }
         }
         else
         {
            angleByte += deltaFrames * 3;
         }

         if (angleByte > 255)
            angleByte -= 256;

      }

      if (rightCommand)
      {
         deltaFrames = (byte) (lastFrameNumber - rightCommandFrame);

         if (deltaFrames == 1)
         {
            angleByte -= 3;

            // try sync
            Integer syncAngleByte = rightMovementAngleToAnglebyte.get(Helper.roundToDigits(theShip.angleOfDirection, 1000));
            if (syncAngleByte != null && !shipsScreenAngleChanged)
            {
               // System.err.println("R Sync " + syncAngleByte + " " + theShip.angleOfDirection);
               if (Math.abs(syncAngleByte - angleByte) != 0)
               {
                  // System.err.println("SYNC !!!");

                  angleByte = syncAngleByte;
                  syncHappened = true;
               }
            }
         }
         else
         {
            angleByte -= deltaFrames * 3;
         }
      }
      if (angleByte < 0)
         angleByte += 256;

      // kick Movement Flags
      leftCommand = false;
      rightCommand = false;
      leftCommandFrame = -1;
      rightCommandFrame = -1;

      if ((lastCommand & Command.ACTION_LEFT) != 0)
      {
         leftCommandFrame = lastFrameNumber;
         leftCommand = true;
      }
      if ((lastCommand & Command.ACTION_RIGHT) != 0)
      {
         rightCommandFrame = lastFrameNumber;
         rightCommand = true;
      }

      return syncHappened;
   }

   /**
    * @return
    */
   public Integer getAngleByte()
   {
      return angleByte;
   }

   /**
    * @return
    */
   public boolean isLeftCommand()
   {
      return leftCommand;
   }

   /**
    * @return
    */
   public boolean isRightCommand()
   {
      return rightCommand;
   }

   /**
    * @return
    */
   public List<Asteroid> getTrackedAsteroids()
   {
      return trackedAsteroids;
   }

   /**
    * @return
    * @uml.property name="oldAsteroids"
    */
   public Asteroid[] getOldAsteroids()
   {
      return oldAsteroids;
   }

   public double getCurrentAngleByWinkelbyte()
   {
      return Helper.winkelByteToAngle.get(this.angleByte);
   }

   public double getAngleByWinkelbyte(int winkelbyte)
   {
      if (winkelbyte > 255)
         winkelbyte -= 256;
      if (winkelbyte < 0)
         winkelbyte += 256;

      return Helper.winkelByteToAngle.get(winkelbyte);
   }

   public HashMap<Integer, Double> getWinkelByteToAngle()
   {
      return Helper.winkelByteToAngle;
   }

}
