/*
 *
 * Asteroids Player
 *
 * for the c't Competition 2008 (Creativ '08) 
 *
 * Copyright 2008, Volker Raum, Erlangen, Germany
 *
 */
package de.volkerraum.asteroids.ui;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.event.MouseEvent;
import java.awt.event.MouseMotionAdapter;
import java.text.DecimalFormat;
import java.util.Observable;
import java.util.Observer;
import java.util.Random;

import javax.swing.JPanel;

import de.volkerraum.asteroids.model.Asteroid;
import de.volkerraum.asteroids.model.AsteroidsModel;
import de.volkerraum.asteroids.model.BaseObject;
import de.volkerraum.asteroids.model.Helper;
import de.volkerraum.asteroids.model.Ship;
import de.volkerraum.asteroids.model.Shot;
import de.volkerraum.asteroids.model.Target;
import de.volkerraum.asteroids.player.Player;

/**
 * @author Volker Raum (C) 2007
 */
public class GameArea extends JPanel implements Observer
{
   AsteroidsModel theModel = null;

   Color[] colors = new Color[100];

   Player thePlayer;
   /**
    * @uml.property name="showTrajectories"
    */
   boolean showTrajectories = false;

   Dimension dim = null;

   DecimalFormat df = new DecimalFormat("0.000");

   public GameArea(AsteroidsModel model, Player player)
   {
      super();
      this.theModel = model;
      thePlayer = player;
      model.addObserver(this);
      Random rand = new Random(System.currentTimeMillis());
      for (int i = 0; i < 100; ++i)
      {
         colors[i] = new Color(rand.nextInt(255), rand.nextInt(255), rand.nextInt(255));
      }

      this.setToolTipText("");

      this.addMouseMotionListener(new MouseMotionAdapter()
      {
         public void mouseMoved(MouseEvent e)
         {
            double screenWidth = dim.getWidth();
            double screenHeight = dim.getHeight();

            double screenX = e.getX();
            double screenY = e.getY();

            double astX = screenX / screenWidth * AsteroidsModel.originalSizeX;
            double astY = AsteroidsModel.originalSizeY - (screenY / screenHeight * AsteroidsModel.originalSizeY);

            boolean set = false;
            int index = 0;
            for (BaseObject currObject : theModel.getAllObjectsInScene())
            {
               if (Helper.calcPythagoras(currObject.rawX - astX, currObject.rawY - astY) < 10)
               {

                  setToolTipText(getTooltipForObject(currObject, index));
                  set = true;
                  break;
               }
               index++;
            }
            if (!set && Helper.calcPythagoras(theModel.getTheShip().rawX - astX, theModel.getTheShip().rawY - astY) < 10)
            {
               setToolTipText(getTooltipForObject(theModel.getTheShip(), 0));
               set = true;
            }
            if (!set)
            {
               setToolTipText(((int) astX) + "/" + ((int) astY));
            }
         }
      });
   }

   private String getTooltipForObject(BaseObject currObject, int index)
   {
      StringBuilder builder = new StringBuilder("<html><body> #" + index + " " + currObject.getClass().getSimpleName() + " Size " + currObject.size + "<br>");

      builder.append("<table>");

      addHTMLTableRow(builder, "X/Y", currObject.x, currObject.y);
      addHTMLTableRow(builder, "rawX/rawY", currObject.x, currObject.y);
      addHTMLTableRow(builder, "Speed/MoveAngle", df.format(currObject.speed), df.format(currObject.angleOfMovement));

      if (currObject instanceof Ship)
      {
         Ship aShip = (Ship) currObject;
         addHTMLTableRow(builder, "InternalAngle/Byte ", df.format(aShip.internalAngle), aShip.angleByte);
         addHTMLTableRow(builder, "Internal DX/DY ", df.format(aShip.internalDX), df.format(aShip.internalDY));
      }

      addHTMLTableRow(builder, "dX/dY ", currObject.dx, currObject.dy);
      addHTMLTableRow(builder, "Dist/Angle to Ship", ((int) currObject.distanceToShip), df.format(currObject.angleToShip));

      addHTMLTableRow(builder, "Intercept possible/Angle/frames ", currObject.interceptPossible, df.format(currObject.interceptAngle), currObject.interceptFrames);
      addHTMLTableRow(builder, "Impact imminent/frames ", currObject.impactImminent, currObject.impactFrames);
      addHTMLTableRow(builder, "Shots At ", currObject.shotsFiredAt);
      addHTMLTableRow(builder, "TargetsInSector ", currObject.futureTargetCountInSector);

      builder.append("</table>");

      builder.append("</body></html>");
      return builder.toString();
   }

   public void addHTMLTableRow(StringBuilder builder, Object... values)
   {
      builder.append("<tr>");
      for (Object currValue : values)
      {
         builder.append("<td>" + currValue.toString() + "</td>");
      }
      builder.append("</tr>");
   }

   public void paintComponent(Graphics gr)
   {
      DecimalFormat df = new DecimalFormat("#0");
      DecimalFormat shotSpeedFormat = new DecimalFormat("#0.0");
      super.paintComponent(gr);

      dim = this.getSize();

      double width = dim.getWidth();
      double height = dim.getHeight();

      double pixelPerX = width / AsteroidsModel.originalSizeX;
      double pixelPerY = height / AsteroidsModel.originalSizeY;

      for (int i = 0; i < theModel.getAsteroidCount(); ++i)
      {
         Asteroid ast = (Asteroid) theModel.getAllObjectsInScene().get(i);
         int size = (int) ast.size;
         double astX = pixelPerX * ast.rawX;
         double astY = pixelPerY * (AsteroidsModel.originalSizeY - ast.rawY);

         gr.setColor(colors[i]);
         gr.fillOval((int) astX - size / 2, (int) astY - size / 2, size, size);

         gr.setColor(Color.black);
         gr.drawString(i + "#" + df.format(ast.speed) + " - " + df.format(ast.dx) + "/" + df.format(ast.dy) + " S(" + ast.shotsFiredAt + ")", (int) astX, (int) astY);
         gr.drawString(df.format(ast.x) + "/" + df.format(ast.y), (int) astX, (int) astY + 14);

         if (showTrajectories)
         {

            gr.setColor(Color.BLACK);
            for (int u = 0; u < 40; u = u + 5)
            {
               int x = (int) (pixelPerX * (ast.rawX + u * ast.dx));
               int y = (int) (pixelPerY * (AsteroidsModel.originalSizeY - (ast.rawY + u * ast.dy)));
               int halfsize = (int) (ast.size / 2);
               gr.drawOval(x - halfsize, y - halfsize, (int) (ast.size ), (int) (ast.size ));
            }

         }

         if (ast.interceptFrames < 10000)
         {
            gr.setColor(Color.BLUE);
            int x = (int) (pixelPerX * (ast.interceptX));
            int y = (int) (pixelPerY * (AsteroidsModel.originalSizeY - ast.interceptY));

            gr.drawOval(x - 3, y - 3, 6, 6);
            gr.drawLine(x, y, (int) astX, (int) astY);

            gr.drawString(ast.interceptFrames + "", x, y);

         }

         if (ast.impactFrames < 10000)
         {
            gr.setColor(Color.RED);
            int x = (int) (pixelPerX * (ast.impactX));
            int y = (int) (pixelPerY * (AsteroidsModel.originalSizeY - ast.impactY));

            gr.drawOval(x - 3, y - 3, 6, 6);
            gr.drawLine(x, y, (int) astX, (int) astY);
            gr.drawString(ast.impactFrames + "", x, y);

         }

      }

      gr.setColor(Color.BLACK);
      for (int i = 0; i < theModel.getShotCount(); ++i)
      {
         Shot shot = theModel.getShots()[i];
         double shotX = pixelPerX * shot.rawX;

         double shotY = pixelPerY * (AsteroidsModel.originalSizeY - shot.rawY);

         gr.drawOval((int) shotX - 1, (int) shotY - 1, 2, 2);

         if (showTrajectories)
         {
            gr.setColor(Color.BLACK);
            for (int u = 0; u < 40; u = u + 5)
            {
               int x = (int) (pixelPerX * (shot.rawX + u * shot.dx));
               int y = (int) (pixelPerY * (AsteroidsModel.originalSizeY - (shot.rawY + u * shot.dy)));
               gr.drawOval((int) x - 1, (int) y - 1, 2, 2);
            }
         }
      }

      Ship ship = theModel.getTheShip();
      if (ship.present)
      {
         double shipX = pixelPerX * ship.x;
         double shipY = pixelPerY * (AsteroidsModel.originalSizeY - ship.y);

         double dx = pixelPerX * Math.cos(ship.angleOfDirection) * 20;
         double dy = pixelPerY * Math.sin(ship.angleOfDirection) * 20;
         gr.drawRect((int) shipX - 5, (int) shipY - 5, 10, 10);

         gr.drawString(df.format(ship.x) + "/" + df.format(ship.y) + " AB " + ship.angleByte, (int) shipX, (int) shipY);

         gr.drawLine((int) shipX, (int) shipY, (int) (shipX + dx), (int) (shipY - dy));

         double shipDX = pixelPerX * (ship.x + ship.dx);
         double shipDY = pixelPerY * (AsteroidsModel.originalSizeY - (ship.y + ship.dy));

         gr.drawLine((int) shipX, (int) shipY, (int) (shipDX), (int) (shipDY));

      }

      if (theModel.getTheSaucer().present)
      {
         double size = theModel.getTheSaucer().size;
         double saucerX = pixelPerX * (theModel.getTheSaucer().rawX - size / 2);
         double saucerY = pixelPerY * ((AsteroidsModel.originalSizeY - theModel.getTheSaucer().rawY) - size / 2);
         gr.drawRect((int) saucerX, (int) saucerY, (int) size, (int) size);

         if (showTrajectories)
         {
            gr.setColor(Color.BLACK);
            for (int u = 0; u < 40; u = u + 5)
            {
               int x = (int) (pixelPerX * (theModel.getTheSaucer().rawX + u * theModel.getTheSaucer().dx));
               int y = (int) (pixelPerY * (AsteroidsModel.originalSizeY - (theModel.getTheSaucer().rawY + u * theModel.getTheSaucer().dy)));
               gr.drawOval((int) x - 1, (int) y - 1, 2, 2);
            }
         }
         double saucerInterceptX = pixelPerX * (theModel.getTheSaucer().interceptX - size / 2);
         double saucerInterceptY = pixelPerY * ((AsteroidsModel.originalSizeY - theModel.getTheSaucer().interceptY) - size / 2);

         gr.setColor(Color.BLUE);
         gr.drawLine((int) saucerX, (int) saucerY, (int) saucerInterceptX, (int) saucerInterceptY);

      }

      if (thePlayer.getNextTarget() != null)
      {
         Target target = thePlayer.getNextTarget();

         gr.setColor(Color.red);
         double targetX = pixelPerX * (thePlayer.getNextTargetedObject().rawX);
         double targetY = pixelPerY * ((AsteroidsModel.originalSizeY - thePlayer.getNextTargetedObject().rawY));
         gr.fillOval((int) targetX - 5, (int) targetY - 5, (int) 10, (int) 10);

         double interceptTargetX = pixelPerX * (thePlayer.getNextTarget().interceptX);
         double interceptTargetY = pixelPerY * ((AsteroidsModel.originalSizeY - thePlayer.getNextTarget().interceptY));

         gr.fillOval((int) interceptTargetX - 5, (int) interceptTargetY - 5, (int) 10, (int) 10);

         gr.drawLine((int) targetX, (int) targetY, (int) interceptTargetX, (int) interceptTargetY);

         double deltaX = pixelPerX * (ship.x + Math.cos(target.angle) * 300);
         double deltaY = pixelPerY * ((AsteroidsModel.originalSizeY - (ship.y + Math.sin(target.angle) * 300)));

         double shipX = pixelPerX * ship.x;
         double shipY = pixelPerY * (AsteroidsModel.originalSizeY - ship.y);

         gr.drawLine((int) shipX, (int) shipY, (int) deltaX, (int) deltaY);

      }

      for (int i = 0; i < theModel.getIdentifiedShots().size(); ++i)
      {
         Shot shot = theModel.getIdentifiedShots().get(i);
         double shotX = pixelPerX * shot.rawX;
         double shotY = pixelPerY * (AsteroidsModel.originalSizeY - shot.rawY);
         gr.setColor(Color.GREEN);
         gr.drawOval((int) shotX - 3, (int) shotY - 3, 6, 6);

         gr.setColor(Color.BLACK);
         gr.drawString(i + "#" + df.format(shot.x) + "/" + df.format(shot.y), (int) shotX, (int) shotY);

      }

   }

   public void update(Observable o, Object arg)
   {
      this.repaint();
   }

   /**
    * @return
    * @uml.property name="showTrajectories"
    */
   public boolean isShowTrajectories()
   {
      return showTrajectories;
   }

   /**
    * @param showTrajectories
    * @uml.property name="showTrajectories"
    */
   public void setShowTrajectories(boolean showTrajectories)
   {
      this.showTrajectories = showTrajectories;
   }
}
