/**
 * File:    Position.java
 * Package: de.heise.asteroid
 * Created: 15.05.2008 07:57:21
 * Author:  Chr. Moellenberg
 *
 * Copyright (c) 2008 by Chr. Moellenberg
 */

package de.heise.asteroid;

/**
 * <code>Position</code> reflects a co-ordinate pair on screen.
 * <p>
 * It is a value object which, once created, cannot be changed. If a
 * <code>Position</code> becomes invalid, it is simply thrown away.
 * 
 * @author Chr. Moellenberg
 */
public class Position {
   public static final Position NULL_POS = new Position(0, 0);

   protected final int x;
   protected final int y;

   public Position(int px, int py) {
      x = px;
      y = py;
   }
   
   /**
    * Returns the horizontal (x) component of this <code>Position</code>.
    * 
    * @return the horizontal (x) component
    */
   public int getX() {
      return x;
   }

   /**
    * Returns the vertical (y) component of this <code>Position</code>.
    * 
    * @return the vertical (y) component
    */
   public int getY() {
      return y;
   }

   /**
    * Returns a distance vector to another position.
    * @param p the other <code>Position</code>
    * @return the distance vector
    */
   public ScreenVector getDistanceTo(Position p) {
      return (p == null) ? null : new ScreenVector(p.x - this.x, p.y - this.y);
   }
   
   /* (non-Javadoc)
    * @see java.lang.Object#equals(java.lang.Object)
    */
   @Override
   public boolean equals(Object o) {
      if (o instanceof Position) {
         Position p = (Position)o;
         return x == p.x && y == p.y;
      }
      return false;
   }

   /* (non-Javadoc)
    * @see java.lang.Object#toString()
    */
   @Override
   public String toString() {
      StringBuilder strb = new StringBuilder();
      strb.append(x).append(';').append(y);
      return strb.toString();
   }
}
