package de.heise.asteroid.comm;

import java.net.DatagramPacket;

/**
 * The <code>KeyPacket</code> class represents the client reply packet. It
 * contains the id-prefix, the pressed keys and the one-byte ping value.
 * 
 * @see ServerConnection
 */
public class KeyPacket {
   private byte[] payload;
   private int targetFameNo;

   private DatagramPacket datagramPacket;

   /**
    * Initializes a new <code>KeyPacket</code>.
    */
   public KeyPacket() {
      payload = new byte[8];
      datagramPacket = new DatagramPacket(payload, payload.length);
      payload[0] = 'c';
      payload[1] = 't';
      payload[2] = 'm';
      payload[3] = 'a';
      payload[4] = 'm';
      payload[5] = 'e';
      payload[6] = '@'; // keys
      payload[7] = 0; // ping
   }

   /**
    * Returns one-byte ping value.
    * 
    * @return the ping value
    */
   public byte getPing() {
      return payload[7];
   }

   /**
    * Sets the ping value. The ping value from this <code>KeyPacket</code>.
    * Use this to detect network latency.
    * 
    * @param ping the one-byte ping value
    * @see FramePacket#getPing()
    */
   public void setPing(byte ping) {
      payload[7] = ping;
   }

   /**
    * Returns the <code>DatagramPacket</code> for sending the network data
    * 
    * @return the <code>DatagramPacket</code>
    * @see ServerConnection
    */
   public DatagramPacket getDatagramPacket() {
      return datagramPacket;
   }

   /**
    * Returns the key status transmitted with this <code>KeyPacket</code>.
    * 
    * @return the key status
    */
   public byte getKeys() {
      return payload[6];
   }

   /**
    * Sets the key status.
    * 
    * @param keys the status of all keys coded in one byte
    * @see Communicator
    */
   public void setKeys(byte keys) {
      payload[6] = keys;
   }

   /**
    * Returns the frame number when this <code>KeyPacket</code> shall be sent.
    * @return the targetFameNo the target frame number
    */
   public int getTargetFameNo() {
      return targetFameNo;
   }

   /**
    * Sets the frame number when this <code>KeyPacket</code> shall be sent.
    * @param targetFameNo the target fame number to set
    */
   public void setTargetFameNo(int targetFameNo) {
      this.targetFameNo = targetFameNo;
   }
}
