/**
 * File:    AbstractProcessor.java
 * Package: de.heise.asteroid
 * Created: 24.05.2008 11:05:22
 * Author:  Chr. Moellenberg
 *
 * Copyright (c) 2008 by Chr. Moellenberg
 */

package de.heise.asteroid.engine;


/**
 * @author Chr. Moellenberg
 *
 */
public abstract class AbstractProcessor implements FrameProcessor {
   private Engine engine;
   private byte keys;
   
   /**
    * Associates this processor with the given <code>Engine</code>.
    *  
    * @param e the <code>Engine</code>
    */
   public AbstractProcessor(Engine e) {
      engine = e;
   }
   
   /**
    * Makes the <code>engine</code> available to the concrete player strategy.
    * 
    * @return the <code>Engine</code>
    */
   protected Engine getEngine() {
      return engine;
   }
   
   /**
    * Presses one or more keys. Several keys can be pressed simultaneously by combining
    * the key codes with a bitwise OR.
    * 
    * @param key one or more keys.
    */
   protected void pressKey(byte key) {
      keys |= key & KEY_MASK;
   }

   /**
    * Releases one or more keys. Several keys can be released simultaneously by combining
    * the key codes with a bitwise OR.
    * 
    * @param key one or more keys.
    */
   protected void releaseKey(byte key) {
      keys &= ~(key & KEY_MASK);
   }

   /**
    * Releases all keys.
    */
   protected void releaseKeys() {
      keys = '@';
   }

   /**
    * Schedules a keypress for a given frame number.
    * @param frameNo the frame number when the keys shell actually be pressed
    * @see Engine#sendKeys(byte, int)
    */
   protected void sendKeys(int frameNo) {
      engine.sendKeys(keys, frameNo);
   }
}
