/**
 * File:    SaucerTracker.java
 * Package: de.heise.asteroid.engine
 * Created: 15.06.2008 18:13:58
 * Author:  Chr. Moellenberg
 *
 * Copyright (c) 2008 by Chr. Moellenberg
 */

package de.heise.asteroid.engine;

import de.heise.asteroid.Position;
import de.heise.asteroid.ScreenVector;
import de.heise.asteroid.model.Saucer;
import de.heise.asteroid.util.SimpleMath;

/**
 * @author Chr. Moellenberg
 *
 */
public class SaucerTracker {
   private Saucer saucer;
   private Explosion explosion;
   private boolean saucerDetected;
   private int px;
   private int py;
   private int vx;
   private int vy;
   private Position pos;
   private ScreenVector speed;
   private int curFrame;
   private int baseFrame;

   public SaucerTracker() {
      reset();
   }
   
   public void reset() {
      saucer = null;
      start(0);
   }

   public void start(int frameNo) {
      curFrame = frameNo;
      saucerDetected = false;
      if (explosion != null) {
         if (explosion.getFramesRemaining(curFrame) <= 0) {
//            System.out.println("End of saucer explosion: " + saucer);
            saucer = null;
            explosion = null;
         }
      }
   }
   
   public boolean trackTarget(int sx, int sy, int type, int size) {
      if (type == GameStatus.SAUCER) {
         saucerDetected = true;
         if (saucer == null) {
            px = sx << 3;
            py = sy << 3;
            vx = sx < 512 ? 16 : -16;
            vy = 0;
            pos = null;
            speed = new ScreenVector(vx, vy);
            saucer = new Saucer(getPos(), size == 15 ? Saucer.Size.BIG : Saucer.Size.SMALL, speed);
//            System.out.println("Saucer detected: " + saucer);
         } else {
            updatePosition(sx, sy);
            saucer.update(getPos(), speed);
         }
         baseFrame = curFrame;
         return true;
      }
      if (saucer != null && !saucerDetected && GameStatus.isExplosionType(type)) {
         if (explosion != null) {
            return explosion.match(curFrame, sx, sy, type, size);
         }
         int dx = SimpleMath.normalize(sx - (px >> 3), -512, 512);
         int dy = SimpleMath.normalize(sy - (py >> 3), -384, 384);
         if (dx == 0 && dy == 0) {
            startExplosion(sx, sy, type, size);
            return true;
         }
      }
      return false;
   }
   
   public Position getPos() {
      if (pos == null) {
         pos = new Position(px, py);
      }
      return pos;
   }
   
   public ScreenVector getSpeed() {
      return speed;
   }
   
   public Saucer getSaucer() {
      return (explosion == null) ? saucer : null;
   }

   private void updatePosition(int sx, int sy) {
      int x = SimpleMath.normalize(px + vx * (curFrame - baseFrame), 0, 8192);
      int y = SimpleMath.normalize(py + vy * (curFrame - baseFrame), 1024, 7168);
      int dx = SimpleMath.normalize(sx - (x >> 3), -512, 512);
      int dy = SimpleMath.normalize(sy - (y >> 3), -384, 384);
      if (dx == 0) {
         px = x;
      } else {
         System.out.println("Saucer seems to hop");
         px = sx << 3;
      }
      if (dy == 0) {
         py = y;
      } else {
//         System.out.println("Saucer seems to change heading");
         vy += (dy < 0) ? -16 : 16;
         py = sy << 3;
         speed = new ScreenVector(vx, vy);
      }
      pos = null;
   }
   
   private void startExplosion(int sx, int sy, int t, int s) {
      vx = 0;
      vy = 0;
      speed = ScreenVector.NULL_VECTOR;
      explosion = new Explosion(curFrame, sx, sy, t, s);
      saucer.setExploding(true);
//      int index = explosion.getIndex();
//      System.out.printf("Saucer exploding at (%d,%d), animation index %d\n", px, py, index);
   }

   public void cleanup() {
      if (!saucerDetected && saucer != null && explosion == null) {
//         System.out.printf("Saucer disappeared at (%d,%d)\n", px, py);
         saucer = null;
      }
   }
}
