/**
 * File:    ModifiedSamplePlayer.java
 * Package: de.heise.asteroid
 * Created: 19.05.2008 22:57:28
 * Author:  Chr. Moellenberg
 *
 * Based on demo program by Harald Boegeholz.
 * Copyright (c) 2008 by Harald Boegeholz, c't Magazin fure Computertechnik
 */

package de.heise.asteroid.player;

import java.util.List;

import de.heise.asteroid.Position;
import de.heise.asteroid.ScreenVector;
import de.heise.asteroid.engine.Engine;
import de.heise.asteroid.model.Asteroid;
import de.heise.asteroid.model.Saucer;
import de.heise.asteroid.model.Ship;
import de.heise.asteroid.util.SimpleMath;

/**
 * The <code>ModifiedSamplePlayer</code> class implements a strategy similar to
 * the one created by Harald Boegeholz, c't Magazin fuer Computertechnik. It fires 
 * only when the target direction seems to be reached. Also, it does not move if 
 * there are no objects on screen.
 * 
 * @see AbstractPlayer
 */
public class ModifiedSamplePlayer extends AbstractPlayer {
   private static final double maxSinPhi = 0.9;
   private static final Position centerPos = new Position(4096, 4096);
   private boolean hasFired = false;
   private boolean initialized = false;
   private Ship ship;

   /**
    * Initializes a Player with the given engine.
    * 
    * @param e the <code>Engine</code>
    */
   public ModifiedSamplePlayer(Engine e) {
      super(e);
   }

   /* (non-Javadoc)
    * @see de.heise.asteroid.player.AbstractPlayer#play()
    */
   public void play() {
      // System.out.println(status); // dump game status
      ship = status.getShip();
      Saucer saucer = status.getSaucer();
      
      releaseKeys();
      if (ship == null) {
         initialized = false;
         sendKeys(curFrame);
      } else if (!initialized) {
         pressKey(KEY_LEFT);
         sendKeys(curFrame);
         releaseKeys();
         sendKeys(curFrame + 6);
         waitForFrame(curFrame + 8);
         initialized = true;
      } else {
         int minSqDist = 0x7fffffff;
         ScreenVector targetDist = null;
         ScreenVector targetSpeed = null;
         ScreenVector targetDir = null;
         int targetRadius = 0;
         // search for nearest asteroid
         List<Asteroid> asteroids = status.getAsteroids();
         for (Asteroid ast : asteroids) {
            if (ast.isExploding()) {
               continue;
            }
            ScreenVector distVector = ship.getDistanceTo(ast);
            distVector.normalize();
            // the squared distance
            int sqDist = distVector.sqAbs();
            // adjust the squared distance for the radius of the asteroid
            sqDist -= ast.getRadius() * ast.getRadius();
            if (sqDist < minSqDist) {
               ScreenVector dir = getTargetAngle(distVector, ast.getSpeed());
               if (dir != null) {
                  targetDir = dir;
                  minSqDist = sqDist;
                  targetDist = distVector;
                  targetSpeed = ast.getSpeed();
                  targetRadius = ast.getRadius();
               }
            }
         }
         if (saucer != null && !saucer.isExploding()) {
            ScreenVector distVector = ship.getDistanceTo(saucer);
            distVector.normalize();
            int sqDist = distVector.sqAbs();
            // adjust the squared distance for the radius of the flying saucer
            sqDist -= saucer.getSize().getDistanceCorrection();
            if ((sqDist >> 3) < minSqDist) { // hack: make ufos more important by letting them appear closer
               ScreenVector dir = getTargetAngle(distVector, saucer.getSpeed());
               if (dir != null) {
                  targetDir = dir;
                  minSqDist = sqDist;
                  targetDist = distVector;
                  targetSpeed = saucer.getSpeed();
                  targetRadius = saucer.getRadius();
               }
            }
         }
         // turn the ship towards a direction where the selected target can be shot
         if (targetDist != null && targetSpeed != null && targetDir != null) {
            turnOrFire(targetDir);
            if (minSqDist > 10000000) {
               pressKey(KEY_THRUST);
            }
         } else {
            if (asteroids.isEmpty()) {
               targetDir = ship.getPos().getDistanceTo(centerPos);
               int d = (int)Math.round(targetDir.abs());
               if (d > 1800) {
                  int crossCenter = getCrossProd(ship.getAngle(), targetDir);
                  if (crossCenter > 0) {
                     pressKey(KEY_LEFT);
                  } else if (crossCenter < 0) {
                     pressKey(KEY_RIGHT);
                  }
                  if (Math.abs(crossCenter) < d * 10) {
                     pressKey(KEY_THRUST);
                  }
               }
            }
         }
         
         // evade collision with jump to hyperspace
         int danger = targetRadius + 200;
         if (minSqDist < danger * danger) { // screen: 28 * 28
            pressKey(KEY_HYPERSPACE);
            releaseKey((byte)(KEY_THRUST | KEY_LEFT | KEY_RIGHT));
         }
         sendKeys(curFrame);
      }
   }

   private void turnOrFire(ScreenVector targetDir) {
      int shipAngle = ship.getAngle();
      int crossLeft = getCrossProd((shipAngle + 3) & 0xff, targetDir);
      if (crossLeft >= 0) {
         pressKey(KEY_LEFT);
         hasFired = false;
      } else {
         int crossRight = getCrossProd((shipAngle - 3) & 0xff, targetDir);
         if (crossRight <= 0) {
            pressKey(KEY_RIGHT);
            hasFired = false;
         } else {
            if (hasFired) {
               // cannot (yet) fire again
               hasFired = false;
            } else {
               pressKey(KEY_FIRE);
               hasFired = true;
            }
         }
      }
   }

   private int getCrossProd(int angle, ScreenVector targetDir) {
      ScreenVector shipDir = new ScreenVector(SimpleMath.astCos[angle], SimpleMath.astSin[angle]); 
      return shipDir.crossProduct2D(targetDir);
   }

   private ScreenVector getTargetAngle(ScreenVector targetDist, ScreenVector targetSpeed) {
      double dVb = 63.0 * targetDist.abs();
      double delta = Math.atan2(targetDist.getY(), targetDist.getX());
      double sinPhi = (double)targetSpeed.diff(ship.getSpeed()).crossProduct2D(targetDist) / dVb;
//      double sinPhi = (double)targetSpeed.crossProduct2D(targetDist) / dVb;
      if (sinPhi <= maxSinPhi && sinPhi >= -maxSinPhi) {
         double phi = Math.asin(sinPhi);
         double sigma = delta - phi;
         return new ScreenVector((int)(100.0 * Math.cos(sigma)), (int)(100.0 * Math.sin(sigma)));
      } else {
         return null;
      }
   }

   /* (non-Javadoc)
    * @see de.heise.asteroid.Player#getInitials()
    */
   public String getInitials() {
      return "MOE"; // Christian Moellenberg
   }
}
