/**
 * File:    Worker.java
 * Package: de.heise.asteroid.util
 * Created: 19.05.2008 23:05:45
 * Author:  Chr. Moellenberg
 *
 * Copyright (c) 2008 by Chr. Moellenberg
 */

package de.heise.asteroid.util;

/**
 * @author Chr. Moellenberg
 *
 */
public class Worker extends Thread implements WorkerRunnable {
   private boolean term;
   private WorkerRunnable runnable;
   
   public Worker(String name) {
      super(name);
      runnable = this;
   }

   public Worker(String name, WorkerRunnable r) {
      super(name);
      runnable = r;
   }

   /* (non-Javadoc)
    * @see java.lang.Thread#run()
    */
   public final void run() {
      term = false;
      runnable.workerInitialize();
      while (!term) {
         try {
            if (!runnable.workerMainLoop()) {
               break;
            }
         } catch (Exception e) {
            e.printStackTrace();
            break;
         }
      }
      runnable.workerFinalize();
      synchronized(this) {
         notify();
      }
   }
   
   /**
    * Called by another thread (e.g. main()) to raise a termination signal for this thread.
    */
   public final void terminate() {
      synchronized(this) {
         System.out.println("About to terminate " + getName());
         term = true;
         if (isAlive()) {
            interrupt();
            try {
               wait();
            } catch (InterruptedException e) {
               // ignore
            }
            System.out.println(getName() + " terminated.");
         } else {
            System.out.println(getName() + " had already terminated.");
         }
      }
   }
   
   /**
    * Returns whether this thread has been requested to terminate.
    * @return true if thread has been requested to terminate
    */
   protected final boolean mustTerminate() {
      return term;
   }

   /**
    * The initialization method of the thread, called once before the main loop 
    * is entered. The default implementation does nothing.
    */
   public void workerInitialize() {
      // nothing to do
   }

   /**
    * The main loop function of the thread. The default implementation just 
    * issues a warning message and returns <code>false</code> so the worker will 
    * terminate immediatly.
    * 
    * @returns <code>false</code> if the thread should terminate, otherwise <code>true</code> 
    */
   public boolean workerMainLoop() {
      System.err.println("Worker " + getName() + ": nothing to do, will terminate immediateley");
      return false;
   }

   /**
    * The finalization method of the thread, called once after the main loop has 
    * been left. The default implementation does nothing.
    */
   public void workerFinalize() {
      // nothing to do
   }
}
